#ifndef _Persistent_cache_h
#define _Persistent_cache_h 1

#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/fields.h"

#include "runtime/or_info.h"

#include "storage_arena.h"
#include "frame_info.h"

#include "common/xref.h"
#include "common/page.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/CHECK_UNSWIZ.h"

#include "utils/map.h"
#include "utils/valuekey.h"
#include "utils/Timer.h"

class OR_obj;
class OR_recv_data_msg;
class Compacted_map;
class Scanned_set;
class Compacted_page;
class ostream;
class Page_map;
class Trace_replacer;
class Bool_vector;
       
struct Last_compacted_entry {
  Page* p;
  int usage_hist[Max_usage+1];
};

typedef Map<UIntKey, Last_compacted_entry*> Last_compacted; 

class Persistent_cache {
  //
  // Overview: Implements the persistent object cache (PC) that holds
  //   the Fields of persistent objects and caches copies of OR pages.
  //   It is organized as a set of pages (i.e. objects of type Page). It
  //   fetches pages from the OR, discards pages to find room for new
  //   ones, and allocates storage for the Fields of newly persistent objects.
  //
public:
  //
  // Constructors:
  //
  Persistent_cache(Storage_arena *sa);
  // Effects: Creates a cache using arena sa.
  
  ~Persistent_cache();
  // Effects: Destroys the PC and all the storage associated with it.
  

  //
  //  Methods for fetching and looking up objects and pages in the PC:
  //
  OR_obj *fetch(OR_num orx, Oref oref);
  // Effects: Looks for a cached copy of the OR page that contains a
  //   valid version of object <"or","oref">. If not found fetches the
  //   page and possibly some other pages of its segment into the cache,
  //   and discards/compacts pages in anticipation of future fetches. In
  //   either case returns a pointer to the Fields portion of object
  //   <"or","oref"> in the page.

  Fields lookup(OR_num orx, Oref oref) const;
  // Effects: Looks for a valid cached copy of the OR page that
  //   contains a valid version of object <"or","oref">. If not found
  //   returns null else returns the Fields of the object.

  Page *lookup_page(OR_num orx, Uint page_id) const;
  // Effects: Returns a pointer to the page <"or","page_id"> if it is
  //   cached in PC. If the page is not cached returns NULL.

  bool valid_page(OR_num orx, Uint page_id, Page* p);
  // effects: Returns TRUE iff p refers to the intact page with page id
  //          "page_id" belonging to OR or

  // 
  // Methods to index the pages in the PC:
  //
  Page *page(unsigned int index) const;
  // Effects: Returns a page given its index.

  unsigned int index(Page *p) const;
  // Effects: Returns the index of a page.

  // 
  // Methods for allocating storage and orefs for newly persistent
  // objects:
  //
  Fields allocate(OR_num orx, Oref oref);
  // Requires: Space for the object was allocated in the page from or 
  //   containing oref.
  // Effects: Returns the fields previously allocated for the object.

  Oref get_new_oref(OR_num orx, int num_slots);
  // Effects: Returns an unused oref at or such that an object of size
  //          num_slots (excludes header slots) can be allocated there.
  //          If no oref/space is available, sends alloc to OR to obtain
  //	      allocation rights to more pages.  If none obatained,
  //	      returns NULL_OREF
  //          In order for this oref/space not to be used in the next
  //          transaction, the caller must call commit_orefs

  void commit_orefs();
  // Effects: All the orefs/space allotted (since the last call to
  //          commit_orefs) are now permanently allocated

  void abort_orefs();
  // Effects: Frees the orefs/space that were allocated since the last call to
  //          commit_orefs. The caller must call commit_orefs after this. ???

 

  //
  // Methods that allow other modules to insert pages into the PC
  //
  Page *alloc_and_lock_page(OR_num orx, Uint page_id);
  // checks:   page <or,page_id> is not cached.
  // effects:  Allocates a page and enters the page as the cached page 
  //   for page_id on OR or. The page is marked as unevictable effectively
  //   locking the page in the cache. Eviction can be allowed by calling 
  //   the next method.

  void lock_page(Page *p);
  // Requires: p is a valid PC page.
  // Effects:  locks p in the PC marking it as unevictable. It is idempotent.

  void unlock_page(OR_num orx, Uint page_id);
  // checks: The page <or, page_id> is cached.
  // effects: Marks the page as evictable. It is idempotent.

 
  //
  // Methods for finding persistent info from OR
  //
  Oref get_root(OR_num ornum);
  // Requires: FE->comm has net connection to "ornum"
  // Effects: Returns the oref of the root in or "ornum".


  // 
  //  Methods used by other modules (e.g. the VH or the code manager) 
  //  to borrow pages from the PC.
  //
  Page *borrow_page(void);
  // Effects: Removes a page from the PC and returns a pointer to it.

  void return_page(Page *p);
  // Effects: Returns a borrowed page back to the PC.

  void invalidate(Core c);
  // Requires: c is not discarded.
  // Effects: Invalidates the fields for this core and removes it from
  //   the compacted page map if it is cached in a compacted page.


  // 
  // Methods used for debugging.
  //
  bool in_pc(Fields f) const;
  // Effects: Returns true iff f is within the bounds of the pc store.

  //
  // Methods used for manipulating statistics:
  //
  void print_stats(ostream &o);
  // Effects: Prints rot statistics in o.

  void reset_statistics();
  // Effects: Reset statistics.

#if CHECK_UNSWIZ
    void check_unswizzling();
#endif



  //-----------------------------------------------------------
  // 
  //    Private section:
  //
  //-----------------------------------------------------------
private:
  friend class OR_recv_data_msg; 
  friend class OR_recv_alloc_reply_msg; 
  friend class Communication;	       	// For testing purposes XXX.

  Compacted_map* cpmap; // Map from page ids to compacted page information.
  Page_map* pmap;  // Map from page ids to non-compacted pages cached in the PC. 

  //
  // Instance variables and methods used to implement page allocation and 
  // eviction/compaction.
  //
  Storage_arena *store; // Storage arena containing pages in PC.

  // Array with one entry per page frame in the store. It contains the
  // page id for the page in the corresponding page frame. and information on
  // whether it was borrowed or not.
  Frame_info *page_frame_info;

  // The next variables are used to ensure that the usage of the objects in 
  // the cache is decayed uniformly, i.e., the usage of all objects should be 
  // decayed once for each full scan of the primary pointer.
  char *page_decay_status;
  char cache_decay_status;

  int scan_index;                 // Index of the last page scanned.
  Scanned_set *last_scanned;      // Set with last pages scanned with low usage 

  // Map with last page that was compacted for each or.   
  Last_compacted *last_compacted; 

  Bool_vector *stack_refs; // Keeps track of pages that may be referenced
                               // by stack or registers.
                                  
  void free_pages(int num_pages);
  // Effects: Discards objects from the PC until there are  "num_pages +
  //   Max_pages_in_seg" free pages.


  Uint discard_multiplier; // Variable used to avoid integer division.

  unsigned collect_usage_and_decay(Uint page_index, bool decay);
  // Effects: Returns the page usage and if decay is true decays object
  // usage (before computing page usage). If decay is false object usage
  // is not decayed.

  void survival_fraction(Uint &usage, Uint count, Uint num_survived);
  // Effects: Computes survival fraction field in page usage and places it 
  //   in usage. num_survived is the number of objects in the page that will 
  //   survive compaction and count is the total number of objects in the page.
  
  void usage_value(Uint &usage, int *usage_hist, Uint count);
  // Effects: Compute the usage value for a page and place it in usage.
  //   usage_hist is an histogram with the usage values of all the objects
  //   in the page and count is the total number of objects in the page.

  void trace_replacement(int pages_to_discard);
  // Effects: Evicts the optimal or LRU page (see optimal_replacer.{h,cc}
  //   and lru_replacer.cc). It is only used if OPTIMAL_PAGE_REPLACEMENT 
  //   or LRU_PAGE_REPLACEMENT are 1.

  unsigned compact(unsigned pages_to_discard, bool evict);
  // Effects: Tries to compact pages until "pages_to_discard" pages are free
  //   and returns the number of pages actually evicted. If evict is true only
  //   retains objects that were modified during the current transaction. 

  int compact_page(Last_compacted_entry *dest, Page *source, unsigned threshold);
  // Requires: dest->p != source, and dest->p is a compacted page or dest == 0.
  // Effects: If dest != 0, moves the objects with usage higher than
  //   threshold from source to dest->p. Otherwise or if source does not
  //   become empty, compacts the remaining objects with usage higher
  //   than threshold within source. If source becomes empty returns -1;
  //   otherwise return the number of free bytes in source.

  bool move_to_page(OR_num orx, Uint page_id, Core c);
  // Requires: c is an object from <or,page_id> that was in a compacted page.
  //
  // Effects: If <or,page_id> is cached intact, moves the fields of c from 
  //   its current location into p, and marks c as not cached in 
  //   <or, page_id>'s entry in the compacted page map. Otherwise, does
  //   nothing. Return true iff object is moved.

  void mark_evicted(Frame_info *pinfo, OR_num orx);
  // Effects: Updates pmap and or_map to reflect the fact that the page
  //  in the frame identified by pinfo was evicted.

  void compact_inside_source(Page *p, Uint index, Frame_info *sinfo, int *hist,
                             bool intact);
  // Requires: hist has at least Max_usage entries.
  // Effects: If intact is false sorts the portion of the offsets array of p
  //   after index "index" by decreasing order of offsets. Always sets p to be an 
  //   empty page (i.e. with no objects) and initializes the hist array to zero.  
  void merge_pages(OR_num orx, Uint pageid, Page* new_page);
  // new version of merge_intact_pages to handle invalidations
  // correctly

  void merge_intact_pages(OR_num orx, Uint pageid, Page* newpage);
  // Requires: page refer to the new version of the page pageid
  //           from OR or
  //           An intact copy of this page is present in the cache
  // Effects: Merges the modifications/installations of page with
  //          the existing intact copy of this page by copying the
  //          uninstalled and the new objects from newpage to the existing page

  void stack_scan();
  // Effects: Scans stack and registers to conservatively determine which 
  //   pages in the PC and cores in the ROT may be pointed to by the stack or
  //   registers.

  Page *alloc_page(void);
  // Effects: Returns a pointer to an empty initialized page and
  //   marks it as evictable.

  void free_page(Page *p);
  // Requires: Page has no fields pointed to by the ROT in it.
  // Effects: Frees the page and marks it ans unevictable.

  unsigned int add(unsigned int a, unsigned int b) const;
  // Requires: a+b < store->size()*2
  // Effects: Adds a to b mod the size of the store.

  unsigned int sub(unsigned int a, unsigned int b) const;
  // Requires: a-b > -store->size()
  // Effects: Subtracts b from a mod the size of the store.

  //
  // Instance variables and methods used to implement the adaptive 
  // prefetching mechanism.
  //
  int num_pages_fetched;   // Estimate of number of pages fetched.
  int num_pages_installed; // Estimate of number of pages installed.
  int pref_group_sz;       // Current prefetch group size.
  
  Seg_bmap get_prefetch_group(OR_num orx, Uint page_id, Uint &num_req);
  // Effects: Returns a bitmap with a bit set for each page in the
  //   segment of page_id that should be prefetched when page_id is
  //   fetched.  Sets num_req to the number of pages requested.

  void update_prefetch_estimates(int num_fetched);
  // Effects: Updates the estimates for the adaptive prefetching mechanism.
  //   Should be called when the fetch reply is received, and num_fetched 
  //   should be the number of pages in the fetch reply.


  // 
  // Method called by OR_recv_data_msg when a fetch reply is received.
  //
  void register_pages(OR_num orx, Uint segid, Seg_bmap sent, 
	      Seg_bmap alloc_rights, Seg_bmap empty, Page **fetched_pages);
  // Requires: Fetch reply was just received for pages in segment
  //   <or,segid>, containing the pages in sent, and allocation rights for
  //   pages in alloc_rights. The fetched pages must have been placed in
  //   fetched_pages. empty indicates which pages are empty at the OR
  // Effects: Incorporates the information in the OR data structures and
  //   handles merges with already cached information.


  //
  // Instance varibales used to implement optimal page replacement:
  //
  Trace_replacer *trace_rep;

  //
  // Statistics:
  //

  // Time spent between sending a fetch message and receiving its reply.
  // Excluding the time spent registering pages.
  Timer net_fetch_time;

  // Time spent creating the fetch message and registering the pages
  // received in the reply.
  Timer fetch_time;
                          
  Timer stack_scan_time;       // Time spent scanning the stack.
  Timer page_scan_time;        // Time spent scanning pages.    
  Timer page_compaction_time;  // Time spent compacting pages.     

  unsigned number_of_scans;       // Number of times pages were scanned.
  unsigned num_pages_scanned;     // Number of pages scanned.
  unsigned tot_pages_compacted;   // Number of pages compacted.
  unsigned tot_pages_dis_wo_comp; // Number of pages discarded w/o comp.
  unsigned tot_objects_compacted; // Number of objects copied when compacting.
  unsigned tot_objects_merged;    // Number of objects merged.
  unsigned tot_objects_evicted;   // Number of objects evicted.
  
  unsigned num_fetches;                // Number of fetches.
  Uint num_delayed_fetches;            // Number of messages received in the
    // eager prefetching case in the "background" (hopefully background)
  Uint tot_fore_pages;             // Number of pages received
    // while the FE was waiting for them
  Uint tot_back_pages;             // Number of pages received
    // in the background
  unsigned tot_num_pages_fetched;      // Number of pages fetched.
  unsigned tot_num_pages_installed;    // Number of pages installed.
  unsigned fetches_for_compacted;      // Number of pages fetched for which there
                                       // was a compacted version of the page cached.
  unsigned fetch_group_buckets[Max_pages_in_seg+1]; // Histogram of prefetch group sizes.
  unsigned usage_buckets[Max_usage+1];   // Number of scanned objects with given usage.
  unsigned num_not_installed; // Number of scanned objects that were not installed.

  unsigned num_borrowed_pages;
  unsigned num_intact_pages;
 

};


inline Page *Persistent_cache::page(unsigned int index) const {
  th_assert(index < store->size(), "Invalid index\n"); 
  return store->start() + index; 
}
 
inline unsigned int Persistent_cache::index(Page *p) const { 
  th_assert(p - store->start() < (int) store->size(), "Invalid page\n");
  return p - store->start(); 
}

inline  unsigned int Persistent_cache::add(unsigned int a, unsigned int b) const {
  unsigned int res = a+b;
  th_assert(res < store->size()*2, "Invalid arguments");
  if (res >= store->size()) res = res - store->size();
  return res;
}

inline unsigned int Persistent_cache::sub(unsigned int a, unsigned int b) const {
  int res = a-b;
  th_assert(res > -((int)store->size()), "Invalid arguments");
  if (res < 0) res = store->size() + res;
  return res;
}

inline bool Persistent_cache::in_pc(Fields f) const {
  if (f > (Fields) store->start() && f < (Fields) (store->start()+store->size()))
    return true;
  return false;
}

inline void Persistent_cache::commit_orefs() {
    FE->or_map->commit_orefs();
}

inline void Persistent_cache::abort_orefs() {
    FE->or_map->abort_orefs();
}

inline bool Persistent_cache::valid_page(OR_num orx, Uint page_id, Page* p) {
    Uint actual_pid = page_frame_info[index(p)].id();
    return actual_pid == page_id && p->get_or() == orx;
}

#endif // _Persistent_cache_h




