#include <stream.h>

#include "compiler/C++wrappers/core.h"

#include "runtime/stats.h"
#include "runtime/trans_log.h"
#include "runtime/tm.h"
   
#include "resident_object_table.h"
#include "persistent_cache.h"
#include "surrogate_table.h"
#include "volatile_heap.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/CHECK_CACHE_INVARIANTS.h"

#include "common/objrefs.h"

static inline void mark_free_volatile(Core c) {
  // Frees a volatile object.
  Shortp m = c->methods_shortp();
  c->mark_free();

  // Save m in object's oref. Need to save m in order to retrieve it later
  // when entry is reused to decrement counts for objects pointed to 
  // by the target's fields.
  c->oref() = (Oref) m;
}


void  Resident_object_table::dec_rc(Core c) {
  th_assert(!c->is_free(), "Adjusting ref count on free entry");

  // Decrement the reference count of target
  int ref_cnt = c->ref_count() - 1;
  th_assert(ref_cnt >= 0, "Negative reference count");
  c->set_ref_count(ref_cnt);

  if (ref_cnt == 0) {
    if (!c->is_persistent() && !handles_freed) {
      // We can imediately free entries for volatile objects as long
      // as no handle for that object was reclaimed during this
      // transaction after it was used to store a pointer to the
      // object in some modified object (handles_freed is a
      // conservative approximation of this).
      mark_free_volatile(c);
      free_list.insert(c);
    } else {
      zero_rc_log2.append(index(c));
    }
  }
}


inline void Resident_object_table::adj_rc_prim(Obj_bitfield bf, 
					       unsigned num_slots,
                                               Obj_x *slots,
                                               int delta) {
  th_assert(delta == -1 || delta == 1, "Invalid argument");
  // Iterate over the slots in slots using bf to identify the 
  // references and adjust counts of cores reference by swizzled 
  // references by delta.
  Object_references iter(bf, num_slots);
  int index;
  while (iter.get(index)) {
    if (slots[index].is_pointer()) {
      // If the reference is swizzled get the core of the object it points to.
      Core target = slots[index].pointer()->to_core();      
      if (delta == 1) inc_rc(target); else dec_rc(target); 
    }
  }
}

void Resident_object_table::adj_rc_obj(Core c, int delta) {
  th_assert(!c->is_free() && !c->is_discarded(), "Invalid argument");
  
  // Iterate over the references of c.
  adj_rc_prim(c->bitfields(), c->num_slots(),
		     (Obj_x *) c->get_fields()->slots(), delta);
}

void Resident_object_table::adj_rc_copy(Fields copy, int delta) {
  Core obj = copy->to_core();
  int bf_off = obj->has_var_bitfields();
  Obj_bitfield bf = (bf_off) ? *((Obj_bitfield *)((char *) copy->slots() + bf_off)) 
                             : obj->bitfields();
  adj_rc_prim(bf, obj->num_slots(), (Obj_x*)copy->slots(), delta);
}


bool Resident_object_table::need_adj_rc_mod(Core c, Fields copy, bool persistent) { 
  Old_ref_log *olog = (persistent) ? per_log : vol_log;
  if (olog == 0) {
    // If needed create a log to record any references in a copy that require
    // decrements.
    olog = new Old_ref_log(FE->tm->transaction_log(), persistent);
    if (persistent) per_log = olog; else vol_log = olog; 
  }

  Fields f = c->get_fields();

  if (f->use_eager_ref_counting() || !f->has_pointer_updates()) return false;
  th_assert(copy != 0, "Object must have copy");
  
  // Check (rare) case where bitfields in object and copy are different.
  int bf_off = c->has_var_bitfields();
  if (bf_off) {
    olog->grow();
    Obj_bitfield obj_bf = *((Obj_bitfield *)((char *) f->slots() + bf_off));
    Obj_bitfield copy_bf = *((Obj_bitfield *)((char *) copy->slots() + bf_off));
    if (obj_bf != copy_bf) {
      adj_rc_obj(c, 1);
      
      Object_references iter(copy_bf, c->num_slots());
      int index;
      Obj_x *copy_slots = (Obj_x*) copy->slots();
      while (iter.get(index)) {
	if (copy_slots[index].is_pointer()) {
	  // If the reference is swizzled get the core of the object it points to.
	  // log that it should be decremented.
	  olog->log_reference(copy_slots+index);
	}
      }
      return false;
    }
  }
  return true;
}

void Resident_object_table::free_entries() {
  FE_STATS(num_frees++)

  for (int i=0; i < 2; i++) {
    int lsize = (i) ? zero_rc_log1.size() : zero_rc_log2.size();
    int *log = (i) ? zero_rc_log1.as_pointer() : zero_rc_log2.as_pointer();
    
    for (int j=0; j < lsize; j++) {
      FE_STATS(num_dropped_to_zero1++)
      int cindex = log[j];
      Core c = fetch(cindex);

      // If entry no longer has a null reference count or has already
      // been freed ignore it.
      if (c->ref_count() != 0 || c->is_free()) continue;
   
      if (c->is_persistent()) {
	if (c->is_discarded()) {
	  // Remove mapping from this object to eventual OR surrogates from
	  // the surrogate table. Do this by managing surrogate table using 
	  // FIFO replacement.
	  // FE->surr_table->remove(c->or_discarded(), c->oref());
	  
	  // Entry corresponds to discarded persistent object; mark
	  // it as free but leave it in its place in the swizzle table.
	  // It will be reused during swizzling.
	  c->mark_free();
	  FE_STATS(num_marked_free++)
	}
      } else {
	mark_free_volatile(c);
	free_list.insert(c);
        FE_STATS(num_marked_free++)
      }
    }
  }
  zero_rc_log1.clear();
  zero_rc_log2.clear();
}

void Resident_object_table::adj_rc_commit_request(void) {
  FE_STATS(rc_commit_adjustment.start())
  FE_STATS(num_commits++)
  
  evicted_trans = 0;
  free_during_trans = false;

  Core obj;
  Fields copy;
  Fields f;
  Trans_log* tlog = FE->tm->transaction_log();

  // Objects that were created in the current transaction have no
  // copies and they can not be on the MOS.
  Trans_log::Volatile_iter iter(tlog, Iter_new_volatile);
  while (iter.get(obj)) {
    th_assert(!obj->is_discarded(), "Modified object was discarded");
    f = obj->get_fields();

    if (!f->use_eager_ref_counting()) {
      adj_rc_obj(obj, 1);
      FE_STATS(num_new_scanned++)
    }
    // Clear bits while iterating over the objects.
    f->unstamp_all();
  }


  // Iterate over volatile objects in MOS adjusting references to reflect
  // their current state. 
  Trans_log::Mod_iter miter(tlog);
  while (miter.get(obj, copy)) {
    th_assert(!obj->is_discarded(), "Modified object was discarded");
    f = obj->get_fields();

    if (need_adj_rc_mod(obj, copy, false)) {
      Obj_x *slots = (Obj_x *) f->slots();
      Obj_x *copy_slots = (Obj_x *) copy->slots();

      int slotno;
      Object_references gen(obj->bitfields(), obj->num_slots());
      while (gen.get(slotno)) {
	if (!slots[slotno].is_data()) 
	  adj_rc_mod(slots[slotno].pointer()->to_core(), copy_slots+slotno, false);
      }
    }
    
    // Clear bits while iterating over the objects.
    f->unstamp_all();
  }

  // Decrement counts for stale references in copies and free entries whose
  // reference counts drop to zero. Needs to be performed after all reference
  adj_rc_copies();
 
  FE_STATS(rc_commit_adjustment.stop())

#if CHECK_CACHE_INVARIANTS
  th_assert(check_rc_invariant(), "Reference count invariant does not hold");
#endif

  // Free unreferenced rot entries.
  FE_STATS(rc_free.start())
  free_entries();
  FE_STATS(rc_free.stop())

#if CHECK_CACHE_INVARIANTS
  th_assert(check_free_invariant(), "Free invariant does not hold");
#endif

  commit_request = true;
}



void Resident_object_table::adj_rc_commit_reply(bool commited) {
  if (!commited) {
    // The transaction aborted; 1. Undo changes made when commit request
    // was sent.
    if (commit_request) {
      log.undo();
    }
    
    // 2. Revert the modified objects to their earlier state (if possible)
    // and adjust reference counts
    Trans_log *tlog = FE->tm->transaction_log();
    
    // Iterate over the log of accesses to persistent objects
    Trans_log::Persistent_iter piter(tlog);
    OR_num orx = Null_or;
    bool has_mods = true;
    Ros *ros;
    Core c;
    
    do {
      // First iteration handles volatile objects.
      if (has_mods) {
	Trans_log::Mod_iter miter(tlog, orx);
	Fields copy;
	
	while (miter.get(c, copy)) {
	  if (copy) {
	    if (!c->is_discarded()) {
	      // The copy and the object exist.
	      Fields f = c->get_fields();
	      
	      // If adj_rc_commit_request has already been called or 
	      // reference counts were being updated eagerly on modifications,
	      // decrement counts for all swizzled references in the object and 
	      // increment them for the copy. Otherwise, do no ref count adjustements.
	      if (commit_request || f->use_eager_ref_counting()) {
		adj_rc_obj(c, -1);
		adj_rc_copy(copy, 1);
	      }
	      
	      // Since  the object was not invalidated, we can
	      // revert the object to the state in the copy.
	      memcpy(f, copy, (c->num_slots() + Fields_c::header_slots())* Slot_size);
	      
	      // Properly initialize fields.
	      f->unstamp_all();
	    }
	  } else {
	    // No copy for the object has been maintained.
	    // Invalidate the object if it has not been discarded yet
	    // and decrement the reference counts for all swizzled references 
	    // in it if adj_rc_commit_request has already been called or 
	    // reference counts were being updated eagerly on modifications.
	    if (!c->is_discarded()) {
	      Fields f = c->get_fields();
	      if (commit_request || f->use_eager_ref_counting()) {
		adj_rc_obj(c, -1);
	      }
	      OR_num orx = f->orx();      
	      f->mark_invalid();
	      c->mark_discarded(orx);
	    }
	  }
	}
      }
    } while (piter.get(orx, ros, has_mods));
	      
    // "Evict" the newly created objects
    Trans_log::Volatile_iter new_iter(tlog, Iter_new_volatile);
    while (new_iter.get(c)) {
      th_assert(!c->is_discarded(), "Modified object was discarded\n");
      Fields f = c->get_fields();
      if (commit_request || f->use_eager_ref_counting()) {
	adj_rc_obj(c, -1);
      }
      FE->vh->free_object(c);
      c->mark_discarded(0);
    }
  
    if (!commit_request) {
      // On an abort during a transaction (before the commit request was sent)
      // may need to decrement reference counts for references in copies to undo
      // the adjustments in adj_rc_during_trans.
      if (per_log != 0 || vol_log != 0) adj_rc_copies();
    }

#if CHECK_CACHE_INVARIANTS
    th_assert(check_rc_invariant(), "Reference count invariant does not hold");
#endif

    // Free unreferenced rot entries.
    FE_STATS(rc_free.start())
    free_entries();
    FE_STATS(rc_free.stop())
    
#if CHECK_CACHE_INVARIANTS
    th_assert(check_free_invariant(), "Free invariant does not hold");
#endif
  }
  
  // Clear the log of changes made while adjusting reference counts.
  log.clear();
 
  // New transaction starting indicate that no handles were freed 
  // during the current transaction.
  handles_freed = false;

  // Note that that the reply to the outstanding commit request was handled.
  commit_request = false;
}


bool Resident_object_table::invalidate_object(OR_num orx, Oref oref) {
  Core c = lookup(orx, oref);
  
  if (!c || c->is_discarded()) {
    // Object is not in swizzle table or was already discarded. Need
    // to invalidate any uninstalled version of the object
    Fields f = FE->pc->lookup(orx, oref);
    if (f) f->mark_invalid();
    return false;
  }

  // Object is in swizzle table and is not discarded (since we never
  // evict rot entries of objects that were used during the current 
  // transaction those are guaranteed to reach this point).
  Fields f = c->get_fields();
  bool used = f->is_read() || f->is_written();
  bool written = f->is_written();

  if (!written || f->use_eager_ref_counting()) {
    // If object was NOT modified or was stamped to use eager reference counting,
    // decrement counts for objects it pointed to.
    adj_rc_obj(c, -1);
  }

  // Discard object.
  FE->pc->invalidate(c);
  c->mark_discarded(orx);

  return used;
}
  


//
// Undo log methods. 
//

inline void Resident_object_table::RC_log::mark_free(Core c) {
  int i = FE->rot->index(c);
  th_assert(i < (1 << index_bits), "Invalid argument");
  log.append((i << operation_bits) | Free);
  log.append(c->methods_shortp());
  log.append(c->oref());
}

inline void Resident_object_table::RC_log::mark_free_and_enqueue(Core c) {
  th_assert(c->oref()==0, "Invalid argument");
  int i = FE->rot->index(c);
  th_assert(i < (1 << index_bits), "Invalid argument");
  log.append((i << operation_bits) | Free_and_enqueue);
  log.append(c->hdr_size());
}

void Resident_object_table::RC_log::clear() { 
  log.clear(); 
  log.reclaim();
}

void Resident_object_table::RC_log::predict_size(int size) {
  log.predict(size); 
}

bool Resident_object_table::RC_log::empty() { return log.size() == 0; }

void Resident_object_table::RC_log::undo() {
  Resident_object_table *rot = FE->rot;
  
  // Process log.
  while (log.size()) {
    Slot header = log.remove();
    int index = header >> operation_bits;
    Core c = rot->fetch(index);
    int op = header & operation_mask;
    
    rot->zero_rc_log2.append(index);

    if (op == Free) {
      c->methods_shortp() = log.remove();
      c->oref() = log.remove();
    } else {
      th_assert(op == Free_and_enqueue, "Invalid operation");
      rot->free_list.remove(c, log.remove());
      c->methods_shortp() = c->oref();
      c->oref() = 0;
    }
  }
  // We currently do not revert the states of dropped_to_zero and surr_table.
  // This does not compromise correctness and should not affect abort 
  // performance significantly.
}


void Resident_object_table::adj_rc_copies() {
  Old_ref_log::Old_ref_iter *iter;

  for (int i=0; i < 2; i++) {
    if (i == 0) {
      if (per_log == 0) continue;
      iter = new Old_ref_log::Old_ref_iter(per_log);
    } else {
      if (vol_log == 0) continue;
      iter = new Old_ref_log::Old_ref_iter(vol_log);
    }

    Core target;
    while (iter->get(target)) {
      int ref_cnt = target->ref_count() - 1;
      target->set_ref_count(ref_cnt);

      // Can free entries immediately if their count drops to zero
      // because all increments have already been performed.
      if (ref_cnt == 0) {
	if (target->is_persistent()) {
	  if (target->is_discarded()) {
	    // We currently only discard entries for persistent objects
            // whose fields were discarded.
	    log.mark_free(target);
	    target->mark_free();
	  }
	} else {
	  // Volatile objects. 
	  log.mark_free_and_enqueue(target);
	  mark_free_volatile(target);
	  free_list.insert(target);
	}
      }
    }    
    delete iter;
  }
  if (per_log) delete per_log;
  if (vol_log) delete vol_log;
  per_log = vol_log = 0;
}


void  Resident_object_table::adj_rc_during_trans(int cache_size) {
  Trans_log *tlog = FE->tm->transaction_log();

  // We delay processing reference count updates for modifications
  // eagerly until percent_useless is 1%. percent_useless is
  // (approximately) the amount of space wasted in unreferenced ROT
  // entries as a percentage of the total PC size. 
  if (zero_rc_log1.size()*sizeof(Core_c)*100 > FE->max_rot_waste*cache_size
      && !commit_request) {
    free_during_trans = true;
  }

  if (!free_during_trans) return;

  // Iterate over the log of modified objects.
  Trans_log::Persistent_iter piter(tlog);
  OR_num orx = Null_or;
  bool has_mods = true;
  Ros *ros;
  Core c;
  int slotno;
  
  // Process modified objects adjusting reference counts such that
  // they accurately reflect their current state; also increment
  // reference counts for objects pointed to by swizzled references in
  // copies of modified objects (this last step ensures that there are
  // no dangling references if the transaction aborts).
  do {
    // First iteration handles volatile objects.
    if (!has_mods) continue;

    Trans_log::Mod_iter miter(tlog, orx, true);
    Fields copy;
      
    while (miter.get(c, copy)) {
      Fields f = c->get_fields();

      // Don't need to process this object because it is already using
      // eager reference counting. 
      if (f->use_eager_ref_counting()) continue;

      bool persistent = (orx != Null_or);
      need_adj_rc_mod(c, copy, persistent);
      Old_ref_log *olog = (persistent) ? per_log : vol_log;
      th_assert(olog, "olog is null");
      olog->grow();

      Obj_x *slots = (Obj_x*) c->slots();
      Obj_x *copy_slots = (Obj_x*) copy->slots();

      Obj_bitfield bf = c->bitfields();
      Object_references iter(bf, c->num_slots());
      while (iter.get(slotno)) {
	if (!slots[slotno].is_data()) {
	  Core target = slots[slotno].pointer()->to_core();
	  inc_rc(target);
	  if (!copy_slots[slotno].is_data())
            olog->log_reference(copy_slots+slotno);
	}
      }
      
      // From this point till the end of the transaction reference counts
      // in this object will be updated eagerly every time a pointer is
      // modified. But cannot mark arrays of data this way because 
      // PUTPV will try to adjust reference counts assuming the array
      // elements are pointers (XXXwhen PUTPV goes away this becomes
      // only a performance optimization).
      if (bf != OBJ_BF_ALLDATA)
	f->stamp_eager_ref_counting();
    }
  } while (piter.get(orx, ros, has_mods));

  // Adjust counts to reflect new objects 
  Trans_log::Volatile_iter iter(tlog, Iter_new_volatile);
  while (iter.get(c)) {
    Fields f = c->get_fields();

    if (!f->use_eager_ref_counting()) adj_rc_obj(c, 1);

    Obj_bitfield bf = c->bitfields();
    if (bf != OBJ_BF_ALLDATA)
      f->stamp_eager_ref_counting();
  }
}


inline void Resident_object_table::adj_rc_update(Shortp oldp, struct Core_s *newp) {
  int ref_cnt;
  if (SHORTP_IS_PTR(oldp)) {
    Core oldc = SHORTP_AS_FULL(Core_c,oldp);
    ref_cnt = oldc->ref_count() - 1;
    th_assert(ref_cnt >= 0, "Negative reference count");
    oldc->set_ref_count(ref_cnt);
    if (ref_cnt == 0) zero_rc_log1.append(index(oldc));
  }

  Core newc = (Core) newp;
  ref_cnt = newc->ref_count() + 1;
  th_assert(ref_cnt < 255, "Large reference counts are not implemented");
  newc->set_ref_count(ref_cnt);
}


void Eager_ref_count_update(Shortp oldp, struct Core_s *newp) {
  FE->rot->adj_rc_update(oldp, newp);
}
 

bool Resident_object_table::check_rc_invariant() const {
  int used_size = index(free_area);
  unsigned *tmp_rc = new unsigned[used_size];
  Core c;

  // Initialize temporary reference counts to 0.
  for (int i=0; i < used_size; i++) tmp_rc[i] = 0;
 
  // Increment temporary reference counts to account for all swizzled 
  // references in non-discarded objects.
  for (int i=0; i < used_size; i++) {
    c = fetch(i);
    // This test assumes that vh gets its storage from the pc but the
    // code manager does not allocate dispatch vectors in pages borrowed
    // from the PC.
    if (!c->is_discarded() && FE->pc->in_pc(c->get_fields())) {
      // Core c is non-discarded; iterate over the references of c.
      Core_c tmp;
      if (c->is_free()) {
         // If object is free and non-discarded it must be volatile.
         // So initialize tmp to look like c, but get the methods
         // from the oref field and re-install them in tmp.
         memcpy(&tmp, c, sizeof(Core_c));
         tmp.methods_shortp() = (Shortp) tmp.oref();
         c = &tmp;
      }
      int slot_index;
      Object_references iter(c->bitfields(), c->num_slots());
      Obj_x *slots = (Obj_x *) c->get_fields()->slots();
      
      while (iter.get(slot_index)) {
	if (slots[slot_index].is_pointer()) {
	  // If the reference is swizzled get the core of the object it points to
          // and increment its entry in the tmp_rc array.
	  Core target = slots[slot_index].pointer()->to_core();
          tmp_rc[index(target)] += 1;
	}
      }
    }
  }

  // Increment counts for handles.
  Handle_generator gen(this);
  while (gen.get(c)) {
    tmp_rc[index(c)] += 1;
  }
    
  // Check if temporary counts match those in the rot.
  bool correct = true;
  for (int i=0; i < used_size; i++) {
    c = fetch(i);
    if (c->is_discarded() || FE->pc->in_pc(c->get_fields())) {
      if (c->ref_count() != tmp_rc[index(c)]) {
	correct = false;
        cout << "Incorrect reference count " << c->ref_count() 
	     << " for core = " << (void*) c << ", should be " << tmp_rc[index(c)]
	     << " (class =  " << ((struct Class_s *) c->get_class())->name << ")\n"; 	  
      }
    }
  }

  delete [] tmp_rc;

  return correct;
}


bool Resident_object_table::check_free_invariant() const {
  int used_size = index(free_area);
  Core c;
  bool correct = true;
  // Skip first object which is a special object used to consume handle 0
  // such that 0 can be used as an invalid handle value.
  for (int i=1; i < used_size; i++) {
    c = fetch(i);
    if (c->is_discarded() && c->ref_count() == 0) {
      if (!c->is_free()) {
	correct = false;
	  cout << "Core was not freed " << (void*) c 
	       << " (class =  " << ((struct Class_s *) c->get_class())->name
	       << ")\n"; 
      }
    }
  }

  return correct;
}



void Resident_object_table::expensive_free(Core c) {
  int index = orefs_bucket(c->oref());
  Core tmp = SHORTP_AS_FULL(Core_c, swiz_tab[index]);
  Core tmp1 = 0;
  for (; tmp != 0; tmp = tmp->next()) {
    if (tmp == c) {
      if (!tmp1) 
	swiz_tab[index] = FULL_AS_SHORTP(tmp->next());
      else
	tmp1->set_next(tmp->next());
      c->mark_free();
      free_list.insert(c);
      return;
    }
    tmp1 = tmp;
  } 
}








