#include <stream.h>

#include "compiler/C/extern.h"

#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/core.h"
#include "compiler/C++wrappers/class.h"

#include "resident_object_table.h"
#include "volatile_heap.h"

#include "fe/main/fe_config.h"

#include "utils/mdebug.h"

implementArray(Slot_array, Slot)

//
// The resident object table.
//

static const unsigned avg_swiz_entries_per_bucket = 3;


//
// Pointer to rot table to allow low level direct access to the table
//
struct Core_s *ROT_table;


Resident_object_table::Resident_object_table(Uint per_sz, 
					     Uint obj_sz, Storage_arena *s) {
  // Estimate the number of buckets we need in the swizzle table 
  // to have an average of avg_swiz_entries_per_bucket entries per
  // bucket.
  swiz_buckets = per_sz/(obj_sz * avg_swiz_entries_per_bucket);

  // Round swiz_buckets to nearest power of 2
  unsigned int mask = 1;
  unsigned bit_count = 0;
  while (mask < swiz_buckets) {
    mask = mask << 1;
    bit_count++;
  }

  if (mask == 1 ||  mask - swiz_buckets <= swiz_buckets - (mask >> 1)) {
    swiz_buckets = mask;
  }  else {
    swiz_buckets = mask >> 1;
    bit_count--;
  }
  shift_bits = sizeof(unsigned)*8 - bit_count;

  // Create an empty swizzle table.
  swiz_tab = new Shortp[swiz_buckets];
  for (Uint i=0; i < swiz_buckets; i++) swiz_tab[i] = 0;

  sa = s;
  size = sa->size()*sizeof(Page)/sizeof(Core_c);
  table = (Core) sa->start();

  // Initialize low level pointer to ROT
  ROT_table = (struct Core_s *) table;
  
  // Consume entry 0 in the table by marking it as non-free. This is
  // to allow use of 0 has the value of an invalid handle in the
  // client/FE interface.
  table->init();
  table->zero_dv();

  // The rest of the ROT is free.
  free_area = table+1;

  // Allocate handle table to have one bit per ROT entry. The 
  // bits are initially false.
  client_handle_table = new Bool_vector(size, false); 

  // Allocate logs to record entries whose reference counts may 
  // have dropped to zero.
  zero_rc_log1.predict(size/1000);
  zero_rc_log2.predict(size/1000);
  
  free_during_trans = false;

  stack_refs = new Bool_vector(size, false);
  
  // No handles were freed
  handles_freed = false;

  vol_log = per_log = 0;
  evicted_trans = 0;

  // Not waiting for comit reply.
  commit_request = false;

  reset_statistics();
}


Resident_object_table::~Resident_object_table(void) {
  delete sa;
  delete client_handle_table;
  delete stack_refs;
  delete swiz_tab;
}



void Resident_object_table::print_stats(ostream &o) {  
  o << "\nTotal number of entries in ROT: " << (int)(free_area-table)
    << "\nAverage installed fields size w/o header (bytes): " 
    << avg_inst_object*sizeof(Slot)/(float)num_installed << "\n";

  o << "\nSwizzle table stats: \n" << " Number of swizzles: " 
    <<  num_swizzles << "\n Number of pc lookups: " << num_pc_lookups;
  print_swizzle_table_stats(o);

  o << "\nReference count stats: \n"
    << " Time spent adjusting ref. counts at commit: " 
    << rc_commit_adjustment.elapsed() 
    << "\n Number of modified objects with copies scanned: " 
    << num_mod_with_copies
    << "\n Number of modified objects without copies scanned: " 
    << num_mod_without_copies
    << "\n Number of new objects scanned: " << num_new_scanned
    << "\n Number of commit adjustments: " << num_commits
    << "\n Time spent freeing cores: " << rc_free.elapsed()
    << "\n Number of entries in dropped to zero before free_entries: "
    << num_dropped_to_zero1
    << "\n Number of entries in dropped to zero after free_entries: "
    << num_dropped_to_zero2
    << "\n Number of times free_entries is called: "
    << num_frees
    << "\n Number of entries marked as free in free_entries: "
    << num_marked_free << endl;  
}

void Resident_object_table::reset_statistics() {
  num_swizzles = 0;
  num_pc_lookups = 0;
  avg_inst_object = 0;
  num_installed = 1;

  rc_commit_adjustment.reset();
  num_mod_with_copies = 0;
  num_mod_without_copies = 0;
  num_new_scanned = 0;
  num_commits = 0;

  rc_free.reset();        
  num_dropped_to_zero1 = 0;
  num_dropped_to_zero2 = 0;
  num_frees = 0;      
  num_marked_free = 0;  
}


//
// Storage management:
//


// Computes number of extra core entries needed to hold a dispatch header 
// with size dispatch vectors.
static inline int Core_size_from_dh_size(int size) {
  // Number of dh slots per core (to get number of pointer to dvs we can store)
  const unsigned core_dhs = sizeof(Core_s)/sizeof(Slot);
  return (size + core_dhs - 2)/core_dhs;
}

Core Resident_object_table::allocate(unsigned int dh_size) {
  // First search for an appropriate entry in the free list.
  Core res = free_list.remove(dh_size);
  
  if (res) {
    if (!res->is_discarded()) {
      // If entry is dirty (i.e. is not marked as discarded) we need
      // to decrement the reference counts for the objects it refers
      // to.  Only volatile objects should be marked as non-discarded
      // at this point, persistent object cores are inserted in the
      // free list only after the object is discarded.  For free
      // objects the oref field is null for persistent objects, and
      // holds the core's dispatch vector pointer (Shortp) for
      // volatile objects.
      th_assert(res->oref() != 0, "Only free volatile cores can be dirty");

      // Re-install methods on core so that we can call the next method.
      res->methods_shortp() = (Shortp) res->oref();
      adj_rc_obj(res, -1);
       
      // Free space occupied by volatile object in the volatile heap
      FE->vh->free_object(res);
    }
    return res;
  }

  // If there was no entry with an appropriate size in the free list
  // get it from the free area. Note that these entries can not be
  // dirty.
  int entry_size = Core_size_from_dh_size(dh_size) + 1;
  res = free_area + entry_size - 1; // res points to the last core in entry

  // If there isn't enough space fail.
  if (res >= table+size) th_fail("ROT is too small");

  free_area = free_area + entry_size;  
 
  return res; 
}


//
// The ROT entries are chained in the free list for their 
// size using the next fields in the last core in the entry.

Resident_object_table::Free_list::Free_list(void) {
  int size = Core_size_from_dh_size(Max_dh_size) + 1;
  free_lists = new Core[size];
  for (int i = 0; i < size; i++)
    free_lists[i] = 0;
}


Resident_object_table::Free_list::~Free_list(void) {
  delete [] free_lists;
}


Core Resident_object_table::Free_list::remove(unsigned size) {
  th_assert(size < Max_dh_size, "Invalid argument");
  
  int index = Core_size_from_dh_size(size);
  if (!free_lists[index]) return 0;
  Core tmp = free_lists[index];  
  free_lists[index] = tmp->next(); 
  return tmp;
} 

     
void Resident_object_table::Free_list::insert(Core c) {
  th_assert(c->is_free(), "Invalid argument\n");

  // Get number of dispatch header words in the entry
  unsigned int size = c->entry_size();
  th_assert(size < Max_dh_size, "Invalid argument");

  int index =  Core_size_from_dh_size(size);

  c->set_next(free_lists[index]);
  free_lists[index] = c; 
}


void Resident_object_table::Free_list::remove(Core c, unsigned size) {
  th_assert(c->is_free(), "Invalid argument\n");
  th_assert(size < Max_dh_size, "Invalid argument");

  int index =  Core_size_from_dh_size(size);
  Core prev = 0;
  Core cur;
  for (cur = free_lists[index]; cur != 0; cur = cur->next()) {
    if (cur == c) break;
    prev = cur;
  }
  th_assert(cur != 0, "Invalid argument");
  
  if (prev != 0)
    prev->set_next(c->next());
  else
    free_lists[index] = c->next();
}


inline Core Resident_object_table::allocate_object_raw(int dhsize, Uint size,
						       bool variable_size) {  
  Core c = allocate(dhsize);
  
  // Allocate space in the VH for the object fields 
  int num_slots = (size + Slot_size - 1)/Slot_size;
  if (variable_size)
      num_slots++;

  Fields f = FE->vh->allocate(num_slots + Fields_c::header_slots());
 
  // Initialize core and fields
  c->oref() = 0;
  c->set_fields(f);
  c->set_ref_count(0);
  Uint cindex = index(c);
  f->init(cindex);

  if (variable_size) 
    // Store object size on the first slot.
    *(c->slots()) = (Slot) size;

  // c has null reference count put it in log of entries with null ref counts.
  zero_rc_log2.append(cindex);
  
  // Return pointer to class dispatch header
  return c;
}

inline Obj Resident_object_table::allocate_object(Class_c cl, Uint size) { 
  // Allocate ROT entry for an object of class cl

  // Calculate the size of the object in bytes
  int osize = size? size: cl.num_slots() * Slot_size;
  Core c = allocate_object_raw(cl.hdr_size(), osize, size != 0);

  // Initialize dispatch header
  cl.init_disp_hdr(c);
  return c->to_obj();
}

struct Obj_s *Allocate_object(struct Class_s *classobj, int size) {
  Class_c cl(classobj);
  return (struct Obj_s *)(FE->rot->allocate_object(cl, (unsigned) size));
}

struct Obj_s *Allocate_object_raw(int dhsize, int size, bool variable_size) {
  Core c = FE->rot->allocate_object_raw(dhsize, (Uint) size, variable_size);
  c->set_dv(NULL); // Zero out the methods
  return (struct Obj_s *) c;
}



void Pin_object(Obj_p x) { // from runtime/C/extern.h
  (void)FE->rot->obj_to_handle((Obj)x);
}


void Resident_object_table::process_zero_rc_log() {
  if (!free_during_trans) return;

  int lsize = zero_rc_log1.size();
  for (int i=0; i < lsize; i++) {
    int index = zero_rc_log1[i];
    Core c = fetch(index);
    if (c->is_free()) continue;
    if (c->ref_count() == 0) {
      if (c->is_discarded()) {
	if (!is_stack_referenced(c)) {
	  c->mark_free();
	} else { 
	  zero_rc_log2.append(index);
	}
      }
    }
  }
  zero_rc_log1.clear();
}










