#ifndef _Resident_object_table_h
#define _Resident_object_table_h 1

#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/core.h"
#include "compiler/C++wrappers/class.h"

#include "runtime/handle.h"

#include "storage_arena.h"
#include "old_ref_log.h"

#include "config/vdefs/USE_REF_COUNTS.h"

#include "common/xref.h"
#include "common/obj_bitfield.h"

#include "utils/array.h"
#include "utils/Timer.h"
#include "utils/bits.h"

declareArray(Slot_array, Slot)

class Bool_vector;
class OR_obj;
class ostream;

const unsigned int Max_dh_size = 16;
const int index_bits = 30;
const int operation_bits = 2;
const int operation_mask = 0x3;

class Resident_object_table {
  //
  // Overview: The resident object table (ROT) contains headers for the
  //   resident objects in the FE, and is both a swizzle table and a handle
  //   table. Each object header in the ROT contains a reference count with
  //   the number of swizzled references and handles that refer to the object.
  //

public:
  //
  // Constructors:
  //

  Resident_object_table(Uint per_sz, Uint obj_sz, Storage_arena *sa);
  // Effects: Creates the resident object table using the arguments to
  //   determine the size of the swizzle table. "per_sz", and "obj_sz" are
  //   estimates of the byte sizes of the persistent cache, and an average object.
  //   "sa" is used to store the rot.

  ~Resident_object_table();
  // Effects: Destroys the ROT and deallocates all the storage
  //   associated with it.


  //
  // Swizzle table methods:
  //

  Core lookup(OR_num orx, Oref oref) const;
  // Effects: Looks for the object <or,oref> in the swizzle table. If
  //   the object is resident, returns a pointer to its core else returns null.

  void install_new_object(OR_num orx, Oref oref, Core c);
  // Requires: c is the core of a newly persistent object and <or,oref>
  //   is its newly assigned xref.
  // Effects: Installs c in the swizzle table. Calls PH and VH to move 
  //   newly persistent object into an appropriate page of the PH.

  Obj install_root(OR_num orx, Oref oref);
  // Requires: <or, oref> is the xref of an OR root object.
  // Effects: If object q with xref = <or,oref> is in the swizzle
  //   table returns a pointer to q. Otherwise, asks the persistent heap
  //   manager for q, installs q in the swizzle table and returns a
  //   pointer to q. In either case, the returned pointer points to q's
  //   class dispatch header, and can be used to invoke methods on q
  //   without requiring further initializations.


  //
  // Handle management methods:
  //

  Obj handle_to_obj(Handle h);
  // Effects: If h is a valid client handle returns the object it 
  //    refers to, otherwise returns 0. The object pointer returned
  //    points to the object's class dispatch header.


  Handle obj_to_handle(Obj o);
  // Requires: o is a valid object
  // Modifies: The ROT entry for o
  // Effects:  Returns o's handle (if o does not have a handle
  //    allocates a handle for o, and increments o's reference count). 


  unsigned free_handles(Handle *h, unsigned int sz);
  // Requires: h is an array of handles with sz elements.
  // Modifies: The ROT entries of objects referenced by valid handles in h
  // Effects: Frees all valid handles in h, and returns the number of valid
  //    handles in h. Decrements the reference counts of all entries 
  //    corresponding to freed handles.


  class Handle_generator {
    //
    // Overview: Generates all ROT entries with valid handles in some
    //   arbitrary order.  It starts by pointing to some ROT entry with a
    //   handle unless there are no such entries.
    //
    public:
      // Constructors:
      Handle_generator(const Resident_object_table *r);

      // Methods
      bool get(Core &c);
      // Modifies: "this"
      // Effects: If there is another unyielded entry, places it in "c"
      //   and returns true. Otherwise, returns FALSE.
    private:
      unsigned cur_ind;            // index of entry being yielded
      unsigned max_ind;            // index of free_area of rot
      const Resident_object_table *rot;  // ROT over which we are iterating
  };


 
  //
  // Reference count management methods:
  //

  void evict_object(Core c, OR_num orx);
  // Requires: c is the core of an object being evicted, and "or" is its
  //   OR's id.
  // Modifies: The entry of c, and the entries it refers to.
  // Effects: Decrements the reference counts for all the swizzled
  //   references in c and marks c's entry in the ROT as "discarded".
  //   The fields c points to are not modified.  Should be called during
  //   eviction for each evicted object.

  bool invalidate_object(OR_num orx, Oref oref);
  // Requires: An invalidation message was received for the object
  //   with xref <or,oref>
  // Effects: Invalidates the object if it is present in the cache
  //   and adjusts  If
  //   the object was not modified by the current transaction decrements
  //   the reference counts for all the swizzled references in
  //   it. Returns true iff the object was used in the current transaction.
  //   (If <or, oref> was modified, then the TM code needs to call
  //   adjust_rc_drop_copy if the copy exists.XXXX)

  void adj_rc_mod(Core target, Obj_x *ref, bool persistent);
  // Requires: target is a swizzled reference in a modified object,
  //   ref is the corresponding reference in the object's copy, and 
  //   persistent is true iff the modified object is persistent.
  // Effects: Increments target's reference count and if ref is swizzled
  //   and points to a core other than target logs that a reference count
  //   decrement needs to be performed later.
  //

  bool need_adj_rc_mod(Core c, Fields copy, bool persistent);
  // Requires: c is a modified object, copy is a pointer to its copy's
  //   first slot (or 0 if no copy exists), and persistent is true iff 
  //   c is persistent.
  // Effects: Returns true if adj_rc_mod needs to be called to adjust
  //   reference counts.

  void adj_rc_commit_request();
  // Requires: A commit request has just been sent to the OR.
  // Modifies: this
  // Effects: Adjusts reference counts to reflect the current state of
  //   the objects modified by the current transaction, and frees
  //   entries with null reference counts.  Also clears bits.  This is done
  //   while waiting for the commit reply, assuming that the transaction
  //   will commit. It creates an undo log. The swizzle table cannot be used
  //   until adj_rc_commit_reply is called.

  void adj_rc_commit_reply(bool committed); 
  // Requires: A transaction has ended (because a reply to a commit request
  //   was received from the OR or an invalidation caused it to abort). 
  //   Committed is true iff the transaction committed.
  // Modifies: this
  // Effects: If committed is true, commits any reference count adjustments
  //   reflecting modifications performed during the current transaction.
  //   Otherwise, it undoes these reference count adjustments and reverts 
  //   modified objects to their base copies (when possible).

  void adj_rc_during_trans(int cache_size);
  // Requires: A transaction is running (no commit request was sent and no
  //   invalidation caused the transaction to abort). cache_size i the 
  //   total PC size in bytes.
  // Effects: Adjusts reference counts to reflect reference counts of 
  //   modified objects and allow freeing of ROT entries during the
  //   transaction.


  //
  // Methods to index the table:
  //
  Core fetch(unsigned int index) const { return table+index; }
  // Requires: index is within bounds. 
  // Effects: Returns a pointer to the i-th core in the ROT.

  Core safe_fetch(unsigned int index) const { 
    return (index < size) ? table+index : 0; 
  }
  // Effects: Returns a pointer to the i-th core in the ROT or 
  //   0 if index is out of bounds.

  unsigned int index(Core el) const {return el - table; }
  // Requires: "el" is a pointer to a core in the ROT
  // Effects:  Returns the index of el

  unsigned int bytes() const {return  (free_area-table)*sizeof(Core_c); }
  // Effects: returns the number of bytes in active portion of the rot.

  //
  // Miscellaneous:
  //
  Bool_vector *get_stack_refs() {return stack_refs; }
  bool is_stack_referenced(Core c) { 
    return stack_refs->test(index(c));
  } //XXXX need to fix for multi core entries

  void reset_stack_referenced(Core c) { 
    stack_refs->reset(index(c));
  } //XXXX need to fix for multi core entries

  void process_zero_rc_log();
  // Effects: Processes the log of ROT entries that droped to zero and 
  //   frees them when possible.


  //
  // Methods used for manipulating statistics:
  //
  void print_stats(ostream &o);
  // Effects: Prints rot statistics in o.

  void reset_statistics();
  // Effects: Reset statistics.

  //------------------------------------------------------
  //                    PRIVATE SECTION
  //------------------------------------------------------

  // The next functions should really be private but they are used by
  // extern "C" functions.  

  Obj swizzle(Oref oref, Fields container, struct Class_s *t);
  // Requires: "container" is a valid object that contains an unswizzled
  //   reference with value "oref" and apparent type "t".
  // Modifies: "container", this.
  // Effects: If object q with xref = <or(container),oref> is in the swizzle
  //   table returns a pointer to q. Otherwise, asks the persistent heap
  //   manager for q, installs q in the swizzle table and returns a
  //   pointer to q. In either case, the returned pointer points to q's
  //   type "t" dispatch header, and can be used to invoke methods on q
  //   without requiring further initializations.
  //   If "container" was modified in the current transaction, marks
  //   "container" as "swizzled after copied".
  //   Otherwise increments reference count of q.

  Core allocate_object_raw(int dhsize, Uint size, bool variable_size);
  // Effects: Allocates a new (volatile) object of size bytes.
  //          if variable_size, object is created as a variable_sized
  //          object and an extra slot is allocated for the size field.
  //          dhsize is the number of dispatch header entries (typically 1)
  //          Initializes the objects's ROT entry and fields header except
  //          for the dispatch header.


  Obj allocate_object(Class_c cl, unsigned size=0);
  // Requires: "cl" is a valid class whose instances have at most
  //   Max_dh_size dispatch vectors in their dispatch header.
  //   size should only be supplied for variable sized objects
  //   (e.g., vectors).
  // Effects: Allocates a new (volatile) instance of class "cl", and
  //   initializes its ROT entry and fields header. Returns a pointer to
  //   the class dispatch vector. If a non-zero size is provided,
  //   this is a variable-sized object, and the size is specified in bytes.
  //   The first slot in the object is initialized to contain the size.
  //   The actual size of the object will be the number of slots required
  //   to contain that number of bytes.
  //
  //   Note that the size of the object does not include this first slot, so
  //   there are actually "size" bytes beyond the first slot.

  Fields make_resident(Core c);
  // Requires: "c" is discarded.
  // Effects: Fetches the fields of "c" from the pc and initializes "c"
  //   to point to its fields. It returns the fields of "c".

  void adj_rc_update(Shortp oldp, struct Core_s *newp);
  // Requires: newp points to a valid core and oldp, if swizzled, also points
  //   to a valid core. 
  // Effects: Decrements the reference count of the core pointed to by oldp and
  //   increments the reference count of the core pointed to by newp.

private:

  //
  // Private methods related to the swizzle table:
  //
 
  Core install_object(OR_num orx, Oref oref, OR_obj *r);
  // Requires: r is an OR object with xref <"or","oref">
  // Effects: Installs r in the swizzle table and returns a pointer to
  //   its core. It initializes the object core and fields.

  Core install_surrogate_target(OR_num surr_or, Oref surr_oref,
		                Xref target_xref);
  // Requires: <surr_or,surr_oref> is a surrogate for target_xref 
  // Effects: Calls  lookup_and_install on target_xref and adds
  //   mapping from target_xref to <surr_or,surr_oref> in the
  //   surrogate table.

  Core lookup_and_install(OR_num orx, Oref oref);
  // Effects: Looks for target_xref in the swizzle table. If not found
  //   fetches it from the ph and calls install_object on it. Returns
  //   a pointer to its core.

  unsigned orefs_bucket(Oref oref) const;
  // Effects: Returns the index of the bucket corresponding to oref.

  //
  // Private methods related to reference count management:
  //

  void inc_rc(Core c);
  // Requires: "c" is not a free entry
  // Effects: Increments c's count.

  void dec_rc(Core c);
  // Requires: "c" is not a free entry
  // Effects: Decrements c's count.

  void adj_rc_obj(Core c, int delta);
  // Requires: "c" is not a free entry and delta is 1 or -1.
  // Effects: Call inc_rc or dec_rc (depending on the value of delta)
  //   on the cores pointed to by swizzled references in c's fields.

  void adj_rc_copy(Fields copy, int delta);
  // Requires: copy is a pointer to a copy of a modified object
  //   and delta is 1 or -1.
  // Effects: Same as adj_rc_obj but for references in the copy.

  void adj_rc_prim(Obj_bitfield bf, unsigned num_slots, Obj_x *slots,
			  int delta);
  // Requires: "bf" and "num_slots" are adequate description of the 
  //   instance variables in slots.
  // Effects: This is a more primitive version of adjust_counts. That
  //   calls adjust_count("o", "delta") on each core "o" pointed by a
  //   swizzled reference in "slots" as described by "bf" and "num_slots".


  void adj_rc_copies();
  // Requires: Helper function for adjust_rc_commit_request.
  // Effects: Decrements reference counts for swizzled references that were
  //   overwritten by ending transaction.

  void free_entries();
  // Effects: Frees entries with null reference counts

  void expensive_free(Core c);
  // Requires: c is persistent, discarded, and has a null reference count
  // Effects: Marks c as free, removes it from its hash chain in the swizzle
  //   table and inserts it in the free list.

  bool check_rc_invariant() const;
  // Effects: Returns true iff the reference count fields in the ROT account
  //   for the swizled references in all non-discarded objects and the handles
  //   in the handle table.
  // Notes: For debugging purposes.

  bool check_free_invariant() const;
  // Effects: Returns true iff all entries in the ROT with a reference count of
  //   0 are free or non-discarded.
  // Notes: For debugging purposes.

  //
  // Private methods related to the allocation of ROT entries:

  Core allocate(unsigned int dh_size);
  // Requires: dh_size <= Max_dh_size
  // Effects: Allocates an entry with a dispatch header "dh_size"
  //   words long and returns a pointer to the last dh word. The entry
  //   is not initialized.



  //
  // General ROT representation:
  //
  Core_c *table; // The entire ROT as a table of Core_c
  unsigned int size; // The number of Core_c in the ROT
  Storage_arena *sa; // The storage arena allocated to hold the table.  

  //
  // Data structures for swizzling:
  // 
  unsigned int swiz_buckets; // Number of buckets in the swizzle table
                             // (it is always a power of 2).
  unsigned int shift_bits;   // Number of bits in an int minus the number
                             // of bits in swiz_buckets.
  Shortp *swiz_tab;          // The swizzle table bucket pointers.


  //
  //  Data structures for storage management:
  //
  class Free_list {
    //
    // Overview: Implements a list of free ROT entries.
    //
  public:
    Free_list();
    // Effects: Creates an empty free list.
    
    ~Free_list();
    // Effects: Destroys the list.

    Core remove(unsigned size);
    // Requires: size < Max_dh_size
    // Effects: Remove the first entry in the list capable of containing
    //   "size" dispatch header words. If there is no such entry returns 0.
    //   It returns a pointer to the last core in the entry.

    void insert(Core c);
    // Requires: "c" is marked free (and thus the number of dispatch 
    //     header words is stored in its first word).
    // Effects: Prepend "c" to the list. "c" should be the last core in the
    //     entry being freed (i.e. the "real" core of the object)
    
    void remove(Core c, unsigned size);
    // Requires: size < Max_dh_size, c is in the free list.
    // Effects: Remove c from the list.

  private:
    // Array of free lists for ROT entries.  free_lists[i-1] contains
    // entries whose size is i*sizeof(Core_c) bytes.
    Core *free_lists; 
  };

  Free_list free_list; // List of free ROT entries
  Core_c *free_area; // Pointer to the portion of the ROT that was never used


  //
  //  Data structures for handle table management: The
  // client_handle_table is a bitmap with one bit per each element in
  // the Core_c "table".  A bit is set iff the "table" element it
  // corresponds to is the Core of an object for which a handle was
  // given to the client. The value of the handle is the index of this
  // entry.
  Bool_vector *client_handle_table;   
  friend class Handle_generator;

  //
  // Data structures for reference count management:
  //

  // zero_rc_log1 logs indices of entries whose reference count dropped to zero 
  // but could not be discarded just because they were pointed to by the stack. The
  // entries in this log are processed right after the next stack scan and they
  // are freed (if possible) or moved to zero_rc_log2 (otherwise). zero_rc_log2
  // also logs indices of volatile objects whose reference count may be zero.
  // It is processed by free_entries at commit time.
  IntArray zero_rc_log1;
  IntArray zero_rc_log2;

  // Flag that is true iff handles were freed during the current
  // transaction.
  bool handles_freed;

  // Flag that is true iff it is safe to evict ROT entries when objects are evicted.
  bool free_during_trans;

  int evicted_trans; // Estimate of number of objects evicted during trans.

  // Volatile and persistent logs of swizzled references in copies 
  // of objects modified during the current transaction that are different 
  // in the new object version.
  Old_ref_log *vol_log;
  Old_ref_log *per_log;

  // Undo log used to adjust reference counts while waiting for the commit
  // reply assuming the transaction will commit.
  class RC_log {
    // Overview: Logs reference count adjustment operations.
  public:
    RC_log() {}
    // Effects: Creates an empty log.

    ~RC_log() {}
    // Effects: deletes the log.

    void mark_free(Core c);
    // Requires: "c" is not free.
    // Effects: Logs the fact that "c" will be marked free.

    void mark_free_and_enqueue(Core c);
    // Requires: "c" is not free and it is volatile.
    // Effects: Logs the fact that "c" will be marked free and inserted 
    //   in the free list.

    void clear();
    // Effects: Clears the log and deallocates its space.

    void predict_size(int);
    // Effects: Predicts the log size to improve performance.

    void undo();
    // Effects: Discards the the changes recorded in the log and clears the log.

    bool empty();
    // Effects: Returns true iff log is empty.

  private:
    Slot_array log;

    // The log entries have a header (a Slot) with the following
    // bitfields: core index:30 | operation:2 

    // The operations are:
    enum {Free = 1, Free_and_enqueue = 3};

    // There are two additional Slots. The Free
    // entry has the core's methods and oref; and the Free_and_enqueue
    // entry has the core's methods and dispatch header size.
  };
  friend RC_log;
  RC_log log; 
  bool commit_request; // true iff has executed adjust_rc_commit_request
                       // and has not executed adjust_rc_commit_reply

  Bool_vector *stack_refs;     // Keeps track of cores that may be referenced
                               // by the stack or registers.

  // 
  // Statistics:
  // 
  unsigned num_swizzles;           // Number of times swizzle is called.
  unsigned num_pc_lookups;         // Number of times lookup on pc is called.
  unsigned avg_inst_object;        // Average size (slots) of installed objects.
  unsigned num_installed;          // Number of objects installed.

  void print_swizzle_table_stats(ostream &o);
  // Effects: Prints swizzle table statistics.


  Timer rc_commit_adjustment;      // Time spent adjusting ref. counts at commit.
  unsigned num_mod_with_copies;    // Number of modified (not new) and new objects 
  unsigned num_mod_without_copies; // scanned to adjust reference counts at commit.
  unsigned num_new_scanned; 
  unsigned num_commits;            // Number of times adjust_rc_commit is called.

  Timer rc_free;                   // Time spent freeing cores at commit.
  unsigned num_dropped_to_zero1;   // Number of cores in dropped to zero before (1)
  unsigned num_dropped_to_zero2;   // and after (2) free_entries is called.
  unsigned num_frees;              // Number of times free_entries is called.
  unsigned num_marked_free;        // Number of entries marked free in free_entries.
};

// 
// Methods inlined for speed.
//

inline unsigned Resident_object_table::orefs_bucket(Oref oref) const {
  // Hash using only the value of the oref (we do not expect that
  // ignoring the or value when producing the hash will result
  // in many collisions because the probability an FE caches two
  // objects with the same oref from different ORs should be low).
  // Note that this code requires swiz_buckets to be a power of 2.
  return hash_unsigned(oref) >> shift_bits;
}


inline Core Resident_object_table::lookup(OR_num orx, Oref oref) const {
  Core tmp = SHORTP_AS_FULL(Core_c, swiz_tab[orefs_bucket(oref)]);
  for (; tmp != 0; tmp = tmp->next()) 
    if (tmp->has_xref(orx, oref)) return tmp;

  return 0;
}


inline void Resident_object_table::inc_rc(Core c) {
  th_assert(!c->is_free(), "Adjusting ref count on free entry");

  // Increment the reference count of target
  int ref_cnt = c->ref_count() + 1;
  th_assert(ref_cnt < 255, "Large reference counts are not implemented");
  c->set_ref_count(ref_cnt);
}


inline void Resident_object_table::evict_object(Core c, OR_num orx) {
  // Breaking into reference iteration abstraction must be kept in sync
  // with objref.h

#if USE_REF_COUNTS
  register Obj_bitfield bf = c->bitfields();
  register int num_slots = c->num_slots();
  register int max = (OBJ_BF_ISSPECIAL(bf)) ? OBJ_BF_SIZE_F(bf) : num_slots;
  register Obj_x *slots = (Obj_x *) c->get_fields()->slots();
  register int i = 0;

  while (1) {
    for (; i < max; i++) {
      if (slots[i].is_pointer() && OBJ_BF_ISREF(bf, i%INT_BITS)) {
        Core target = slots[i].pointer()->to_core();
	int ref_cnt = target->ref_count()-1;
	target->set_ref_count(ref_cnt);
	if (ref_cnt == 0) {
	  if (target->is_discarded()) {
	    if (!is_stack_referenced(target) && free_during_trans) {
	      target->mark_free();
	    } else {
	      reset_stack_referenced(target);
	      zero_rc_log1.append(index(target));
	    }
	  }
	}
      }
    }
  
    if (max == num_slots) break;

    if (OBJ_BF_REFS(bf)) {
      max = num_slots;
      bf = ~0;
    } else
      break;
  }
#endif

  // Mark core as discarded.
  c->mark_discarded(orx);

#if USE_REF_COUNTS
  if (c->ref_count() == 0) {
    if (!is_stack_referenced(c) && free_during_trans) {
      c->mark_free();
    } else {
      reset_stack_referenced(c);
      zero_rc_log1.append(index(c));
    }
  }
#endif

}


inline void Resident_object_table::adj_rc_mod(Core target, Obj_x *ref, bool persistent) {
  Old_ref_log *olog = (persistent) ? per_log : vol_log;

  if (ref->is_data()) {
    // Reference is swizzled in object but not in copy increment
    // count for core pointed by object
    inc_rc(target);
  } else {
    Core copy_target = ref->pointer()->to_core();
    if (target != copy_target) {
      // If they are different add 1 to count of core pointed by
      // object and log that copy\'s reference is different.
      inc_rc(target);
      olog->log_reference(ref);
    }
  }
} 


#endif //_ResidentObjectTable_h


