#ifndef _Scanned_set_h 
#define _Scanned_set_h 1

#include "compiler/C++wrappers/fields.h"

#include "utils/basic.h"
#include "utils/array.h"

declareArray(Scanned_array, unsigned)

const unsigned Extra_usage_bits = 2;
const unsigned Scanned_usage_bits = Max_usage_bits + Extra_usage_bits;
const unsigned Max_scanned_usage = (1 << Scanned_usage_bits) - 1;
const unsigned Scanned_index_bits = sizeof(unsigned)*8 - Scanned_usage_bits;
const unsigned Max_scanned_index = (1 << Scanned_index_bits) - 1;

class Scanned_set {
  //
  // Overview: The system scans a page to compute its usage using the
  //   usage of its objects. Scanned_set contains the index and usage of
  //   the last pages scanned. The set contains pages from n different
  //   scans. A scan is a sequence of insertions between consecutive
  //   removals. It corresponds to a group of pages scanned while waiting
  //   for the same fetch reply.  As pages from new scans are inserted in
  //   the set, the pages from the oldest scan in the set are removed.
  //
public:
  Scanned_set(unsigned nscans);
  // Effects: Creates a set with pages from "nscans" different scans.

  ~Scanned_set();
  // Effects: Destroys the set and all the associated storage.

  unsigned num_remembered(void);
  // Effects: Returns the number of scans remembered by the scanned set.

  void insert(unsigned short index, unsigned usage);
  // Requires: usage <= Max_scanned_usage and index should not have an entry in
  //   the scanned set for performance reasons.
  // Effects: Inserts a page with index "index" and usage "usage" in
  //   the set.

  bool remove(unsigned short &index, unsigned short &usage);
  // Effects: Removes the page with lowest usage from the set, returns
  //   true, sets "index" to the page index. If the set is empty returns
  //   false.


  void end_scan(void);
  // Effects: Ends the scan.

private:
  bool slow_remove(unsigned short &index, unsigned short &usage);
  // Requires: usage == min_usages[current];
  // Effects: Removes the page with lowest usage from the set, returns
  //   true, sets "index" to the page index. If the set is empty returns
  //   false.

  Scanned_array *set;    // Array with one scanned array per scan.
  unsigned nscans;       // Number of scans in set.		      
  int current;           // Index of scanned array being used.

  int *min_usages;     // Array with minimum usage in each scan.

  Scanned_array *bins;   // Bins used for bin sorting.
  unsigned num_bins;

  //
  // The next methods manage an unsigned as a bitfield with
  // usage:Scanned_usage_bits | index:Scanned_index_bits.
  //
  inline unsigned make_index_usage_pair(unsigned index, unsigned usage) {
    return ((usage << Scanned_index_bits) | index);
  }
  inline unsigned get_usage(unsigned pair) { return pair >> Scanned_index_bits;}
  inline unsigned get_index(unsigned pair) { return pair & Max_scanned_index;}
};

inline  unsigned Scanned_set::num_remembered(void) { return nscans; }

inline void Scanned_set::insert(unsigned short index, unsigned usage) {
  th_assert(usage <= Max_scanned_usage, "Usage larger than maximum\n");
  bins[usage].append(index);
}


inline bool Scanned_set::remove(unsigned short &index, unsigned short &usage) {
  unsigned min_index = current;
  usage = min_usages[current];
  for (int i=0; i < (int)nscans; i++) {
    if (min_usages[i] < (int)usage) { 
       usage = min_usages[i]; 
       min_index = i; 
    }
  }

  // Set is empty.
  if (usage == Max_scanned_usage + 1) return false;

  index = get_index(set[min_index].remove());
  min_usages[min_index] =  
    (set[min_index].size()) ? get_usage(set[min_index].high()) : Max_scanned_usage+1;
  
  return true;
}
  
#endif //_Scanned_set_h

