#ifndef _SURROGATE_TABLE_H
#define _SURROGATE_TABLE_H

#include "utils/array.h"
#include "utils/openhashmap.h"
#include "common/xref.h"
#include "common/xrefs.h"

declareOpenHashMap(Xref_to_Xrefs, Xref, Xrefs*, Xref_hash, Xref_equal)


class Surrogate_table {
    // Overview: Implements the surrogate table, which maps an xref to its OR
    //           surrogates at different ORs (The map is Xref ---> Set of xrefs.
    //           It allows the FE to reuse OR surrogates when commiting the
    //           new states of modified objects.

  public:
    // Constructors:
    Surrogate_table();
    // Effects: Creates an empty surrogate table.

    ~Surrogate_table();
    // Effects: Destroys the surrogate table.

    // Methods:

    Oref lookup(OR_num orx, Oref oref, OR_num surr_or);
    // Effects: Searches the table for a surrogate at or "surr_or"
    //          corresponding to the object <or, oref>. Returns the oref of
    //          the surrogate if it finds one, else returns the Null_oref.
    //          (This method is called by the commit code when it is
    //          unswizzling modified objects and finds an inter-OR reference
    //          to object <or, oref>)

    void insert(OR_num orx, Oref oref, OR_num surr_or, Oref surr_oref);
    // Requires: <surr_or, surr_oref> refers to an OR surrogate for <or, oref>
    // Effects: Inserts a mapping from <or,oref> to <surr_or, surr_oref>in the
    //          table. If an OR surrogate for <or, oref> at "surr_or" already
    //          exists in the table, nothing is done

    void remove(OR_num orx, Oref oref);
    // Effects: Removes the surrogates for <or, oref> from the table
    //   If there are any.

    class Iter {
	//  An iterator for yielding the surrogate xrefs

	//  Once created, an iterator must be used and destroyed before any
	//  objects are added to the read/write set.
	//  The effect is undefined if an iterator method is called after
	//  such a change.
	
      public:
	Iter(Surrogate_table* table);
	// effects Return an iterator over the surrogate table "table"

	bool get(Xref &dest, Xref &surr);
	// modifies "surr", "dest"
	// effects  Sets "surr", "dest" to the next mapping and returns true.
	//          I.e. "surr" is a surrogate for "dest"
	//          Returns false if there are no more mappings (and does not
	//          modify "source", "dest")

    private:
	Xref_to_Xrefs::Bindings iter; // Iterator for yielding the mappings
	Xref dest_xref; // The current xref being yielded
	Xrefs *current; // Current surrogate array being yielded
	int index;      // Element being yielded from current
    };

  private:
    friend Iter;
    // Surrogate table rep.
    Xref_to_Xrefs rep;  // Maps xref -> set of xrefs
};


#endif /* _SURROGATE_TABLE_H */
