#ifndef _C_COMPILER_H
#define _C_COMPILER_H

#include <stddef.h>
#include <stdlib.h>
#include <malloc.h>

#include "obj.h"
#include "residency.h"
#include "fields.h"
#include "methods.h"
#include "dv.h"
#include "basic_types.h"
#include "math.h"
#include "flags.h"
#include "array.h"
#include "exceptions.h"
#include "extern.h"

#include "common/slot.h"
#include "common/shortp.h"
#include "common/null_or_ref.h"

#include "config/vdefs/DV_RESIDENCY_CHECK.h"
#include "config/vdefs/SELF_RESIDENT.h"
#include "config/vdefs/COLLECT_EXECUTION_TRACE.h"
#include "config/vdefs/OPTIMAL_PAGE_REPLACEMENT.h"
#include "config/vdefs/LRU_PAGE_REPLACEMENT.h"

/* This include file defines the runtime interface used by the compiler.
   
   About macros
   ------------

   This file and the others in the fe/compiler/C directory define macros
   and provide an essentially procedural specification for the macros.
   Although the specifications look like C++ procedure specifications,
   they are intended to be called from C. Macros that are declared to
   return a "statement" actually generate a C statement, and may be
   used in any context where a statement is permitted. They may not be
   used as expressions. Arguments of type "type" require a type
   name. For example, the type "Foo" would be passed as "Foo", without
   any quoting.

   C types and variables
   ---------------------

   For any type T, there are a number of associated C types and variables:

   A type T is associated with:

    T                   : A pointer to a struct T_s

    struct T_s          : A portion of the ROT entry for an object of type T

    struct Class_s *T_V : Pointer to the corresponding class object


   If type T is also a class, then it is associated with the following
   additional C types and variables: 
   (Interfaces in Java are types but not classes.)

    struct DV_hdr_s     : The header on every T_DV

    struct T_DV         : The dispatch vector for class T
                          Contains only the virtual methods in type T

    struct T_bidirectional_DV : The bidirectional dv for T
                          Contains all instance methods on the -ve side

                          Also contains pointers to:
                          T'_P_s for all superclasses T'of T including T

    struct T_P_s        : Contains ptrs to T'_V for all T' used in all
                          methods M actually implemented in class T

                          Also contains ptrs to M_htable for all above M 
                          (for exception handling)

    struct T_f_s        : The fields for an object of class T.


   For any class T, information is split between the structs Class_s,
   DV_hdr_s, and T_P_s in the following way:

    T_P_s is the only variable sized struct among the 3.  At a later
    point of time, T_bidirectional_DV could actually contain T'_P_ss
    rather then pointers to them.

    DV_hdr_s contains info specific to classes, while Class_s contains info
    common to classes and types.


   A class T can also contain static methods and fields.  This is treated as
   another class T_static which has exactly one instantiation.
   Consequently, for a class T, we have the following additional C types and
   variables.  

   The static types/variables however do not contain methods/fields of
   their superclasses, since the Java bytecodes always refer to the exact
   class and not some subclass when using statics.

    T_static
    struct T_static_s
    struct Class_s *T_static_V 
    struct DV_hdr_s
    struct T_static_DV
    struct T_static_bidirectional_DV
    struct T_static_P_s
    struct T_static_f_s


   Defining ROT entries
   --------------------

    The type T_s is defined using the DECLARE_type macro:

    DECLARE_type(type T)
      // Declare the ROT entry type T_s corresponding to T

    Defining object fields
    ----------------------

    New object field structures are defined in the following way. With
    each class "T" is an associated C structure "struct T_f_s",
    which defines the layout of its fields. If "T" has a superclass "S",
    the structure has the following form:

        struct T_f_s {
            struct S_f_s inh;
            ... field decls ...
        };
    
    If it has no superclass, the structure has

        struct T_f_s {
            struct Fields_s inh;
            ... fields decls ...
        };

    In the field declarations, the types are either the corresponding
    primitive C type, or the type "Shortp" in the case of a pointer type.
 

    An array containing elements of type FT has the type FT_Array.  It is a
    subtype of java.lang.Object, and has two additional fields: int length,
    and FT elem[N], where N is the size of the array.  Also, the clone
    method should be overloaded in the array.  */

#define DECLARE_type(T)                                                       \
    typedef struct T##_s {                                                    \
        Shortp dv;                                                            \
    } *T;                                                                     \


/*  All the type-specific structures are declared by the compiler generated
       code. */

/*
    Defining dispatch vectors
    -------------------------

    The dispatch vector for a class T, struct T_DV, contains the all the
    methods of its superclass, in the same exact order as in the
    superclass, followed by the methods of T that are not also methods of
    the superclass.

    For example, say S is superclass of T.  Also, let M_S1 ... M_Sn be the
    methods of the S, and M_T1 ... MTm be the additional methods in T.
    Then S_DV and T_DV would look like this.

        struct S_DV {
            struct DV_hdr_s hdr;
            M_S1
            M_S2
            ...
            M_Sn
        };

        struct T_DV {
            struct DV_hdr_s hdr;
            M_S1
            M_S2
            ...
            M_Sn
            M_T1
            M_T2
            ...
            M_Tm
        };
    
    Calling virtual methods
    -----------------------

    The DISPATCH_V_D method is used to call a method on a discovered
    object that may not be resident, and the DISPATCH_V_R method is
    used when the object is known to be resident. (See residency.h
    for a discussion of object residency).

    Consider that method "m" has the C signature "msig", where "OT"
    is the object's type:

        typedef Tr msig(OT, Ta1, Ta2, ...);
    
    Then, it can be called with either of two macros, depending on the
    object's state. For example, we should do the following:

        Tr tempR = DISPATCH_V_R(o,m,OT)(o,a1,a2,...);

    Note that in the following, "OT" must support the method "m".

        msig *DISPATCH_V_D(OT o, symbol m, type OT)
        msig *DISPATCH_V_R(OT o, symbol m, type OT)
            Return a function pointer for the method "m" of the object
            "o", with the appropriate type for that method. If DISPATCH_V_R
            is used, the object must be resident.


    Calling non-virtual methods
    ---------------------------

    These are exactly identical to the above.  But since these are
    non-virtual methods, they are fetched from the class OT's dispatch
    vector rather than the object O's dispatch vector.

        msig *DISPATCH_N_D(OT o, symbol m, type OT)
        msig *DISPATCH_N_R(OT o, symbol m, type OT)


    Calling interface methods
    -------------------------

    Similar macros are defined for calling interface methods.  The main
    difference is, instead of calling with the method name, they are called
    with the hashcode of the method signature.

        msig *DISPATCH_I_D(OT o, int h)
        msig *DISPATCH_I_R(OT o, int h)
            Return a function pointer for the method of the object
            "o" that has hachCode "h", by doing a runtime lookup. If 
            DISPATCH_I_R is used, the object must be resident.


    NOTE: DISPATCH_[VNIS]_D currently does extra work to check for residency.
    The work can be avoided by putting a special dispatch header on
    non-resident objects. The current implementation incurs a performance
    penalty for its simplicity. 

*/

#define SETUP_SELF(O,OT)                                                      \
    struct OT##_bidirectional_DV *self_dv = GET_BIDIRECTIONAL_DV(O, OT);      \
    struct OT##_P_s *self_P = self_dv->OT##_P;

#define SETUP_STATIC_SELF(O,OT)                                               \
    struct OT##_static_bidirectional_DV *self_dv =                            \
        GET_BIDIRECTIONAL_DV(O, OT##_static);                                 \
    struct OT##_P_s *self_P = self_dv->  OT##_P;

#define DISPATCH_S(M,OT)                                                      \
  (((struct OT##_static_DV *)(self_P-> OT##_static_V->dv))->M)

#define DISPATCH_V_R(O,M,OT) (GET_DV(O, OT)->M)

#define DISPATCH_N_R(O,M,OT) (GET_BIDIRECTIONAL_DV(O, OT)->M)

#define DISPATCH_I_R(O,H) (Dispatch_Interf(                                   \
    GET_DV(O, Obj)->hdr.methods,                                              \
    GET_DV(O, Obj)->hdr.num_methods, H))


#if DV_RESIDENCY_CHECK || !SELF_RESIDENT
/*
   Either the dispatch will make it resident, or else it doesn't need to
   be resident.
*/
#  define DISPATCH_V_D(O,M,OT)   (DISPATCH_V_R(O,M,OT))
#  define DISPATCH_N_D(O,M,OT)   (DISPATCH_N_R(O,M,OT))
#  define DISPATCH_I_D(O,H) (DISPATCH_I_R(O,H))
#else
/*
   Make sure the object is resident on entry.
*/
#  define DISPATCH_V_D(O,M,OT)   (RESIDENT_OBJ(O),DISPATCH_V_R(O,M,OT))
#  define DISPATCH_N_D(O,M,OT)   (RESIDENT_OBJ(O),DISPATCH_N_R(O,M,OT))
#  define DISPATCH_I_D(O,H) (RESIDENT_OBJ(O),DISPATCH_I_R(O,H))
#endif


/*
    Preparing for reads and writes
    ------------------------------

    An object may be in one of four modification states: UNKNOWN, READ,
    WRITTEN, or PWRITTEN (with modified pointers). These states are
    nested, so a WRITTEN object has also been READ. Before object
    fields may be accessed, the PREPARE_READ and PREPARE_WRITE macros
    must be used to ensure the object is in the right state.

        statement PREPARE_READ(Core_p x);
        statement PREPARE_READ_F(Core_p x, Fields_p v);
        statement PREPARE_WRITE(Core_p x);
        statement PREPARE_WRITE_F(Core_p x, Fields_p v);
          Requires: x is RESIDENT
          Effects:
            Make sure that "x" is in the READ/WRITTEN state. "v"
            must be the fields of "x" where applicable. These macros
            never downgrade the modification state of the object; for
            example, calling PREPARE_READ on a WRITTEN object leaves
            the object in the WRITTEN state. The _F macros are slightly
            faster if the fields are already available.

    To modify a pointer field in an object, the object must be
    in the PWRITTEN state.

        statement PREPARE_PWRITE(Core_p x);
        statement PREPARE_PWRITE_F(Fields_p v);
            Requires: The object must be in the PWRITTEN or WRITTEN state,
               and must be RESIDENT.
            Effects: Move the object to the "PWRITTEN" state.
                    
*/

#define PREPARE_READ_F(X,V)                                                   \
    _BEGIN_ MARK_USED_F(V); if (!IS_READ(V)) Log_read((Core_p)(X)); _END_
#define PREPARE_WRITE_F(X,V)                                                  \
    _BEGIN_ MARK_USED_F(V); if (!IS_WRITTEN(V)) Log_write((Core_p)(X)); _END_
#define PREPARE_READ(X) PREPARE_READ_F(X, FIELDS(X))
#define PREPARE_WRITE(X) PREPARE_WRITE_F(X, FIELDS(X))
#define PREPARE_PWRITE_F(V) MARK_PWRITTEN(V)
#define PREPARE_PWRITE(X) PREPARE_PWRITE_F(FIELDS(X))


/* On entry to a method, the compiler must mark an object as "used" with
   one of the following macros:

   statement MARK_USED(Core_p x);
   statement MARK_USED_F(Fields_p v);
      Effects: Marks the object denoted by c or v as being used recently. */

#if COLLECT_EXECUTION_TRACE 
    /* Collect execution trace */
#   define MARK_USED_F(V)                                                     \
    _BEGIN_                                                                   \
        SET_USAGE(V);                                                         \
        Execution_trace((struct Fields_s *)V);                                \
    _END_
#else
#   if OPTIMAL_PAGE_REPLACEMENT || LRU_PAGE_REPLACEMENT
        /* Perform optimal page replacement */
#       define MARK_USED_F(V) (Access_counter++)
#   else
        /* Normal case: simply mark object as used */
#       define MARK_USED_F(V) SET_USAGE(V)
#   endif
#endif


#define MARK_USED(X)   SET_USAGE(GET_FIELDS(X))

/*
   Objects may be in several residency states. See "runtime/C/residency.h"
   for details.

   Object fields are accessed using the following macros. First, some
   macros for reading object fields:

    Common requires clauses for the macros below:

    GET{A}[PV]_R 
     Requires: the object must be RESIDENT

    GET{A}[PV]_F
     Requires: the object must be fixed by "v"

    Common effects clauses for the macros below:

    GET{A}P_[DRF]
     Effects: the accessed field is automatically discovered.
       The return value is RESIDENT.


   statement GETP_F(FT &result, struct OT_f_s *v, symbol f, type OT, type FT)
   statement GETP_R(FT &result, OT o, symbol f, type OT, type FT)
   statement GETP_D(FT &result, OT o, symbol f, type OT, type FT)
    Effects: Load a pointer value from the field "f" of the object
      "o" and place it in "result".  The static type of "o" must be
      "OT".  The field "f" must have type "FT".


   statement GETV_F(FT &result, struct OT_f_s *v, symbol f, type OT)
   statement GETV_R(FT &result, OT o, symbol f, type OT)
   statement GETV_D(FT &result, OT o, symbol f, type OT)
    Effects: Load a non-pointer value from the field "f" of the object "o".
      The static type of "o" must be "OT". The return type is the type of the
      field "f". 

   statement GETAP_F(FT &result, struct Shortp_Array_f_s *v, int i, type FT)
   statement GETAP_R(FT &result, Shortp_Array a, int i, type FT)
   statement GETAP_D(FT &result, Shortp_Array a, int i, type FT)
    Effects: Load a pointer value of type FT from a[i] and place it in "result"

   statement GETAV_F(FT &result, struct FT_f_s *v, int i, type FT)
   statement GETAV_R(FT &result, FT_Array a, int i, type FT)
   statement GETAV_D(FT &result, FT_Array a, int i, type FT)
    Effects: Load a non-pointer value of type FT from a[i] and place it in
    "result"

*/

#define GETP_F(R,V,FN,OT,FT) RESIDENT_SHORTP(R,V,FN,OT,FT,self_P->FT##_V)
#define GETP_R(R,O,FN,OT,FT) GETP_F(R,FIX_FIELDS(O,OT),   FN,OT,FT)
#define GETP_D(R,O,FN,OT,FT) GETP_F(R,RESIDENT_CORE(O,OT),FN,OT,FT)

#define GETV_F(R,V,FN,OT) _BEGIN_ R = (V)->FN; _END_
#define GETV_R(R,O,FN,OT) GETV_F(R, FIX_FIELDS(O,OT),FN,OT)
#define GETV_D(R,O,FN,OT) GETV_F(R, RESIDENT_CORE(O,OT),FN,OT)

#define GETAP_F(R,V,I,FT) GETP_F(R,V,elems[I],Shortp_Array,java_lang_Object)
#define GETAP_R(R,A,I,FT) GETAP_F(R,FIX_FIELDS(A,Shortp_Array),   I,FT)
#define GETAP_D(R,A,I,FT) GETAP_F(R,RESIDENT_CORE(A,Shortp_Array),I,FT)

#define GETAV_F(R,V,I,FT) GETV_F(R,V,elems[I],FT##_Array)
#define GETAV_R(R,A,I,FT) GETAV_F(R,FIX_FIELDS(A,FT##_Array),   I,FT)
#define GETAV_D(R,A,I,FT) GETAV_F(R,RESIDENT_CORE(A,FT##_Array),I,FT)

/*
    Similar macros are defined for writing into object fields:

    Common requires clauses for the macros below:

    PUT{A}[PV]_R 
     Requires: the object must be resident

    PUT{A}[PV]_F 
     Requires: the object must be fixed by "v". 
       "v" must be a variable name or an expression without side effects.

    void PUT[PV]_F(struct OT_f_s *v, symbol f, FT x, type OT)
    void PUT[PV]_R(OT o, symbol f, FT x, type OT)
    void PUT[PV]_D(OT o, symbol f, FT x, type OT)
      Effects: Store the value "x" into the field "o.f".

    void PUTAP_F(struct Shortp_Array_f_s *v, int i, FT x, type FT)
    void PUTAP_R(Shortp_Array a, int i, FT x, type FT)
    void PUTAP_D(Shortp_Array a, int i, FT x, type FT)
      Effects: Store the value "x" into the field "a[i]".

    void PUTAV_F(struct FT_Array_f_s *v, int i, FT x, type FT)
    void PUTAV_R(FT_Array a, int i, FT x, type FT)
    void PUTAV_D(FT_Array a, int i, FT x, type FT)
      Effects: Store the value "x" into the field "a[i]".

*/

#if USE_REF_COUNTS
#    define PUTP_F(V,FN,X,OT)                                                 \
    _BEGIN_                                                                   \
        if (USE_EAGER_REF_COUNTING((Fields_p)(V)))                            \
            Eager_ref_count_update((Shortp)((V)->FN), (Core_p)(ptrdiff_t)X);  \
            (V)->FN = FULL_AS_SHORTP(X);                                      \
    _END_
#else
#    define PUTP_F(V,FN,X,OT) ((V)->FN = FULL_AS_SHORTP(X)) 
#endif

#define PUTP_R(O,FN,X,OT) PUTP_F(FIX_FIELDS(O,OT), FN, X, OT)
#define PUTP_D(O,FN,X,OT) PUTP_F(RESIDENT_CORE(O,OT), FN, X, OT)

#define PUTV_F(V,FN,X,OT) ((V)->FN = (X))
#define PUTV_R(O,FN,X,OT) PUTV_F(FIX_FIELDS(O,OT), FN, X, OT)
#define PUTV_D(O,FN,X,OT) PUTV_F(RESIDENT_CORE(O,OT), FN, X, OT)

#define PUTAP_F(V,I,X,FT) PUTP_F(V,elems[I],X,Shortp_Array)
#define PUTAP_R(A,I,X,FT) PUTAP_F(FIX_FIELDS(A,Shortp_Array), I,X,FT)
#define PUTAP_D(A,I,X,FT) PUTAP_F(RESIDENT_CORE(A,Shortp_Array), I,X,FT)

#define PUTAV_F(V,I,X,FT) PUTV_F(V,elems[I],X,FT##_Array)
#define PUTAV_R(A,I,X,FT) PUTAV_F(FIX_FIELDS(A,FT##_Array), I,X,FT)
#define PUTAV_D(A,I,X,FT) PUTAV_F(RESIDENT_CORE(A,FT##_Array), I,X,FT)


/* Object allocation
   -----------------

   Objects of fixed size are allocated with the ALLOC_OBJ macro. The
   ALLOC_ARRAY macro is used to allocate arrays.  New objects are created
   in the WRITTEN state.

   ALLOC_OBJ(OC &result, class OC)
    
    Create a new object of class OC and store a reference to it in "result".
    The fields of the new object are uninitialized.

   ALLOC_ARRAY_P(Shortp_Array &result, type FT, int size, int Ndim)
   ALLOC_ARRAY_V(FT_Array &result, type FT, int size, int Ndim)
    
    Create a new array object of Ndim dimensions with space for "size"
    items of type "FT", and store a reference to it in "result". The fields
    of the new object are uninitialized.
    "ALLOC_ARRAY_V" is for arrays of primitive types, while,
    "ALLOC_ARRAY_P" is for arrays of objects.

   ALLOC_ARRAY_P_I(Shortp_Array &result, type FT, int size, int Ndim, FT e)
   ALLOC_ARRAY_V_I(FT_Array &result, type FT, int size, int Ndim, FT e)
    
    Same as above, except that it initializes all the fields to "e".
    Warning: If this is used to initialize an array of objects to some
    non-null value, then one should also call PREPARE_PWRITE on the array
    object.

   DEALLOC_OBJ(OBJ o)

    Indicate that there are no more pointers to the object "o".  */

#define ALLOC_RAW_G(R,OC,SZ,OC_V)                                               \
    _BEGIN_                                                                   \
        R = (OC)Allocate_object(OC_V, SZ);                                    \
        Log_new((struct Core_s *)R);                                          \
    _END_ 

#define ALLOC_RAW(R,OC,SZ,OC_V) ALLOC_RAW_G(R,OC,SZ,self_P->OC_V)

#define ALLOC_OBJ_G(R,OC) ALLOC_RAW_G(R,OC,0,OC##_V)

#define ALLOC_OBJ(R,OC) ALLOC_RAW(R,OC,0,OC##_V)

#define MAX_ARRAY_SZ (SLOTS * Slot_size)

#define ARRAY_SIZE(SZ,FT,N)                                                   \
    _BEGIN_                                                                   \
        SZ = sizeof(int) + sizeof(FT[N]);                                     \
        /* If size is zero, the ROT assumes the object is fixed size */       \
        /* if (SZ == 0) SZ = 1; */                                            \
        /* But SZ can never be 0 because of sizeof(int) */                    \
        if (SZ > MAX_ARRAY_SZ)                                                \
          ;                                                                   \
    _END_

#define ALLOC_ARRAY_RAW_G(R,FT,N,FT_V)                                        \
    _BEGIN_                                                                   \
        unsigned SZ;                                                          \
        if (N < 0)                                                            \
          ;                                                                   \
        ARRAY_SIZE(SZ,FT,N);                                                  \
        ALLOC_RAW_G(R,FT##_Array,SZ,FT_V);                                    \
        PUTV_R(R, length, N, FT##_Array);                                     \
    _END_ 

#define ALLOC_ARRAY_RAW(R,FT,N,FT_V)  \
    ALLOC_ARRAY_RAW_G(R,FT,N,self_P->FT_V)

#define ALLOC_ARRAY_P(R,FT,N,Ndim)                                            \
    ALLOC_ARRAY_RAW(R,Shortp,N,FT##_Array_##Ndim##_V)

#define ALLOC_ARRAY_V(R,FT,N)                                                 \
        ALLOC_ARRAY_RAW(R,FT,N,FT##_Array_1_V)


#define INIT_ARRAY_RAW(R,FT,N,E)                                              \
    _BEGIN_                                                                   \
        int i;                                                                \
        struct FT##_Array_f_s *f = (struct FT##_Array_f_s *) GET_FIELDS(R);   \
        for(i=0; i<N; i++) {                                                  \
          f->elems[i] = E;                                                    \
        }                                                                     \
    _END_ 

#define ALLOC_ARRAY_P_I(R,FT,N,Ndim,E)                                        \
    _BEGIN_                                                                   \
        ALLOC_ARRAY_RAW(R,Shortp,N,FT##_Array_##Ndim##_V);                    \
        INIT_ARRAY_RAW(R,Shortp,N,FULL_AS_SHORTP(E));                         \
    _END_ 

#define ALLOC_ARRAY_V_I(R,FT,N,E)                                             \
    _BEGIN_                                                                   \
      ALLOC_ARRAY_RAW(R,FT,N,FT##_Array_1_V);                                 \
      INIT_ARRAY_RAW(R,FT,N,E);                                               \
    _END_ 

#define DEALLOC_OBJ(o) _BEGIN_ _END_

/*
    Hacks for initialization
*/

#define NEW(T) ((T *)Alloc_class(sizeof(T)))
#define NEW_CLASS() NEW(struct Class_s)

/*
   INIT_DV_I(struct Class_s *&var, class C)
   INIT_DV(class C)
*/

#define CREATE_DV_I(CLASS_VAR,CLASS_DV,DV_STRUCT)                             \
    _BEGIN_                                                                   \
        DV_STRUCT *TMP =  NEW(DV_STRUCT);                                     \
        memcpy((char *)TMP, (char *)&CLASS_DV, sizeof(DV_STRUCT));            \
        CLASS_VAR->dv = &TMP->hdr;                                            \
        CLASS_VAR->dv->self_class = CLASS_VAR;                                \
    _END_

#define CREATE_DV(CLASS)                                                      \
    CREATE_DV_I(CLASS##_V, The_##CLASS##_DV,                                  \
        struct CLASS##_bidirectional_DV)

#define INSTANCE_OF(O,C) \
    InstanceOf(O, self_P->  C##_V, 0)

#define ARRAY_INSTANCE_OF(O,C,N) \
    InstanceOf(O, self_P->##C##_Array_##N##_V, 0)

#define THROW_G(ET)                                                           \
    _BEGIN_                                                                   \
    {                                                                         \
        ET e;                                                                 \
        ALLOC_OBJ_G(e, ET);                                                   \
        Throw(e);                                                             \
    }                                                                         \
    _END_

#define THROW(ET)                                                             \
    _BEGIN_                                                                   \
    {                                                                         \
        ET e;                                                                 \
        ALLOC_OBJ(e, ET);                                                     \
        Throw(e);                                                             \
    }                                                                         \
    _END_


#endif /* _C_COMPILER_H */

