#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/class.h"

#include "trans_log.h"
#include "commit_info.h"

#include "cache1/persistent_cache.h"
#include "cache1/surrogate_table.h"
#include "cache1/resident_object_table.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/USE_REF_COUNTS.h"

#include "common/or_num.h"
#include "common/objrefs.h"
#include "common/or_obj.h"
#include "common/transaction.h"
#include "common/ros.h"
#include "common/modset.h"

#include "utils/map.h"
#include "utils/tstamp.h"
#include "utils/sort.h"
#include "utils/mdebug.h"


// Instantiate the Hash table
#define IMPL_KEY  Core_key
#define IMPL_VALUE Xref
#include "utils/impl_map.t"

#define GENERATOR_ELEMENT Core_key
#include "utils/impl_generator.t"

#define PTRKEY Core_c
#include "utils/impl_ptrkey.t"


// Instantiate the Hash table
#define IMPL_KEY  UIntKey
#define IMPL_VALUE OR_commit_data
#include "utils/impl_map.t"

static OR_set empty_or_set;  // An empty OR_set to send to participants 

Commit_info::Commit_info() {
    const Uint Ntable_size = 100;
    or_set = new OR_commit_set;
    new_table = new New_object_table(Ntable_size);
    parts = new OR_set;

    new_surrs = new Surrogate_table;
    read_only = TRUE;
}

Commit_info::~Commit_info() {
    // The OR set and parts were deleted while waiting for the commit reply
    delete new_table;
    delete new_surrs;
}

Transaction* Commit_info::slow_get_transaction(OR_num orx, bool create) {
    // requires: No entry for "or" is present in this.
    // effects:  Same as get_transaction

    if (!create) return NULL;

    // Create a new transaction object
    OR_commit_data info;
    Transaction* trans = new Transaction;
    info.trans = trans;
    trans->ros =0; // Will be set by make transaction
    // For single-OR transactions, it is taking extra space for MOS objects 
    trans->mos = new Modset; // Not using predict for MOS currently
    trans->nos = new Modset;
    trans->participants = 0;
    //  trans->tid is uninitialized at this point.It is set in make_transaction
    or_set->store1(orx, info); // Add the transaction for this OR
    return trans;
}

void Commit_info::process_before_commit_reply() {
    // effects: Cleans up data structures that are not needed after commit reply
       
    // Deletes parts, or_set and all the transaction objects contained in it.
    UIntKey orx;

    MapGenerator<UIntKey, OR_commit_data> gen(*or_set);
    OR_commit_data or_data;
    while (gen.get(orx, or_data)) {
	Transaction* trans = or_data.trans;
	delete trans->mos;
	delete trans->nos;
	delete trans;
    }
    delete or_set;
    delete parts;
}

Timer lixao;


bool Commit_info::make_transaction(Trans_log *tlog) {
  bool need_to_send_to_or = FALSE;

  // Sort logs.
  tlog->sort_logs();
  
  // Iterate over the log of accesses to persistent objects
  Trans_log::Persistent_iter piter(tlog);
  OR_num orx;
  bool has_mods;
  Ros *ros;
  read_only = true;

  while (piter.get(orx, ros, has_mods)) {
    need_to_send_to_or = true;
    last_trans = slow_get_transaction(orx, TRUE);
    last_trans->ros = ros;
    
    if (has_mods) {
      read_only = false;
      Trans_log::Mod_iter miter(tlog, orx);
      Core obj;
      Fields copy;

      while (miter.get(obj, copy)) {
	// Recursively unswizzle obj and add the relevant objs to MOS/NOS
	Xref x;
        x.orx = orx; x.oref =  obj->oref();
	recursive_unswizzle(obj, copy, x, last_trans->mos);
	obj->get_fields()->unstamp_all();
      }
    }
  }

  // Find the coordinator for this transaction
  OR_num coord = get_coord_participants(parts);
  
  // Set the tid, coordinator and the participants fields of the transaction
  // objects for each OR. Sort the ROS to remove the duplicates also
  
  Tid tid = Tid(FE->address, Tstamp(TRUE)); // Timestamp the trans
  
  MapGenerator<UIntKey, OR_commit_data> gen(*or_set);
  UIntKey key;
  OR_commit_data or_data;
  while (gen.get(key, or_data)) {
    Transaction *trans = or_data.trans;
    trans->tid = tid;
    trans->coordinator = read_only ? 0 : coord;
    // Participants need not be informed about other participants
    trans->participants = ((Uint) key.val == coord) ? parts: &empty_or_set;
  }
  
  return need_to_send_to_or;
}

void Commit_info::commit_reply_processing(bool committed) {
    if (!committed) {
	FE->pc->abort_orefs();
	return;
    }
    FE->pc->commit_orefs();
    // Add the newly created surrogates to the global surrogate table
    Surrogate_table::Iter iter(new_surrs);

    Xref dest, surr;
    while (iter.get(dest, surr)) {
	FE->surr_table->insert(dest.orx, dest.oref, surr.orx, surr.oref);
    }
}

//
// Internal Methods
//

inline Oref Commit_info::get_oref(OR_num dest_or, Oref dest_oref, OR_num orx) {
    // Builtin types have null or these orefs cannot be assigned to
    // any class at an orXXX.
    if (dest_or == orx || dest_or == Null_or) return dest_oref;
    th_assert(0, "Surrogates do not exist currently!!!");
    return get_surr_oref(dest_or, dest_oref, orx);
}
    
void Commit_info::recursive_unswizzle(Core c, Fields copy, Xref xref, Modset *mset) {
    int num_slots = c->num_slots();
    Fields f = c->get_fields();
    Class_c cl(c->get_class());
    Xref class_xref = cl.class_xref();
    Oref class_oref = get_oref(class_xref.orx, class_xref.oref, xref.orx);

    Obj_handle obj_handle = mset->add_object(xref.oref, (OR_obj*)f,
					    num_slots, class_oref);
    // Unswizzle the referred to objects first and this object
    // Note: We cannot use the routine "unswizzle_object" because it requires
    //       that all the referred to objects be persistent objects

    // Iterate through all the slots of the object, unswizzle them
    // and increment reference counts for references created during the
    // transaction.
#if USE_REF_COUNTS
    bool need_to_adjust = (copy != 0) && FE->rot->need_adj_rc_mod(c, copy, true);
    Obj_x *copy_slots = (Obj_x *) copy->slots();
#endif

    Obj_x *slots = (Obj_x *) f->slots();
  
    Object_references gen(c->bitfields(), num_slots);
    int slotno;
    while (gen.get(slotno)) {
	if (slots[slotno].is_data()) continue;

	// Swizzled pointer needs to be unswizzled
	Core target = slots[slotno].pointer()->to_core();

#if USE_REF_COUNTS
	if (need_to_adjust) {
	  FE->rot->adj_rc_mod(target, copy_slots+slotno, true);
	}
#endif

	Xref target_xref;
	if (target->is_persistent()) {
	    target_xref = target->xref();
	} else {
	    // Volatile object
	    if (!new_table->find(target, target_xref)) {
		// Get a new xref and store entry for object's new xref
		// We place a new object at the OR of the first object
		// we find that references it.
		Oref target_oref = FE->pc->get_new_oref(xref.orx, target->num_slots());
		th_assert(target_oref != NULL_OREF, "pc did not grant new oref");
		target_xref.orx = xref.orx;
		target_xref.oref = target_oref;
		new_table->store1(target, target_xref);
		// Recursively unswizzle the new object
		recursive_unswizzle(target, 0, target_xref, last_trans->nos);
	    }
	    // New object is now in the NOS set and is unswizzled.
	}
	Oref target_oref = get_oref(target_xref.orx, target_xref.oref, xref.orx);
	mset->set_oref(obj_handle, slotno, target_oref);
    }
}

Oref Commit_info::get_surr_oref(OR_num dest_or, Oref dest_oref, OR_num orx) {
    th_assert(orx != dest_or, "Surrogate asked for same OR");

    Oref result = NULL_OREF;
    // Look up in the global surrogate table and then lookup in the
    // table for newly created surrogates
    result = FE->surr_table->lookup(dest_or, dest_oref, orx);
    if (result != NULL_OREF) return result;

    // Look up the newly created surrogates
    result = new_surrs->lookup(dest_or, dest_oref, orx);
    if (result != NULL_OREF) return result;

    // Create a surrogate for this object and enter into new_surrs
    int total_slots = Surrogate_size()/Slot_size;
    result = FE->pc->get_new_oref(orx, total_slots - OR_obj_headers);
    new_surrs->insert(dest_or, dest_oref, orx, result);

    // The surrogate has to be added to the NOS of "or" also
    last_trans->nos->add_surrogate(result, dest_or, dest_oref);
    return result;
}

OR_num Commit_info::get_coord_participants(OR_set* parts) {

    // Method of picking coordinator:
    // 1) Find OR which maximizes | MOS | + | ROS | for objects at the
    //    OR for this transaction.
    // 2) If no OR has any modified or new objects, 
    //    return Null_or (transaction is read only).

    OR_num max_write_or = Null_or;
    int    write_objs = 0;

    MapGenerator<UIntKey, OR_commit_data> gen(*or_set);
    UIntKey key;
    OR_commit_data or_data;
    while (gen.get(key, or_data)) {
	OR_num or_num = key.val;
	Transaction *trans = or_data.trans;
	parts->add(or_num);
	if (max_write_or == Null_or)
	    max_write_or = or_num;
	int msize = trans->mos->count();
	int count = trans->ros->size() + msize; 
	if (msize && count > write_objs) {
	    write_objs = count;
	    max_write_or = or_num;
	}
    }
    return max_write_or;
}
