#ifndef _FE_SEND_MESSAGE_H
#define _FE_SEND_MESSAGE_H

#include "utils/basic.h"
#include "utils/intarray.h"
#include "utils/send_message.h"
#include "common/fe_message.h"
#include "common/oref.h"

class Transaction;
class Network;

// This file has messages that are sent by the FE   AND
// require FE-sepcific data strutures to be used. The other messages sent by
// the FE are in common/fe_message.h


class FE_send_get_root_msg: public Send_message {
    // Overview: For getting OR root oref
  public:
    Ubits32 type() const {return FE_get_root_type;}
    bool encode(Network* net) {return net->flush();}
};

class FE_send_fetch_msg: public Send_message {
    // Overview: For sending a fetch message.
  public:
    FE_send_fetch_msg(Oref oref, Seg_bmap prefetch_bmap, bool alloc_request);
    // Create a message to fetch "oref" and prefetch pages of the
    // containing segment as indicated by "prefetch_bmap".
    // If "alloc_request" is true, allocation rights are requested for
    // the pages.

    Ubits32 type() const {return FE_fetch_type;}

    bool encode(Network* net);

  private:
    Oref oref;               // oref requested
    Seg_bmap prefetch_bmap;  // bitmap for pages to be prefetched
    bool alloc_request;      // whether allocation rights are requested
};

class FE_send_commit_msg: public Send_message {
    // Overview: Message used by the FE to send the commit requests

  public:
    FE_send_commit_msg(bool if_coord, Transaction *t):
	is_coord(if_coord), trans(t){}
    // effects: Creates an initialized commit message with transaction
    //          object t and marks it as coordinator iff is_coord is TRUE

    bool encode(Network* net);

    Define_message_type(FE_commit_type);

  private:
    Ubits32 is_coord; // If prepare request is for the coordinator or participant
    Transaction *trans; // The transaction object to be sent to the OR
};

class FE_send_invalid_ack_msg: public Send_message {
    // Overview: For sending an invalidation ack to the OR
  public:
    FE_send_invalid_ack_msg(Ubits32 last_msg_seen) :
	last_message_seen(last_msg_seen) {}
    // effects: Creates an initialized msg with last_msg_seen being the
    //          object that was last processed by the FE

    bool encode(Network* net);

    Define_message_type(FE_invalid_ack_type);
  private:
    Ubits32 last_message_seen;  // Last invalidation message seen by FE
};

class FE_send_alloc_msg: public Send_message {
    // Overview: FE sends this message to get allocation rights for objects
  public:
    FE_send_alloc_msg() {};
    // effects: Creates an initialized msg for getting allocation rights
    //          from an OR.

    bool encode(Network* net) { return net->flush();}

    Define_message_type(FE_alloc_type);
  private:
    // Currently there is nothing sent to the OR
};

class FE_send_debug_msg: public Send_message {
    // Overview: For sending a debug message.
  public:
    FE_send_debug_msg() {};

    FE_send_debug_msg(Ubits32 debug_msg_type, IntArray *data_arr):
	debug_type(debug_msg_type), data(*data_arr) {}
    // effects: Create a debug message with relevant "debug_msg_type".
    //          "data_arr" is the data for this message and is interpreted by
    //          the appropriate message type

    Define_message_type(FE_debug_type);
    bool encode(Network* net);

  private:
    Ubits32  debug_type;
    IntArray data;
};

class FE_send_trigger_msg: public Send_message {
    // Overview: Generic message from FE to trigger one of several 
    // activities at the OR. The activity is identified by an integer code.
  public:
    FE_send_trigger_msg (Ubits32 code_): code(code_) {}
    bool encode(Network* net) {return net->send_ubits32(code) && net->flush();}
    Define_message_type(FE_trigger_type);
  private:
    Ubits32 code; // idenitifies the activity to trigger
};

#endif /* _FE_SEND_MESSAGE_H */
