#ifndef _OR_INFO_H
#define _OR_INFO_H

#include "utils/bits.h"
#include "utils/map.h"
#include "utils/set.h"
#include "utils/valuekey.h"
#include "common/oref.h"
#include "common/page.h"
#include "common/or_num.h"

// This file maintains an abstraction for the information maintained at the FE
// per OR by the transaction manager

class IntArray;
class OR_info;
class Page_alloc_info;
class Alloc_set;
class OR_info_map;

// Info the PC needs about a single segment.
class Seg_info {
  public:
    Seg_bmap    cached_pages;  // Seg pages that are cached non-compacted
    Seg_bmap    alloc_rights;  // Seg pages to which this FE has alloc rights.
    Seg_bmap    empty;         // Seg pages which are empty at the OR
};

// Map from the segment number to information about a segment
typedef Map<UIntKey, Seg_info *> Seg_map;
typedef Set<UIntElem> Page_set; // For a set of pages in an OR

class OR_info {
  public:

    Ubits32 last_invalidation; // Last invalidation msg number received from OR
    Ubits32 last_inv_acked;    // Last invalidation message acked to OR

    OR_info(OR_num orx);
    // Creates an initialized object for OR or

    ~OR_info();

    OR_num orx() { return ornum; }
    // effects: Returns the OR number to which this infor belongs


    Seg_info* lookup_seg_info(Uint segid);
    // effects: Returns the seg_info object corresponding to segid in
    //          this or. If none exists, it creates one and returns the
    //          object. Caller should not free space

    void add_alloc_rights(Uint page_id, Page* p);
    // requires: p point to an intact page corresponding to page_id
    // effects: Registers p as a potential candidate for allocation rights if
    // necessary

    Oref get_new_oref(int num_slots);
    // Same specs as Persistent_cache::get_new_oref except that it does NOT
    // send any message to the OR.

  private:
    friend OR_info_map; // The map should only call methods as discussed below 

    // Representation
    OR_num ornum;                   // Name of this OR
    // The following are used by the PC calling code
    Page       *current_page;       // Page currently being used for allocation,
                                    // or NULL if none.
    Uint        current_page_id;    // Page_id of current_page.
    Seg_map    *segment_map;        // Map of segment information.
    Alloc_set  *alloc_set;          // Pages used for allocation during this
                                    // transaction for this OR

    Page_set *avail; // Pages available for allocating orefs
    // The ids are maintained here. If the available bytes are needed,
    // the page must be cached.

    // Private methods:

    Oref allocate(Page *p, Uint pid, Uint num_slots);
    // requires: or_info corresponds to the OR for page p
    //           pid is the pageid for p. Allocate rights are available on pid
    // effects: Allocates num_slots on the page p and returns the oref
    //          of the new object. If fails, returns NULL_OREF

    Page* get_uncached_page(Uint &pid);
    // modifies: pid
    // effects:  Get an uncached page in the or that is empty and
    //           place the pageid in pid.


    // The OR_info_map should only call the following methods: 

    void commit_alloc_set();
    // effects: Commits the orefs that have been allocated since the last
    //          call to commit/abort_alloc_set. Clears the alloc_set also
    //          Marks the non-empty pages as non-empty in the segment map

    void abort_alloc_set();
    // effects: Aborts the orefs that have been allocated since the last
    //          call to commit/abort_alloc_set. Clears the alloc_set also
    //          Resets the state of the pages where objects have been allocated

    bool allocatable_page(Page *p);
    // Returns TRUE if the page p has sufficient space to be considered
    // worthwhile for future allocations

};

class OR_info_map : private Map<UIntKey, OR_info*> {
    // Overview: A map that keeps track of information about every OR

  public:
    OR_info_map(); 
    ~OR_info_map();

    void add_or(OR_num orx);
    // requires: No entry exists for or in this map
    // effects: Adds an initialized or_info object for or


    // Following are to be used only by the PC

    void commit_orefs();
    void abort_orefs();
    // The specs for these are the same as Persistent_cache methods

    OR_info* lookup(OR_num orx);
    // checks:  An entry exists for or
    // effects: Returns an or_info object for OR or. Space should not be freed
    //          by caller

private:
    // Caches last value returned because we expect good locality.
    OR_info *last_returned;
};

inline OR_info * OR_info_map::lookup(OR_num orx) {
    if (orx == last_returned->orx()) return last_returned;
    return fetch(orx);
}

#endif /* _OR_INFO_H */
