//  CFile.java -- general procs for writing .c files, excluding method code

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;

class CFile {

static private StringBuffer strpool;	// string character pool
static private int strnums[];		// indices of used strings
static private int strlens[];		// lengths of those string
static private int strcount;		// number of used strings



//  write(d, c) -- write header info for class c on stream d.

static void write(PrintWriter d, ClassData c)
{

    // include general definitions
    d.println();
    d.println("#include \"" + c.fname + ".h\"");

    // Thor:
    d.println();
    d.println("struct Class_s *" + c.cname + "_V;");
    d.println("struct Class_s *" + c.cname + "_static_V;");

    d.println();
    d.println("struct " + c.cname + "_P_s *" + c.cname + "_P;");


    gfloats(d, c.constants);		// generate floating constants

    d.println();
    d.println("static Char ch_" + c.cname + "[];");	// declare string pool
    d.println("static Char *st_" + c.cname + "[];");	// declare string ptrs

    d.println();
    d.println("static java_lang_String strings_" + c.cname + "[];");

    // ZZZ: removed for now, Nov 25
    // clname(d, c);			// generate class name string


    // initialize bookkeeping for string pool
    strnums = new int[c.constants.length];	// init constant index array
    strlens = new int[c.constants.length];	// init string lengths array
    strpool = new StringBuffer(c.name);		// init pool with class name
    strlens[0] = c.name.length();		// record it as first entry
    strcount = 1;				// and count it

    // generate method code
    for (int i = 0; i < c.methods.length; i++) {
	Method m = new Method(c, c.methods[i]);
	if ((m.fl.access & ClassData.ACC_ABSTRACT) == 0) {
	    // generate body code if not native
	    if ((m.fl.access & ClassData.ACC_NATIVE) == 0) {
		try {
		    MethGen.mgen(d, m);
		} catch (Error e) {
		    Trans.abort("method " + m.fl.name + ": " + e.getMessage());
		}
	    }
	    // if synchronized, generate wrapper function
	    if ((m.fl.access & ClassData.ACC_SYNCHRONIZED) != 0)
		MethGen.syncwrap(d, m);
	}
    }

    // declare superclass list
    d.println();
    d.println("static struct Class_s * supers[" + c.nsupers + "];");

    // declare interfaces list
    if (c.ninters > 0) 
      d.println("static struct Class_s * inters[" + c.ninters + "];");
    else
      d.println("static struct Class_s * inters[1]; /* Actually 0 */");

    // declare others list
    if (c.nothers > 0) 
      d.println("static struct Class_s * others[" + c.nothers + "];");
    else
      d.println("static struct Class_s * others[1]; /* Actually 0 */");      

    cls_methods(d, c);	        // generate methods struct for runtime lookup
    cls_dv(d, c);		// generate the dispatch vector
    cls_static_dv(d, c);	// generate the static dispatch vector

    cls_init(d, c);		// generate code to initialize this class

    // dump string pool
    strdump(d, c);
}

// Thor: Generate methods for runtime lookup

static private void cls_methods(PrintWriter d, ClassData c)
{
  // ZZZ: Ensure: virtual_mtable is sufficient & instance_mtable is not needed
  Field[] mtable = c.virtual_mtable;

  d.println();  d.println();
  d.println("static struct Method_s " + c.cname + "_methods[] = {");

  for (int i = 0; i < mtable.length; i++) {
    Field m = mtable[i];
    if ((m.access & ClassData.ACC_ABSTRACT) != 0)
      d.println("\t{0, 0},");
    else {
      String s = String.valueOf(Names.hashinterface(m.name, m.signature));
      d.println("\t{" + s + ", (void(*)())" + m.cname + "},");
    }
  }
  d.println("};");
}

// Thor: Generate dv hdr

static private void cls_dv_hdr(PrintWriter d, ClassData c) 
{
    d.println("\t{");

    d.println("\t\t0, \t\t\t/* self_class */"); 

    d.print("\t\t"+"0x"+Long.toHexString(c.instance_bit_fields));
    d.println(", \t\t\t/* bf */"); 

    d.print("\t\t" + c.instance_num_slots);
    d.println(", \t\t\t/* num_slots */"); 

    d.println("\t\t" + c.virtual_mtable.length + ", \t\t\t/* num_methods */"); 
    d.println("\t\t&" + c.cname + "_methods[0], \t/* methods */"); 

    d.println("\t\t" + c.nsupers + ", \t\t\t/* nsupers */"); 
    d.println("\t\tsupers, ");

    d.println("\t\t" + c.ninters + ", \t\t\t/* ninters */"); 
    d.println("\t\t" + c.interfaces.length + ", \t\t\t/* ndinters */"); 
    d.println("\t\tinters, ");

    d.println("\t\t" + c.nothers + ", \t\t\t/* nothers */"); 
    d.println("\t\tothers, ");

    d.println("\t},");
}

// Thor: Generate dv

static private void cls_dv(PrintWriter d, ClassData c)
{
    d.println();
    d.print("struct " + c.cname + "_bidirectional_DV ");
    d.println("The_" + c.cname + "_DV = {");

    for (ClassData k=c; k!=null; k=k.superclass) {
      d.println("\n\t0, /* " + k.cname + "_P */");
      method_decls(d, k.instance_mtable, k.cname);
    }

    // ZZZ: Jul 3: Hack for now: java.lang.Object's constructor
    //d.println("\n\t0, /* java_lang_Object_P */");
    //d.println("\t(Void (*)(java_lang_Object)) init__AAyKx,");

    cls_dv_hdr(d, c);
    method_decls(d, c.virtual_mtable, c.cname);	

    d.println("};");
}

// Thor: Generate dv hdr

static private void cls_static_dv_hdr(PrintWriter d, ClassData c) 
{
    d.println("\t{");

    d.println("\t\t0, \t\t\t/* self_class */"); 

    d.print("\t\t"+"0x" + Long.toHexString(c.static_bit_fields));
    d.println(", \t\t\t/* bf */"); 

    d.print("\t\t" + c.static_num_slots);
    d.println(", \t\t\t/* num_slots */"); 

    d.print("\n\t\t");
    d.println("/* The following fields are not used for static objects */\n");

    d.println("\t\t0, \t\t\t/* num_methods */"); 
    d.println("\t\t0, \t\t\t/* methods */"); 
    d.println("\t\t0, \t\t\t/* nsupers */"); 
    d.println("\t\t0, \t\t\t/* supers */"); 
    d.println("\t\t0, \t\t\t/* ninters */"); 
    d.println("\t\t0, \t\t\t/* ndinters */"); 
    d.println("\t\t0, \t\t\t/* inters */"); 
    d.println("\t\t0, \t\t\t/* nothers */"); 
    d.println("\t\t0, \t\t\t/* others */"); 

    d.println("\t},");
}

// Thor: Generate static dv

static private void cls_static_dv(PrintWriter d, ClassData c)
{
    d.println();
    d.print("struct " + c.cname + "_static_bidirectional_DV ");
    d.println("The_" + c.cname + "_static_DV = {");

    d.println("\t0, /* " + c.cname + "_P */");
    cls_static_dv_hdr(d, c);
    method_decls(d, c.static_mtable, c.cname);
    d.println("};");
}

// Thor: generate the class initialization routine

static private void cls_init(PrintWriter d, ClassData c) {
  d.println("\n");
  d.println("void Init_" + c.cname + "()");
  d.println("{");

  d.println("\tstruct " + c.cname + "_bidirectional_DV *self_dv;");
  d.println("\tstruct " + c.cname + "_P_s *self_P;");
  d.println("\t" + c.cname + "_static static_obj;");

  d.println();
  d.println("\tint i,j;");
  d.println("\tChar *start, *end;");

  d.println();
  d.println("\tif (" + c.cname + "_V != 0) return;");
  d.println("\t" + c.cname + "_V = NEW_CLASS();");
  d.println("\t" + c.cname + "_static_V = NEW_CLASS();");
  d.println("\t" + c.cname + "_P = NEW(struct " + c.cname +"_P_s);");

  d.println();
  d.println("\tCREATE_DV(" + c.cname + ");");
  d.println("\tCREATE_DV(" + c.cname + "_static);");
  d.println("\tALLOC_OBJ_G(static_obj, " + c.cname +"_static);");
  d.println("\tPin_object((Obj_p) static_obj);");
  d.println("\t"+ c.cname + "_static_V->static_obj = (Obj_p) static_obj;");

  d.println();
  for (int i = 0; i < c.fields.length; i++) {
    Field f = c.fields[i];
    String obj_class = c.cname + "_static";
    String tgt_var = f.cname;

    if (c.static_num_slots > 31) {
      d.println();
      d.println("\tth_fail(\"Cannot construct big static object - " + c.cname + "\");");
    }

    if ((f.access & ClassData.ACC_STATIC) != 0) {
      if ((f.signature.charAt(0) == '[')  // Array type
	  || (f.signature.charAt(0) == 'L')) { // Object type

	// PUTP_D(static_obj, tgt_var, JAVA_NULL, obj_class);
	d.print("\tPUTP_D(static_obj, " + tgt_var + ", JAVA_NULL");
	d.println(", " + obj_class + ");"); 

      } else { // Basic type
	// PUTV_D(static_obj, tgt_var, 0, obj_class);
	// ZZZ: d.print("\tPUTV_D(static_obj, " + tgt_var + ", 0");
	// ZZZ: d.println(", " + obj_class + ");");
      }
    }
  }

  // +1 is for struct Fields_s inh in the beginning of each T_f_s
  d.println();
  d.println("\tth_assert((sizeof(struct " + c.cname + "_f_s) + Slot_size -1)" +
	    "/Slot_size == " + (c.instance_num_slots+1) + ","); 
  d.println("\t\t\"Mismatch in num_slots calculation\");");

  d.println("\tth_assert((sizeof(struct "+c.cname+"_static_f_s)+Slot_size-1)" +
	  "/Slot_size == " + (c.static_num_slots+1) + ","); 
  d.println("\t\t\"Mismatch in num_slots calculation\");");

  d.println();
  for (int i=1; i<c.nsupers; i++) {
    d.println("\tInit_" + (String) c.supers.elementAt(i) + "();");
  }
  for (int i=0; i<c.ninters; i++) {
    d.println("\tInit_" + (String) c.inters.elementAt(i) + "();");
  }
  for (int i=0; i<c.nothers; i++) {
    d.println("\tInit_" + (String) c.others.elementAt(i) + "();");
  }

  d.println();
  for (ClassData k=c; k!=null; k=k.superclass) {
    d.println("\tThe_"+ c.cname +"_DV." + k.cname + "_P = " + k.cname + "_P;");
  }
  // ZZZ: Jul 3: Hack for now: java.lang.Object's constructor
  //d.println("\tThe_"+ c.cname +"_DV.java_lang_Object_P = java_lang_Object_P;");

  d.println();
  d.println("\tThe_"+ c.cname +"_static_DV." + c.cname + "_P = " + c.cname + "_P;");

  d.println();
  for (int i=0; i<c.nsupers; i++) {
    d.println("\tsupers["+i+"] = " + (String) c.supers.elementAt(i) + "_V;");
  }

  for (int i=0; i<c.ninters; i++) {
    d.println("\tinters["+i+"] = " + (String) c.inters.elementAt(i) + "_V;");
  }

  for (int i=0; i<c.nothers; i++) {
    d.println("\tothers["+i+"] = " + (String) c.others.elementAt(i) + "_V;");
  }

  d.println();
  if ((c.access & ClassData.ACC_INTERFACE) != 0)
    d.println("\t" + c.cname + "_V->flags = IS_INTERFACE;");
  else if (c.is_persistable)
    d.println("\t" + c.cname + "_V->flags = (IS_CLASS | IS_PERSISTABLE);");
  else
    d.println("\t" + c.cname + "_V->flags = IS_CLASS;");

  d.println("\tCREATE_DV(" + c.cname + ");");

  d.println();
  d.println("\t" + c.cname + "_static_V->flags = IS_STATIC;");
  d.println("\tCREATE_DV(" + c.cname + "_static);");
  d.println("\tSET_DV(static_obj, " + c.cname + "_static_V->dv);");

  d.println();
  d.println("\tself_dv = & The_" + c.cname + "_DV;");
  d.println("\tself_P = self_dv->" + c.cname + "_P;");

  d.println();
  for (int i = 1; i < c.constants.length; i++) {
    Constant k = c.constants[i];
    if (k != null) {
      String type = null;
      if (k.tag == Constant.CLASS)
	type = Names.baseclass((String)k.value);
      else if (k.tag == Constant.INTERFACE)
	type = ((Ref) k.value).classname;
      else if (k.tag == Constant.FIELD) {
	String t = ((Ref) k.value).signature;
	if (t.charAt(0) == 'L') {
	  t = t.replace('/', '.');
	  type = t.substring(1, t.indexOf(';'));
	}
      }

      if (type != null) {
	type = Names.hashclass(type);
	d.println("\t" + c.cname + "_P->" + type + "_V = " + type + "_V;");
	d.println("\t"+c.cname+"_P->"+type +"_static_V = "+type+"_static_V;");
      }
    }
  }
  Enumeration e = c.arrays.keys();
  while (e.hasMoreElements()) {
    String type = (String) e.nextElement();
    int nDim = ((Integer) c.arrays.get(type)).intValue();
    type = Names.hashclass(type);
    // type = Repr.ctype_field_slots(type);
    for (int i=0; i<=nDim; i++) {
      d.print("\t"+c.cname+"_P->"+type+"_Array_"+i+"_V = ");
      if (i==0) {
	d.println(type + "_V;");
      } else if (i==1) {
	d.println("Instantiate_array(" + type + "_V);");
      } else {
	d.println("Instantiate_array("+c.cname+"_P->"+type+"_Array_"+(i-1)+"_V);");
      }
    }
  }

  d.println();
  d.println("\tstart = ch_" + c.cname + ";");
  d.println("\tfor (i=0; st_" + c.cname +"[i] != NULL; i++) {");
  d.println("\t\tjava_lang_String s;");
  d.println("\t\tstruct java_lang_String_f_s *s_f;");
  d.println("\t\tChar_Array c;");
  d.println("\t\tstruct Char_Array_f_s *c_f;");
  d.println("\t\tint n;");
  d.println("\t\tend = st_" + c.cname +"[i];");
  d.println("\t\tn = (end - start);");
  d.println("\t\tALLOC_ARRAY_V(c, Char, n);");
  d.println("\t\tc_f = FIX_FIELDS(c, Char_Array);");
  d.println("\t\tfor(j=0; j<n; j++) {");
  d.println("\t\t\tPUTAV_F(c_f, j, start[j], Char);");
  d.println("\t\t}");
  d.println("\t\tALLOC_OBJ(s, java_lang_String);");
  d.println("\t\tPin_object((Obj_p) s);");
  d.println("\t\ts_f = FIX_FIELDS(s, java_lang_String);");
  d.println("\t\tPREPARE_PWRITE_F(s_f);");
  d.println("\t\tPUTP_F(s_f, th_value, c, java_lang_String);");
  d.println("\t\tPUTV_F(s_f, th_offset, 0, java_lang_String);");
  d.println("\t\tPUTV_F(s_f, th_count, (end-start), java_lang_String);");
  d.println("\t\tstrings_" + c.cname + "[i] = s;");
  d.println("\t\tstart=end;");
  d.println("\t}");
  d.println("\t" + c.cname + "_V->strings = (Obj_p *)strings_" + c.cname + ";");  
  d.println("\t" + c.cname + "_static_V->strings = (Obj_p *)strings_" + c.cname + ";");  

  d.println();
  d.println("\t" + c.cname + "_V->name = " + c.cname + "_V->strings[0];");  
  d.println("\t" + c.cname + "_static_V->name = " + c.cname + "_V->strings[0];");  

  Field m;
  if ((m = c.getmethod("<clinit>", false)) != null) { 
    String obj_class = c.cname;
    String method_name = m.cname;

    d.println();
    d.print("\tDISPATCH_S(" + method_name + ", "+ obj_class + ")");
    d.println("(GET_STATIC_OBJ(" + obj_class + "_static));");
  }

  d.println("};");
}


//  methodref(d, c, name, climb) -- find "()V" method and write its name

static private void methodref
    (PrintWriter d, ClassData c, String name, boolean climb)
{
    Field m = c.getmethod(name, climb);
    if (m != null)
	d.println("\t" + m.cname + ",");
    else
	d.println("\t0,");			// not found
}

//  constructor(c, d) -- generate constructor or exception tosser reference

static private void constructor(PrintWriter d, ClassData c)
{
    if ((c.access & (ClassData.ACC_ABSTRACT | ClassData.ACC_INTERFACE)) != 0) {
	d.println("\tthrowInstantiationException,");
	return;
    }
    Field m = c.getmethod("<init>", false);
    if (m == null)
	d.println("\tthrowNoSuchMethodError,");
    else
	d.println("\t" + m.cname + ",");
}



//  method_decls(d, mtable) -- generate signatures & references for dv

static private void method_decls(PrintWriter d, Field[] mtable, String name)
{
    for (int i = 0; i < mtable.length; i++) {
	Field m = mtable[i];

	d.print("\t(" + Repr.retType(m) +" (*)(" + Repr.argTypes(m) + ")) ");

	if ((m.access & ClassData.ACC_ABSTRACT) != 0)
	    d.println("0,");
	else {
	    d.println(m.cname + ",");
	}
    };
}

//  gfloats(d, table) -- generated needed floating constants for class.

static void gfloats(PrintWriter d, Constant table[])
{
    d.println();
    for (int i = 1; i < table.length; i++) {
	Constant c = table[i];
	if (c == null || (c.tag != Constant.FLOAT && c.tag != Constant.DOUBLE))
	    continue;
	double v = ((Number)c.value).doubleValue();
	if (Repr.simplefloat(v))
	    continue;			// will generate in-line value
	if (c.tag == Constant.FLOAT)
	    d.println("static union fconst fc" + i + " = { 0x" +
		Integer.toHexString(Float.floatToIntBits((float)v)) + " };");
	else {
	    long b = Double.doubleToLongBits(v);
	    String lh = Integer.toHexString((int)(b >> 32));
	    String rh = Integer.toHexString((int)(b));
	    d.println("static union dconst dc" + i + " = { (ULong)0x" + lh +
		"<<32 | (ULong)(UInt)0x" + rh + " };");
	}
    }
}



//  strref(k) -- generate reference index for constant k, which is a string.

static int strref(Constant k)
{
    int n;

    n = strnums[k.index];		// get index into string pool
    if (n != 0)				// if already allocated
    	return n;			// return index
    
    // append to string pool and create a new slot
    n = strcount++;			// count entry
    strpool.append((String)k.value);	// append characters to pool
    strlens[n] = ((String)k.value).length();	// remember length
    strnums[k.index] = n;		// remember index
    return n;				//   and return it
}



//  strdump(d, c) -- dump string pool on stream d for class c.

static void strdump(PrintWriter d, ClassData c)
{
    String chname = "ch_" + c.cname;

    // generate string pool (a C array of Chars)
    d.println();
    d.println();
    d.println();
    d.print("static Char " + chname + "[] = {  /* string pool */");
    for (int i = 0; i < strpool.length(); i++) {
	if (i % Repr.GCHARS_PER_LINE == 0)
	    d.print("\n");    		/* break lines every so often */
	char ch = strpool.charAt(i);
	if (Repr.isQuotable(ch))
	    d.print("'" + ch + "',");	/* generate ASCII char if possible */
	else
	    d.print((int)ch + ",");	/* generate numeric value if not */
    }
    d.println("0};");			/* unneeded 0 makes generation simpler*/

    // generate array of pointers to ends of strings
    int base = 0;		/* starting point of current string */
    d.println();
    d.println("static Char *st_" + c.cname + "[] = {");
    for (int i = 0; i < strcount; i++) {
	/* generate pointer */
        d.print("\t" + chname + "+" + (base + strlens[i]) +
	    ",\t/* " + i + ". ");
	/* in comment, print first 40 chars of each constant */
	for (int j = 0; j < Repr.GCHARS_PER_CMMT && j < strlens[i]; j++) {
	    char ch = strpool.charAt(base + j);
	    if (ch >= ' ' && ch <= '~' && ch != '/' && ch != '?')
	    	d.print(ch);
	    else
	    	d.print("%");	/* use % as stand-in for troublesome chars */
	}
	d.println(" */");
	base += strlens[i];	/* incr base for next string */
    }
    d.println("\t0};");	/* terminate table with sentinel */

    base = 0;		/* starting point of current string */
    d.println();
    d.println("static java_lang_String strings_" + c.cname + "[] = {");
    d.println("\t/* To be initialized at boot time */");
    d.println();
    for (int i = 0; i < strcount; i++) {
      d.print("\t0,\t/* ");
      /* in comment, print first 40 chars of each constant */
      for (int j = 0; j < Repr.GCHARS_PER_CMMT && j < strlens[i]; j++) {
	char ch = strpool.charAt(base + j);
	if (ch >= ' ' && ch <= '~' && ch != '/' && ch != '?')
	  d.print(ch);
	else
	  d.print("%");	/* use % as stand-in for troublesome chars */
      }
      d.println(" */");
      base += strlens[i];	/* incr base for next string */
    }
    d.println();
    d.println("\t0};");	/* terminate table with sentinel */
}

} // class CFile
