//  Constant.java -- values from a Java classfile "constant" table

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.  


package toba;

import java.io.*;
import java.util.*;



public class Constant {		// constant table entry

    public int index;		// entry index
    public int tag;		// constant type
    public Object value;	// constant value



// tag types

public static final int CLASS		=  7;
public static final int FIELD		=  9;
public static final int METHOD		= 10;
public static final int INTERFACE	= 11;
public static final int STRING		=  8;
public static final int INTEGER		=  3;
public static final int FLOAT		=  4;
public static final int LONG		=  5;
public static final int DOUBLE		=  6;
public static final int NAMETYPE	= 12;
public static final int UTF8		=  1;
public static final int UNICODE		=  2;


//  load(d) -- read and crack constant table from stream d.

static Constant[] load (DataInputStream d)
    throws ClassFormatError, IOException
{
    // allocate constant table; first entry is unused
    Constant[] table = new Constant[d.readUnsignedShort()];
    table[0] = null;

    // read constant table entries
    for (int i = 1; i < table.length; i++) {
	Constant c = new Constant();
	table[i] = c;
	c.index = i;
	c.tag = d.readByte();
	switch (c.tag) {
	    case UTF8:
		c.value = d.readUTF();
		break;
	    case UNICODE:
		int len = d.readUnsignedShort();
		char s[] = new char[len];
		for (int j = 0; j < len; j++)
		    s[j] = (char)d.readUnsignedShort();
		c.value = new String(s);
		break;
	    case INTEGER:
		c.value = new Integer(d.readInt());
		break;
	    case FLOAT:
		c.value = new Float(d.readFloat());
		break;
	    case LONG:
		c.value = new Long(d.readLong());
		table[++i] = null;	// following entry unused
		break;
	    case DOUBLE:
		c.value = new Double(d.readDouble());
		table[++i] = null;	// following entry unused
		break;
	    case CLASS:
	    case STRING:
		c.value = new Integer(d.readUnsignedShort());
		break;
	    case FIELD:
	    case METHOD:
	    case INTERFACE:
	    case NAMETYPE:
		Pair p = new Pair();
		p.i1 = d.readUnsignedShort();
		p.i2 = d.readUnsignedShort();
		c.value = p;
		break;
	    default:
		throw new ClassFormatError(
		    "unrecognized constant tag " + c.tag);
	}
    }

    // first, replace string and class pointers with actual values
    for (int i = 1; i < table.length; i++) {
	Constant c = table[i];
	if (c != null) switch (c.tag) {
	    case STRING:
		c.value = table[((Integer)c.value).intValue()].value;
		break;
	    case CLASS:
		String s = (String)table[((Integer)c.value).intValue()].value;
    		c.value = s.replace('/', '.');	// fix path-style classname
		break;
	}
    }

    // then, fill in fields of reference entries
    for (int i = 1; i < table.length; i++) {
	Constant c = table[i];
	if (c != null) switch (c.tag) {
	    case FIELD:
	    case METHOD:
	    case INTERFACE:
		Ref r = new Ref();
		r.pair = (Pair)c.value;
		r.classname = (String)table[r.pair.i1].value;
		Pair p = (Pair)table[r.pair.i2].value;
		r.name = (String)table[p.i1].value;
		r.signature = (String)table[p.i2].value;
		c.value = r;
		break;
	}
    }

    return table;
}



//  dump(d, table) -- dump a constant table

static void dump(PrintWriter d, Constant table[])
{
    d.println();
    d.println("/*  Constant table:");
    for (int i = 1; i < table.length; i++) {
	Constant c = table[i];
	if (c == null)				// if unused entry
	    continue;
	d.print("    c" + i + ".  (" + c.tag + ")  ");
	switch (c.tag) {
	    case FIELD:
	    case METHOD:
	    case INTERFACE:
		Ref r = (Ref)c.value;
		d.println(r.signature + " " + r.classname + "." + r.name);
		break;
	    case NAMETYPE:
		Pair p = (Pair)c.value;
		d.println("c" + p.i1 + ", c" + p.i2);
		break;
	    default:
		d.println(c.value);
		break;
	}
    }
    d.println("*/");
    d.println();
}



} // class Constant
