//  HFile.java -- methods for writing .h files

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;

class HFile {

static private Hashtable declared;	// set of class structs declared

static private Hashtable ivtable;	// table of visible instance variables
static private int hvnum;		// hidden variable number



//  write(d, c) -- write header info for class c on stream d.

static void write(PrintWriter d, ClassData c)
{
    Field m, f;
    boolean hascv;

    // header file protection
    d.println();
    d.println("#ifndef " + c.cname + "_H");
    d.println("#define " + c.cname + "_H");

    declared = new Hashtable();		// clear list of classes

    // include general definitions, present class
    d.println();
    d.println("#include \"compiler/C/compiler.h\"");

    // Thor:
    d.println();
    d.println("DECLARE_type(" + c.cname + ");");
    d.println("DECLARE_type(" + c.cname + "_static);");

    d.println();
    d.println("struct " + c.cname + "_P_s;");
    // d.println("#define " + c.cname + "_static_P " + c.cname + "_P");

    d.println();
    d.println("#include \"compiler/toba.h\"");
    d.println("#include \"utils/th_assert.h\"");

    d.println();
    d.println("#include \"" + c.fname + ".externs.h\"");
    d.println("#include \"java/builtins/exports/java.h\"");

    // include ancestors classes and their interfaces
    d.println();
    declared.put(c.cname, "");
    for (ClassData t = c; t != null; t = t.superclass) {
	include(d, t.fname);
	for (int i = 0; i < t.interfaces.length; i++)
	    include(d, t.interfaces[i]);
    }
    
    // include declarations for other referenced classes
    for (ClassData k = c; k != null; k = k.superclass) {
      for (int i = 1; i < k.constants.length; i++) {
	Constant co = k.constants[i];
	if (co != null) {

	  String s = null;
	  if (co.tag == Constant.CLASS)
	    s = Names.baseclass((String)co.value);
	  else if (co.tag == Constant.INTERFACE)
	    s = ((Ref) co.value).classname;
	  else if (co.tag == Constant.FIELD) {
	    String type = ((Ref) co.value).signature;
	    if (type.charAt(0) == 'L') {
	      type = type.replace('/', '.');
	      s = type.substring(1, type.indexOf(';'));
	    }
	  }

	  if (s != null)
	    include(d, s);

	  if (co.tag == Constant.UTF8) {
	    String type = ((String)co.value);
	    try {
	      if (type.charAt(0) == '(') {
		
		String retType = type.substring(type.lastIndexOf(')') + 1);
		if (retType.charAt(0) == 'L')
		  include(d, Repr.ctype_actual(retType));

		String argType = type.substring(1, type.lastIndexOf(')'));
		while (! argType.equals("")) {
		  if (argType.charAt(0) == 'L')
		    include(d, Repr.ctype_actual(argType));
		  argType = Repr.nextSignature(argType);
		}
	      }
	    } catch (StringIndexOutOfBoundsException e) {}
	  }
	}
      }
    }

    // ZZZ: removed for now, Nov 25
    // define an "init_classname" macro, possibly having no effect if unneeded
    // d.println();
    // d.print("#define init_" + c.cname + "() ");
    // if (c.getmethod("<clinit>", false) == null)
    //	d.println("(void)0");
    // else // ZZZ: may have to fix initclass(&cl_ ...
    //	d.println("if (cl_" + c.cname + ".C.needinit) initclass(&cl_" + 
    //	    c.cname + ".C)"); 

    //Thor:
    d.println();
    d.println("#ifdef __cplusplus");
    d.println("extern \"C\" {");
    d.println("#endif");

    // Thor: function declarations: needed because some methods could be native
    //   and some methods could de defined in a super class
    d.println();
    for (int i = 0; i < c.methods.length; i++) {
	Field mt = c.methods[i];
	d.println("extern " + Repr.retType(mt.signature) + "\t" + mt.cname +
		  "(" + Repr.argTypes(mt) + ");");
    }

    //Thor:
    d.println();
    d.println("#ifdef __cplusplus");
    d.println("}");
    d.println("#endif");


    // Toba: instance struct
    // ...
    // d.println("    struct monitor *monitor;");


    // Thor: dvs

    d.println(); 
    d.println("struct " + c.cname +"_DV {");
    d.println("\n\tstruct DV_hdr_s hdr;\n");
    methodsigs(d, c.virtual_mtable, c.cname, "");
    d.println("};");

    d.println(); 
    d.println("struct " + c.cname +"_bidirectional_DV {");
    for (ClassData k=c; k!=null; k=k.superclass) {
      d.println("\n\tstruct " + k.cname + "_P_s *" + k.cname + "_P;");
      methodsigs(d, k.instance_mtable, k.cname, "_");
    }
    // ZZZ: Jul 3: Hack for now: java.lang.Object's constructor
    //d.println("\n\tstruct java_lang_Object_P_s *java_lang_Object_P;");
    //d.println("\tVoid (*init__AAyKx_)(java_lang_Object);");

    d.println("\n\tstruct DV_hdr_s hdr;\n");
    methodsigs(d, c.virtual_mtable, c.cname, "");	
    d.println("};");

    d.println(); 
    d.println("struct " + c.cname +"_static_DV {");
    d.println("\n\tstruct DV_hdr_s hdr;\n");
    methodsigs(d, c.static_mtable, c.cname, "");	
    d.println("};");

    d.println(); 
    d.println("struct " + c.cname +"_static_bidirectional_DV {");
    d.println("\tstruct " + c.cname + "_P_s *" + c.cname + "_P;");
    d.println("\n\tstruct DV_hdr_s hdr;\n");
    methodsigs(d, c.static_mtable, c.cname, "");	
    d.println("};");

    // Thor: T_P_s
    d.println(); 
    d.println("struct " + c.cname +"_P_s {");

    declared = new Hashtable();		// reset list of classes
    for (int i = 1; i < c.constants.length; i++) {
	Constant k = c.constants[i];
	if (k != null) {
	  String type = null;
	  if (k.tag == Constant.CLASS)
	    type = Names.baseclass((String)k.value);
	  else if (k.tag == Constant.INTERFACE)
	    type = ((Ref) k.value).classname;
	  else if (k.tag == Constant.FIELD) {
	    String t = ((Ref) k.value).signature;
	    if (t.charAt(0) == 'L') {
	      t = t.replace('/', '.');
	      type = t.substring(1, t.indexOf(';'));
	    }
	  }

	  if (type != null) {
	    type = Names.hashclass(type);
	    if (!declared.containsKey(type)) {
	      declared.put(type, "");
	      d.println("\tstruct Class_s *" + type + "_V;");
	      d.println("\tstruct Class_s *" + type + "_static_V;");
	    }
	  }
	}
    }
    Enumeration e = c.arrays.keys();
    // d.println("e = " + e);
    while (e.hasMoreElements()) {
      String type = (String) e.nextElement();
      int nDim = ((Integer) c.arrays.get(type)).intValue();
      // d.println("type = " + type + " nDim = " + nDim);
      type = Names.hashclass(type);
      // d.println("names.hashclass(type) = " + type);
      for (int i=0; i<=nDim; i++) {
	d.println("\tstruct Class_s *" + type + "_Array_" + i + "_V;");
      }
    }
    d.println("};");


    // Thor: object fields
    ivtable = new Hashtable();
    ivregister(c);			// make table of names
    d.println();
    d.println("struct " + c.cname + "_f_s {");
    d.println("\tstruct Fields_s inh;");
    // d.println("\tstruct monitor *monitor;");
    hvnum = 0;				// init counter of shadowed vars
    ivfield(d, c);			// declare instance variable fields
    d.println("};");

    // Thor: static fields
    d.println();
    d.println("struct " + c.cname + "_static_f_s {");
    d.println("\tstruct Fields_s inh;");
    for (int i = 0; i < c.fields.length; i++) {
      f = c.fields[i];
      if ((f.access & ClassData.ACC_STATIC) != 0)
	declare(d, f, f.cname);
    }
    d.println("};");

    //Thor: external variables
    d.println();
    d.println("extern struct Class_s *" + c.cname + "_V;");
    d.println("extern struct Class_s *" + c.cname + "_static_V;");
    d.println("extern struct " + c.cname + "_P_s *" + c.cname + "_P;");

    // tail
    d.println();
    d.println("#endif /* " + c.cname + "_H */");
}


//  include(d, name) -- generate #include for a class, if not already generated

static private void include(PrintWriter d, String name)
{
    name = Names.classfile(name);
    if (!declared.containsKey(name)) {
	d.println("#include \"" + name + ".h\"");
	declared.put(name, "");
    }
}


//  declare(d, v, name) -- generate a declaration for a variable.

static private void declare(PrintWriter d, Field v, String name)
{
    String prefix = "\t";
    if ((v.access & ClassData.ACC_VOLATILE) != 0)
	prefix = "\tvolatile ";
    d.println(prefix + Repr.ctype_field_slots(v.signature) + " " + name + ";");
}


//  ivregister(c) -- register instance variables in ivtable.
//
//  The order of recursion is such that only visible variables remain,
//  overwriting their shadowed ancestors in ivtable.

static private void ivregister(ClassData c)
{
    if (c == null)
	return;
    ivregister(c.superclass);
    for (int i = 0; i < c.fields.length; i++) {
	Field f = c.fields[i];
	if ((f.access & ClassData.ACC_STATIC) == 0)
	    ivtable.put(f.name, f);
    }
}



//  ivfield(d, c) -- declare instance variables for c and ancestors.

static private void ivfield(PrintWriter d, ClassData c)
{
    if (c == null)
	return;
    ivfield(d, c.superclass);
    for (int i = 0; i < c.fields.length; i++) {
	Field f = c.fields[i];
	if ((f.access & ClassData.ACC_STATIC) == 0) {
	  if (ivtable.get(f.name) == f) {
	    declare(d, f, f.cname);
	  } else {
	    declare(d, f, "th_" + Names.hashvar(++hvnum + "shadowed"));
	  }
	}
    }
}

//  methodsigs(d, mtable) -- generate method signatures for dispatch vector

static private void methodsigs(PrintWriter d, Field[] mtable, String name, String suffix)
{
    for (int i = 0; i < mtable.length; i++) {
	Field m = mtable[i];
	d.print("\t" + Repr.retType(m.signature));
	d.println(" (*" + m.cname + suffix + ")(" + Repr.argTypes(m) + ");");
    };
}



} // class HFile
