//  Supers.java -- operations involving superclasses

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;



public class Supers {



private static ClassData previous;	// last class loaded



//  load(k) -- load and link superclasses of ClassData k
//
//  sets k.superclass and k.mtable (and those of all ancestors)
//
//  It is assumed that previously seen classes can be reused,
//  allowing caching.

static void load(ClassData k)
    throws ClassNotFoundException, IOException
{
    ClassData kp;
    String sname;

    sname = k.supername;		// name of superclass

    if (k.superclass == null && sname != null) {

    	// there is a superclass and it has not been linked in

	for (kp = previous; kp != null; kp = kp.superclass)
	    if (kp.name.equals(sname))
		break;

	if (kp != null)
	    k.superclass = kp;		// found in existing chain
	else {
	    // not found; find and load superclass from file
	    k.superclass = new ClassData(ClassFile.find(sname), sname);
	    load(k.superclass);		// load superclass's superclasses
	}
    }

    methTable(k);	// fill in method table
    classes(k);         // fill in superclasses, interfaces & other classes
    fields(k);          // fill in instance & static _fields
    persistable(k);     //

    // ZZZ: Add these always
    String type = "Char";
    if (k.arrays.get(type) == null)
      k.arrays.put(type, new Integer(1));

    previous = k;	// remember class just loaded

}



//  methTable(class) -- scan methods and build method table.
//
//  Assumes that superclasses have been loaded.

static void methTable(ClassData k)
{
    int static_mtsize;
    int instance_mtsize;
    int virtual_mtsize;

    if (k.instance_mtable != null)
    	return;				// method table already initialized


    // virtual table includes methods from superclasses, other two tables dont

    static_mtsize = 0;
    instance_mtsize = 0;

    if (k.superclass == null) {
	virtual_mtsize = 0;
    }
    else {
	virtual_mtsize = k.superclass.virtual_mtable.length;
    }

    //  assign method table slots, checking for overrides of ancestors.
nextmethod:
    for (int i = 0; i < k.methods.length; i++) {
	Field f = k.methods[i];

	if ((f.access & ClassData.ACC_STATIC) != 0) {
	    static_mtsize++;
	    continue;
	}

	// this is an instance method
	instance_mtsize++;

	if ((f.access & ClassData.ACC_PRIVATE) != 0) {
	    continue;
	}

	if (f.name.equals("<init>")) {
	    continue;
	}

	// this is a virtual method
	String name = f.name;
	int gen = 0;
	for (ClassData c = k.superclass; c != null; c = c.superclass) {
	    gen++;
	    for (Field a = (Field)c.symtab.get(name); a != null; a = a.next) {
	      if ((a.access & ClassData.ACC_STATIC) != 0)
		continue;
	      if ((a.access & ClassData.ACC_PRIVATE) != 0)
		continue;
	      if (a.name.equals("<init>"))
		continue;
	      if (f.signature.equals(a.signature)) {
		// this method overrides an ancestor
		f.overrides = a.overrides + gen;
		f.mtslot = a.mtslot;
		continue nextmethod;
	      }
	    }
	}
	// method does not override an ancestor, so it gets a method table slot
	f.mtslot = virtual_mtsize++;
    }

    // allocate and initialize method table
    k.static_mtable = new Field[static_mtsize];
    k.instance_mtable = new Field[instance_mtsize];
    k.virtual_mtable = new Field[virtual_mtsize];

    if (k.superclass != null) {
        System.arraycopy(k.superclass.virtual_mtable, 0, k.virtual_mtable,
            0, k.superclass.virtual_mtable.length);
    }

    int static_i = 0;
    int instance_i = 0;

    for (int i = 0; i < k.methods.length; i++) {
	Field f = k.methods[i];

	if ((f.access & ClassData.ACC_STATIC) != 0) {
	    k.static_mtable[static_i++] = f;
	    continue;
	}

	k.instance_mtable[instance_i++] = f;

	if (f.mtslot >= 0)
	    k.virtual_mtable[f.mtslot] = f;
    }

}

//  classes(class) -- builds list of superclasses, interfaces and other classes
//
//  Assumes that superclasses have been loaded.

static void classes(ClassData k)
{ 

    if (k.supers != null) return;       // already initialized

    Hashtable declared;	                // set of class structs declared
    declared = new Hashtable();		// clear list of classes

    k.supers = new Vector();
    k.inters = new Vector();
    k.others = new Vector();

    k.nsupers = 0;
    for (ClassData c=k; c != null; c = c.superclass) {
      k.supers.addElement(c.cname);
      declared.put(c.cname, "");
      k.nsupers++;
    }

    k.ninters = 0;
    for (ClassData c=k; c != null; c = c.superclass) {
	for (int j = 0; j < c.interfaces.length; j++) {
	    String s = Names.hashclass(c.interfaces[j]);
	    if (! declared.containsKey(s)) {
	      k.inters.addElement(s);
	      declared.put(s, "");
	      k.ninters++;
	    }
	}
    }

    k.nothers = 0;
    for (int i = 1; i < k.constants.length; i++) {
	Constant c = k.constants[i];
	if (c != null) {
	  String s = null;
	  if (c.tag == Constant.CLASS)
	    s = Names.baseclass((String)c.value);
	  else if (c.tag == Constant.INTERFACE)
	    s = ((Ref) c.value).classname;
	  else if (c.tag == Constant.FIELD) {
	    String type = ((Ref) c.value).signature;
	    if (type.charAt(0) == 'L') {
	      type = type.replace('/', '.');
	      s = type.substring(1, type.indexOf(';'));
	    }
	  }

	    if (s != null) {
		s = Names.hashclass(s);
		if (! declared.containsKey(s)) {
		  k.others.addElement(s);
		  declared.put(s, "");
		  k.nothers++;
		}
	    }
	}
    }
}

//  fields(class) -- fill in instance & static _fields
//
//  Assumes that superclasses have been loaded.

static void fields(ClassData k)
{
  if (k.instance_fields != null) return;

  ClassData c;

  if ((c = k.superclass) != null) {
    k.instance_fields = (Vector) c.instance_fields.clone();
  } else {
    k.instance_fields = new Vector();
  }
    k.static_fields = new Vector();

  for (int i = 0; i < k.fields.length; i++) {  
    Field f = k.fields[i];
    if ((f.access & ClassData.ACC_STATIC) == 0) {
      k.instance_fields.addElement(f);
    } else {
      k.static_fields.addElement(f);
    }
  }
  computeBitFields(k, true, k.static_fields);
  computeBitFields(k, false, k.instance_fields);
}

public static final int sizeOfType(char sig) {
  switch (sig) {
    
  // ZZZ: Should make it system independent
  case 'B':           return 1; // Byte
  case 'C':           return 2; // Char
  case 'D': case 'd': return 8; // Double
  case 'F': case 'f': return 4; // Float
  case 'I': case 'i': return 4; // Int
  case 'J': case 'l': return 8; // Long
  case 'S':           return 2; // Short
  case 'Z':           return 1; // Boolean
  case 'L':           return 4; // Object
  case '[':           return 4; // Array

  default: throw new RuntimeException("Unknown field type\n");
  }
}

public static final void computeBitFields(ClassData c, boolean static_, Vector fields) {
  long bit_fields = 0;

  int slotSize = 4;
  int numSlots = 1; // struct Field_s inh
  int room_in_slot = 0; /* Can be 0, 1, 2, or 3 */

  for (int i=0; i<fields.size(); i++) {

    char sig = ((Field) fields.elementAt(i)).signature.charAt(0);
    boolean reference = ((sig == 'L') || (sig == '['));
    int field_size = sizeOfType(sig);

    switch (field_size) {

    case 8: 
      // linux/i386 doesn't want this rounding up XXX
      // if ((numSlots % 2) == 1) numSlots++; // Has to be word aligned
      /* Take over two new slots */
      numSlots += 2;
      room_in_slot = 0;
      break;

    case 4:
      /* Take over a new slot */
      if (reference) 
	bit_fields |= (1 << (numSlots-1));
      numSlots++;
      room_in_slot = 0;
      break; 

    case 2: 
      if (room_in_slot < 2) {
	/* Take over a new slot */
	numSlots++;
	room_in_slot = 2; 
      } else {
	/* Use the current slot */
	room_in_slot = 0;
      } 
      break;

    case 1: 
      if (room_in_slot < 1) {
	/* Take over a new slot */
	numSlots++;
	room_in_slot = 3; 
      } else {
	/* Use the current slot */
	room_in_slot--;
      } 
      break;

    }
  }
  numSlots--; // struct Field_s inh

  if (numSlots > 31) {
    //    Trans.abort("Sorry, " + c.cname + " has more than 31 fields:" + numSlots + ":");
    System.out.println("Warning: " + c.cname + " has " + numSlots + " > 31 fields");
  }

  if (static_) {
    c.static_bit_fields = bit_fields;
    c.static_num_slots = numSlots;
  } else {
    c.instance_bit_fields = bit_fields;
    c.instance_num_slots = numSlots;
  }
}


//  persistable(class) -- determines if this class is persistable
//
//  Assumes that superclasses have been loaded.

static void persistable(ClassData k) {

  String persistable_classes[] = {
    "java_lang_Object",
  };

  k.is_persistable = true;

  // If superclass is not persistable, this class is not persistable
  if ((k.superclass != null) && (!k.superclass.is_persistable)) {
    k.is_persistable = false;
  }

  // if has native methods, this class is not persistable
  for (int i = 0; i < k.methods.length; i++) {
    Field m = k.methods[i];
    if ((m.access & ClassData.ACC_NATIVE) != 0) {
      k.is_persistable = false;
    }
  }
  
  // override previous rules, if so specified
  for (int i=0; i<persistable_classes.length; i++) {
    if (persistable_classes[i].equals(k.cname)) {
      k.is_persistable = true;
    }
  }

}	

} // class Supers
