//  Trans.java -- Toba classfile translator
//
//  usage:  xtoba [-dxxxx] [-m mainclass] [-P dir] [-r] class...
//
//  Usually, the translator is called by the "toba" script.
//  If called directly, the "-dxxxx" command option sets debugging
//  flags.  Debugging output is written on standard output.
//  -m is used to specify a mainclass to generate
//  -r recursively pull in dependencies
//

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;

class Trans {



static final String PROGNAME = "Toba";
static final String usagestring =
    "usage: xtoba [-dxxxx] [-m mainclass] [-P dir] [-r] class...";

static final int cBufferSize = 8192;
static final int hBufferSize = 8192;
static final int dataBufferSize = 8192;
static final int externsBufferSize = 8192;

static final int mBufferSize = 8192;
static final int classBufferSize = 8192;
static final int peekBufferSize = 1024;

static String fileheader = "/*  created by " + PROGNAME + "  */";

static boolean echo_names;	// echo file names when compiling?



//  command line debugging options
static final char dbgAll = 'a';	// everything
static final char dbgGeneral = 'g';	// generally useful stuff

static final char dbgConstants = 'k';	// dump constant table of initial class
static final char dbgTrace = 't';	// trace major actions by translator
static final char dbgTemp = 'x';	// for temporary debugging hacks
static final char dbgCode = 'c';	// dump analyzed bytecode as a unit
static final char dbgInstrs = 'i';	// list instrs with generated code

private static boolean dbgflag[] = new boolean[256];

private static boolean autodep;		// pull in all dependencies
private static String mainclass;	// name of main class
private static String packagedir;	// destination package directory 

private static boolean needmain;	// need to generate main
private static Hashtable genlist;	// list of classes we generated
private static String tobapath;		// toba.class.path property value


//  main(args) -- main program.

public static void main(String args[]) 
{
    int i;

    options(args);			// process options

    tobapath = System.getProperty("toba.class.path");
    genlist = new Hashtable();

    // we need a main unless we're making a package
    if (packagedir == null)
        needmain = true;

    // locate main from user provided name
    if (mainclass != null) {
        doname(mainclass);
        if (needmain)
            abort("static void main(String[]) not found in " + mainclass);
    }

    // process all command line classes
    for (i = 0; i < args.length; i++) {
	if (args[i] != null)
	    doname(args[i]);		// process other arguments
    }
}



//  options(args[]) -- record options and null them in arglist.
//
//  very simple -- really need a "getopt" to do this right

private static void options(String args[])
{
    int i;
    String s;
    char c;
    int nopts = 0;

    for (i = 0; i < args.length; i++) {
	s = args[i];
	if (s.length() < 2 || s.charAt(0) != '-')
	    continue;
	nopts++;
	args[i] = null;
	c = s.charAt(1);
	switch (c) {
	    case 'd':
		setdbg(s.substring(2));
		break;
            case 'm':
                i++;
                if (i >= args.length)
                    abort(usagestring);
                mainclass = args[i];
                args[i] = null;
                break;
	    case 'P':
                i++;
                if (i >= args.length)
                    abort(usagestring);
                packagedir = args[i];
                args[i] = null;
                break;
            case 'r':
                autodep = true;
                echo_names = true;
                break;
	    default:
		abort(usagestring);
	}
    }

    ClassFile.trace = debugging(dbgTrace);

    if (args.length - nopts > 1)
	echo_names = true;
}



//  setdbg(s) -- set debugging options based on flag characters

static void setdbg(String s)
{
    int i;

    // set selected flags
    for (i = 0; i < s.length(); i++)
	dbgflag[(int)s.charAt(i)] = true;

    // any options at all imply general debugging
    dbgflag[(int)dbgGeneral] = true;

    // option 'a' implies all flags
    if (dbgflag[(int)dbgAll])
	for (i = 0; i < dbgflag.length; i++)
	    dbgflag[i] = true;
}



//  debugging(c) -- check if a particular debugging char is selected

static boolean debugging(char c)
{
    return dbgflag[(int)c];
}



//  doname(name) -- process one class, given by file or class name.

private static void doname(String name)
{
    // stop if we've already done this one
    // XXX can we do this here?  can differences between name and
    // k.name give us headaches?
    if (genlist.get(name) != null)
        return;

    ClassFile cf = null;
    ClassData k = null;

    try {				// load class
        cf = ClassFile.find(name);
	k = new ClassData(cf);
	Supers.load(k);			// load superclasses and method tables
    } catch (ClassNotFoundException e) {
        e.printStackTrace();
	abort("cannot load class " + name);
    } catch (FileNotFoundException e) {
	abort("cannot load file " + e.getMessage());
    } catch (IOException e) {
	abort("I/O error loading " + name + ": " + e);
    }

    // stop if we've already done this one
    if (genlist.get(k.name) != null)
        return;

    genlist.put(k.name, k.name);

    // try to generate a main method if none has been generated yet
    // ZZZ: Jul 18: Removed genmain for now.
    /* 
    if (needmain)
        genmain(k);
    */
    // don't generate C if its from the toba path - we generated previously
    //if (cf.path != null && cf.path.equals(tobapath))
    //    return;

    if (echo_names)
	System.err.println(name + ":");

    // copy the class to the package directory
    if (packagedir != null) {
        try {
            ClassInstall.install(name, k.name, packagedir);
        } catch(IOException e) {
            abort("Could not copy " + k.name + " to " + packagedir);
        } catch(ClassNotFoundException e) {
            abort(name + " disappeared!");
        }
    }

    PrintWriter cstream = oopen(k, ".c", cBufferSize);
    if (debugging(dbgConstants))
	Constant.dump(cstream, k.constants);
    CFile.write(cstream, k);		// write .c file
    cstream.close();

    PrintWriter hstream = oopen(k, ".h", hBufferSize);
    HFile.write(hstream, k);		// write .h file
    hstream.close();

    PrintWriter datastream = oopen(k, ".data", dataBufferSize);
    DataFile.write(datastream, k);		// write .data file
    datastream.close();

    PrintWriter externs_stream = oopen(k, ".externs.h", externsBufferSize);
    ExternsFile.write(externs_stream, k);		// write .data file
    externs_stream.close();

    /*
    boolean hasNatives = false;
    for (int i = 0; i < k.methods.length; i++) {
        Field f = k.methods[i];
	if ((f.access & ClassData.ACC_NATIVE) != 0) {
	  hasNatives = true;
	}
    }
    if (hasNatives) {
      PrintWriter nstream = oopen(k, "_native.c", cBufferSize);
      NativesFile.write(nstream, k);		// write .c file
      nstream.close();
    }
    */

    // recursively generate for all dependencies
    if (autodep) {
        for (int i = 1; i < k.constants.length; i++) {
            Constant c = k.constants[i];
            if (c != null) {
	      String depclass = null;
	      if (c.tag == Constant.CLASS)
		depclass = Names.baseclass((String)c.value);
	      else if (c.tag == Constant.INTERFACE)
		depclass = ((Ref) c.value).classname;
	      else if (c.tag == Constant.FIELD) {
		String type = ((Ref) c.value).signature;
		if (type.charAt(0) == 'L') {
		  type = type.replace('/', '.');
		  depclass = type.substring(1, type.indexOf(';'));
		}
	      }

	      if (depclass != null)
		doname(depclass);

	      if (c.tag == Constant.UTF8) {
		String type = (String) c.value;
		try {
		  if (type.charAt(0) == '(') {
		
		    String retType = type.substring(type.lastIndexOf(')') + 1);
		    if (retType.charAt(0) == 'L') {
		      String s = retType.replace('/', '.');
		      doname(s.substring(1, s.indexOf(';')));
		    }
		    
		    String argType = type.substring(1, type.lastIndexOf(')'));
		    while (! argType.equals("")) {
		      if (argType.charAt(0) == 'L') {
			String s = argType.replace('/', '.');
			doname(s.substring(1, s.indexOf(';')));
		      }
		      argType = Repr.nextSignature(argType);
		    }
		  }
		} catch (StringIndexOutOfBoundsException e) {}
	      }
            }
        }
    }
}



//  oopen(class, suffix, bufsize) -- open output file for generated code.

static PrintWriter oopen(ClassData k, String suffix, int bufsize)
{
    String filename = k.fname + suffix;
    File f = new File(filename);

    if (foreign(f))
	abort(filename + ": contents unrecognized; will not overwrite");

    if (f.exists() && !f.canWrite())
	abort(filename + ": cannot write");

    try {
	PrintWriter d = new PrintWriter(
	    new BufferedOutputStream(new FileOutputStream(f), bufsize));
	d.println("/*  " + f.getName() + " -- from Java class " 
	    + k.name + "  */");
	d.println(fileheader);
	return d;
    } catch(Exception e) {
	abort(filename + ": cannot open for output");
    }
    return null;
}


// genmain(class) -- try to generate a C main

static void genmain(ClassData k)
{
    Field mainmethod = null;

    // check if there is a public static void main(String[])
    for (int i = 0; i < k.methods.length; i++) {
        Field m = k.methods[i];
        if (m.name.equals("main") &&
            m.signature.equals("([Ljava/lang/String;)V") &&
            m.access == (ClassData.ACC_PUBLIC | ClassData.ACC_STATIC)) {
             mainmethod = m;
             break;
        }
    }  

    if (mainmethod == null)
        return;        

    PrintWriter mstream = oopen(k, "_main.c", mBufferSize);

    mstream.println("#include \"toba.h\"");
    mstream.println("#include \"" + k.fname + ".h\"");
    mstream.println("\n\n");
    mstream.println("\n");
    mstream.println("int main(int argc, char *argv[])\t\t/* main entry */");
    mstream.println("{");
    mstream.println("\treturn start(&cl_" + k.cname + ".C, " +
        mainmethod.cname + ", argc, argv);");
    mstream.println("}");     
    mstream.close();

    // we generated main, don't need one anymore
    needmain = false;
}


//  foreign(f) -- is file f a "foreign" file (not to be overwritten)?

private static boolean foreign(File f)
{
    if (!f.exists() || f.length() == 0)
      return false;
    try {
      // DataInputStream d = new DataInputStream(
      BufferedReader d = new BufferedReader(new InputStreamReader(
	    new BufferedInputStream(new FileInputStream(f), peekBufferSize)));
      d.readLine();
      String s = d.readLine();
      d.close();
      return !s.equals(fileheader);
    } catch(Exception e) {
      return false;
    }
}



//  abort(s) -- abort run, with a message.

static void abort(String s)
{
    try {
      throw new Exception(s);
    } catch (Exception e) {
      e.printStackTrace();
    }
    System.exit(1);
}



} // class Trans
