#include <iostream.h>
#include "utils/array.h"
#include "utils/bool_vector.h"
#include "utils/hashfuncs.h"
#include "common/oref.h"
#include "rwlock_unfair.h"
#include "cachedir.h"

// State of the pages in the segment. 2 bits per page
#define BITMAP_CLASS Page_states
#define BITMAP_SIZE 2 * Max_pages_in_seg
#include "utils/bitmap.h"

class Segment_map_info;
// Instantiate the hash table

#define IMPL_KEY IntKey
#define IMPL_VALUE Segment_map_info *
#include "utils/impl_map.t"

#define SET_ELEM FE_manager
#include "utils/impl_ptrset.t"
#define PTR_ELEM void
#include "utils/impl_ptrelem.t"
#include "utils/impl_ptrsets.t"

/* Rep of CacheDir:
   The segmap maps segments to array of FE_cache_state.
   Each FE_cache_state has a bitmap for the pages cached at that FE
   The absent state for a segment is represented by the absence of an entry
   for that FE
*/

struct FE_cache_state { // Information about a segment at an FE
    FE_manager* fe;          // The FE whose information this is for
    Page_states page_map; // State of the pages for this segment at FE "fe"
    // The state is captured using 2 bits.
};

declareArray(FE_cache_states, FE_cache_state)
implementArray(FE_cache_states, FE_cache_state)

// For each segment, we keep track of the following information.
struct Segment_map_info {
    FE_manager* fe_alloc;      //  FE to which this seg has alloc rights.
    FE_cache_states fe_states; // State of this segment at various FEs
};

static const Seg_bmap empty_bmap(FALSE); // Map used for returning all bits reset
static const Seg_bmap full_bmap(TRUE);   // Map used for returning all bits set


static inline FE_state get_page_state(Page_states page_map, int pagenum) {
    // effects: Returns the status of page "pagenum" in page_map

    th_assert((Uint)pagenum < Max_pages_in_seg, "Page number too big");
    // The state is coded in 2 bits at pagenum*2 + pagenum*2 + 1
    int bit1 = page_map.test(pagenum*2) == TRUE;
    int bit2 = page_map.test(pagenum*2+1) == TRUE;
    FE_state result = bit1 + bit2*2;

    // We could use the fact above that TRUE is 1 and FALSE is 0
    return result;
}

static inline void set_page_state(Page_states *page_map, Uint pagenum,
				  FE_state status) {
    // effects: Sets the status of page "pagenum" in page_map to be "status"

    th_assert(pagenum < Max_pages_in_seg, "Page number too big");
    // The state is coded in 2 bits at pagenum*2 + pagenum*2 + 1
    bool bit1 = status & 1;
    bool bit2 = (status & 2) >> 1;
    page_map->assign(pagenum * 2, bit1);
    page_map->assign(pagenum * 2 + 1, bit2);
}



CacheDir::CacheDir () {
    segmap = new Seg_map(10);
    lock = new RW_lock_unfair();
}

CacheDir::~CacheDir () {
    delete segmap;
    delete lock;
}

FE_state CacheDir::lookup(Uint pageid, FE_manager *fe) const {
    Segment_map_info *sinfo;
    Uint index;
    FE_state status = Page_absent;
    lock->read_lock();

    Uint segid = Segid_from_pageid(pageid);
    // Get the information about <segment, FE>
    bool found = get_seg_bmap(segid, fe, sinfo, index);
    if (found) {
	// Get the status of the page from sinfo
	Page_states page_map = sinfo->fe_states[index].page_map;
	Uint pagenum = Pagenum_from_pageid(pageid);
	status = get_page_state(page_map, pagenum);
    }
    lock->read_unlock(); 
    return status;
}
    
void CacheDir::create_fe_entry(Uint segid, FE_manager *fe,
			    Segment_map_info* &sinfo, Uint &index) {
    if (!sinfo) {
	// Segment entry does not exist. Allocate entry for the segment
	sinfo = new Segment_map_info;
	sinfo->fe_alloc = NULL;
	segmap->store1(segid, sinfo);
    }
    // Allocate the entry for the FE in sinfo
    FE_cache_state fe_state;
    fe_state.fe = fe;
    fe_state.page_map = Page_states(FALSE);
    index = sinfo->fe_states.size();
    sinfo->fe_states.append(fe_state);
}

void CacheDir::enter_single_page(Uint pageid, FE_manager *fe, FE_state status) {
    th_assert(status != Page_absent, "Bad status sent to enter_single_page");

    Segment_map_info *sinfo;
    Uint index = 0;
    Uint segid = Segid_from_pageid(pageid);
    lock->write_lock();

    bool found = get_seg_bmap(segid, fe, sinfo, index);
    if (!found)
	create_fe_entry(segid, fe, sinfo, index);
    // Now sinfo and index point to the right piece of information.
    Page_states *page_map = &sinfo->fe_states[index].page_map;
    set_page_state(page_map, Pagenum_from_pageid(pageid), status);
    lock->write_unlock();
}


FE_manager* CacheDir::alloc_rights(Uint pageid) const {
    Uint segid = Segid_from_pageid(pageid);
    Segment_map_info *sinfo;
    FE_manager *fe = NULL;

    lock->read_lock(); {
	if (segmap->find(segid, sinfo))
	    fe = sinfo->fe_alloc;
    } lock->read_unlock();
    return fe;
}

Seg_bmap CacheDir::enter(Uint segid, Seg_bmap bmap, FE_manager* fe,
			 FE_state status, bool alloc_if_possible) {
    Segment_map_info *sinfo;
    Uint index = 0;
    Seg_bmap result = empty_bmap;
    lock->write_lock();

    bool found = get_seg_bmap(segid, fe, sinfo, index);

    if (!found && status == Page_absent) {
	lock->write_unlock();
	return result;
    }

    // Note this code is not optimized for the Page_absent state

    if (!found)
	create_fe_entry(segid, fe, sinfo, index);
    // Now sinfo and index point to the right piece of information.

    bool all_absent = TRUE; // Whether all pages in segment are absent
    Page_states *page_map = &sinfo->fe_states[index].page_map;

    // Set the status of the desired pages to "status"
    for (int i = 0; i < bmap.size(); i++) {
	if (bmap.test(i))
	    set_page_state(page_map, i, status);
	FE_state pstate = get_page_state(*page_map, i);
	all_absent = all_absent && pstate == Page_absent;
    }

    // See if we have to and can grant allocation rights to this fe
    // Note that rights may be granted even if this fe has all pages absent
    if (alloc_if_possible && (sinfo->fe_alloc == NULL
			      || sinfo->fe_alloc == fe)) {
	sinfo->fe_alloc = fe;
	result = full_bmap;
    }

    if (all_absent) {
	// FE has no pages from segid. Remove its entry from sinfo
	remove_fe_entry(segid, sinfo, index);
    }
    lock->write_unlock();
    return result;
}

void CacheDir::alter(Uint pageid, FE_state old, FE_state now) {
    lock->write_lock();

    Segment_map_info *sinfo;
    Uint index;
    FE_state status = Page_absent;
    Uint segid = Segid_from_pageid(pageid);
    // Get the information about <segment, FE>. 0 is a dummy fe index
    bool found = get_seg_bmap(segid, 0, sinfo, index);
    if (!found && !sinfo) { lock->write_unlock(); return; }

    Uint pagenum = Pagenum_from_pageid(pageid);
    // For all FEs, where state is old, set it to now
    for (int i = 0; i < sinfo->fe_states.size(); i++) {
	FE_cache_state *fe_state = &sinfo->fe_states.slot(i);
	status = get_page_state(fe_state->page_map, pagenum);
	if (status == old)
	    set_page_state(&fe_state->page_map, pagenum, now);
    }
    lock->write_unlock(); 
}


void CacheDir::remove_fe(FE_manager *fe) {
    lock->write_lock();
    MapGenerator<IntKey, Segment_map_info*> sgen(*segmap);
    /// For each segment, look for the FE entry and remove it
    IntKey segkey;
    Segment_map_info *sinfo;
    Segment_map_info **to_delete = NULL;
    int *to_remove = NULL;
    int num_to_delete = 0;

    while (sgen.get(segkey, sinfo)) {
	Uint segid = (Uint) segkey.val;
	// Withdraw the allocation rights
	if (sinfo->fe_alloc == fe) sinfo->fe_alloc = NULL;
	FE_cache_states *fe_states = &sinfo->fe_states;

	for (int i = 0; i < fe_states->size(); i++) {
	    FE_cache_state *fe_state = &fe_states->slot(i);
	    if (fe_state->fe == fe) {
	      // There should not be any more entries for fe in fe_states
	      // So we should break now (since the size of array is altered)
	      sinfo->fe_states[i] = sinfo->fe_states.high();
	      sinfo->fe_states.remove();
	      if (sinfo->fe_states.size() == 0 && sinfo->fe_alloc == NULL) {
		// fe was the only FE_manager using this segment, so
		// we should remove the entry for the segment from the
		// segmap and mark the sinfo for deletion
		++num_to_delete;
		to_delete = (Segment_map_info **) realloc(to_delete,
				    sizeof(Segment_map_info)*num_to_delete);
		to_remove = (int *) realloc(to_remove,
					    sizeof(int)*num_to_delete);
		to_delete[num_to_delete - 1] = sinfo;
		to_remove[num_to_delete - 1] = segid;
	      }
	      break;
	    }
	}  // end of loop through fe_states
    }

    // Now that we have finished going through the generator, we can
    // delete the sinfo's that we found that were only matched to the
    // fe in question and remove the segid from the segmap.
    for (int i = 0; i < num_to_delete; i++) {
      segmap->remove_fast(to_remove[i]);
      delete to_delete[i];
    }    
    
    lock->write_unlock();
}

bool CacheDir::search(Uint pageid, FE_state status, FE_set *avoid_fes,
		      FE_manager* &fe) const {
    Segment_map_info *sinfo;
    Uint index;
    Uint pagenum = Pagenum_from_pageid(pageid);
    Uint segid = Segid_from_pageid(pageid);

    lock->read_lock();

    // Get the information about <segment, FE>. 0 is a dummy fe index
    bool found = get_seg_bmap(segid, 0, sinfo, index);
    if (!found && !sinfo) { lock->write_unlock(); return FALSE;}

    FE_cache_states *fe_states = &sinfo->fe_states;
    for (int i = 0; i < fe_states->size(); i++) {
	FE_cache_state *fe_state = &fe_states->slot(i);
	FE_state pstate = get_page_state(fe_state->page_map, pagenum);
	if (pstate == status && !avoid_fes->contains(fe_state->fe)) {
	    fe = fe_state->fe;
	    lock->read_unlock();
	    return TRUE;
	}
    }
    
    lock->read_unlock();
    return FALSE; // Used to be (status == Page_absent) XXX
}


void CacheDir::print(FILE *fp) {
    extern Address manager_to_address(FE_manager *fe);

    if (fp == NULL) fp = stderr;
    fprintf(fp, "Cache Directory:\n");

    MapGenerator<IntKey, Segment_map_info*> sgen(*segmap);
    IntKey segkey;
    Segment_map_info *sinfo;
    while (sgen.get(segkey, sinfo)) {
	Uint segid = (Uint) segkey.val;
	fprintf(fp, "Segment = %u, FE Alloc =", segid);
	Address fe_address = manager_to_address(sinfo->fe_alloc);
	fe_address.print(fp);
	fprintf(fp, "\n    ");
	FE_cache_states *states = &sinfo->fe_states;
	// Print the information for the segment (i.e. One FE at a time)
	for (int i = 0; i < states->size(); i++) {
	    FE_cache_state *state = &states->slot(i);
	    Address addr = manager_to_address(state->fe);
	    fprintf(fp, "FE: ");
	    addr.print(fp);
	    // Print the pagemap now for this FE
	    Page_states *page_map = &state->page_map;
	    for (int j = 0; j < page_map->size()/2; j++) {
		FE_state pstate = get_page_state(*page_map, j);
		switch(pstate) {
		    case Page_absent:      fprintf(fp, "A"); break;
		    case Page_unreparable: fprintf(fp, "U"); break;
		    case Page_reparable:   fprintf(fp, "R"); break;
		    case Page_complete:    fprintf(fp, "C"); break;
		}
	    }
	    fprintf(fp, "\n    ");
	}
    }
    fprintf(fp, "\n");
}

// Internal methods


bool CacheDir::get_seg_bmap(Uint segid, FE_manager *fe,
			    Segment_map_info* &sinfo, Uint &index) const {
    sinfo = NULL;
    if (!segmap->find(segid, sinfo)) return FALSE;
    
    FE_cache_states *fe_states = &sinfo->fe_states;
    // Search the segment info to see if FE has cached the segment/page
    for (int i = 0; i < fe_states->size(); i++) {
	FE_cache_state *fe_state = &fe_states->slot(i);
	if (fe_state->fe == fe) {index = i; return TRUE;}
    }
    return FALSE;
}

void CacheDir::remove_fe_entry(Uint segid, Segment_map_info *sinfo, Uint index) {
    sinfo->fe_states[index] = sinfo->fe_states.high();
    sinfo->fe_states.remove();
    if (sinfo->fe_states.size() == 0 && sinfo->fe_alloc == NULL) {
      // Remove the entry for the segment
      segmap->remove_fast(segid);
      delete sinfo;
    }
}


/*  Methods that may be required later:
FeMap const* CacheDir::where(int seg) const {
    FeMap *femap;
    if (! segmap->fetch(seg, femap)) return 0;
    return (femap);
}
*/
