#include <iostream.h>
#include "utils/network.h"
#include "common/oref.h"
#include "common/page.h"
#include "or.h"
#include "or_config.h"
#include "cachedir.h"
#include "thread.h"
#include "or_send_message.h"
#include "mm/mm.h"
#include "mm/segtable.h"
#include "mm/segment.h"
#include "fe_recv_message.h"

// The code here is for receiving the debug message and handling it.
// It has been moved from fe_recv_message.cc since it was getting quite big


static void print_segment(int segid) {
    // effects: Prints information about segment segid
    Segment *seg = orx->mm->fetch_segment(segid, FALSE);
    if (seg) { 
	seg->print();
	orx->mm->mutex->grab(); {
	    seg->unpin();
	} orx->mm->mutex->release();
    } else {
	fprintf(stderr, "Segment %u does not exist\n", segid);
    }
}

static void print_page(int pageid) {
    // effects: Prints the info about page "pageid" on stderr

    Uint segid = Segid_from_pageid((Uint) pageid);
    Uint pnum  = Pagenum_from_pageid(pageid);
    Segment *seg = orx->mm->fetch_segment(segid, FALSE);
    fprintf(stderr, "Page %d (%u, %u) --- ", pageid, segid, pnum);
    if (seg) {
	Page *page = seg->lookup_page(pnum);
	if (page) {
	    page->print(orx->config->debug_level > 1);
	} else
	      fprintf(stderr, " page not found\n");
	orx->mm->mutex->grab(); {
	    seg->unpin();
	} orx->mm->mutex->release();
    } else {
	fprintf(stderr, "segment not found\n");
    }
}

static void print_object(int obj_oref) {
    // effects: Prints the info about object "Obj_oref" on stderr

    Oref oref = (Uint) obj_oref;
    Uint segid = Oref_segment(oref);

    Segment *seg = orx->mm->fetch_segment(segid, FALSE);
    fprintf(stderr, "Object ");
    Oref_print(oref);
    fprintf(stderr, " --- ");
    if (seg) {
	Uint pnum  = Oref_pagenum(oref);
	Page *page = seg->lookup_page(pnum);
	if (page) {
	    Uint onum = Oref_onum(oref);
	    Slot* slot;
	    int num_slots = page->search(onum, slot);
	    if (num_slots < 0) {
		fprintf(stderr, "object not found\n");
		return;
	    }
	    ((OR_obj *) slot)->print(num_slots - OR_obj_headers);
	} else {
	    fprintf(stderr, "page not found\n");
	}
	orx->mm->mutex->grab(); {
	    seg->unpin();
	} orx->mm->mutex->release();
    } else {
	fprintf(stderr, "segment not found\n");
    }
}

bool FE_recv_debug_msg::decode(Network *net, Ubits32 msg_id) {
    if (!net->recv_ubits32(&debug_type)) return FALSE;
    Ubits32 num;
    if (!net->recv_ubits32(&num)) return FALSE;
    data._enlarge_by(num);
    net->recv_buffer(data.as_pointer(), num*sizeof(int));
    // Now depending on the debug type, call the appropriate debug routine

    fprintf(stderr, "Debug message: Type = %d received\n", debug_type);
    if (num) {
	fprintf(stderr, "Data = %d\n", data[0]);
    }

    Oref oref;
    switch (debug_type) {
	case FE_debug_seg_table:
	    orx->mm->mutex->grab(); {
		orx->mm->segtable->print();
	    } orx->mm->mutex->release(); 
	    break;
	case FE_debug_root:
	    oref = orx->mm->directory();
	    fprintf(stderr, "Root Oref = ");
	    Oref_print(oref);
	    fprintf(stderr, "\n");
	    break;
	case FE_debug_segment:
	    print_segment(data[0]);
	    break;
	case FE_debug_page:
	    print_page(data[0]);
	    break;
	case FE_debug_object:
	    print_object(data[0]);
	    break;
	case FE_debug_fe:
	    break;
	case FE_debug_cachedir:
	    orx->cachedir->print();
	    break;
	default:
	    fprintf(stderr, "Debug type = %d not supported yet\n", debug_type);
	    break;
    }
    OR_send_debug_reply_msg rmsg;
    Send_net_message(&rmsg, net, msg_id);
    return TRUE;
}
