// Copyright 1995 Barbara Liskov

/*
\section{Front-End Connection Management}

This module maintains the connection with a particular FE.  Each "FE_Manager"
runs as a separate thread.  The thread loops reading messages from the
TCP connection to the FE and replying to them.

\subsection{Todo}
\begin{itemize}
\item Rework interface so that the lifetime of the FE management
      is not tied to the lifetime of a TCP connection.
\item Concurrent access?
\item Handle protocol errors
\end{itemize}
*/

#ifndef _FE_MANAGER_H
#define _FE_MANAGER_H

#include "utils/basic.h"
#include "utils/Timer.h"
#include "thread.h"
#include "common/oref.h"
#include "common/tid.h"
#include "rwlock.h"
#include "prefetch_info.h"
#include "inv_set.h"
#include "cachedir.h"

#include "common/page.h"

class Samples;
class FE_table;
class Fetch_Stats;
class Network;
class MM_HandleList;
class Pref_set;
class Transaction;
class Mutex;
class RW_lock;
class Condition;
class fe_recv_fetch_msg;
class prefetch_hint;
union OR_slot;

// \subsection{FE Manager}

class FE_manager : public Thread {
  public:
    FE_manager(Network* net, Network *ir_net);
    // requires - net is a TCP connection to the FE.
    // modifies - fe_map
    // effects  - Create a thread that handles messages from the FE.
    //		  The thread shuts down the connection when appropriate. 

    ~FE_manager();
    // effects - Shutdown the FE_manager.

    virtual void main();
    // effects - Handle messages from connection.

    // Communication interface
    Address id() const;
    // effects - Returns the address of the FE we're connected to.
    Network*  net;
    Mutex*    net_wlock;  // to control concurrent writing to the net
    Network*  iread_net;  // Network used to ask FEs for iread avoidance

    // Transaction interface
    // \subsection{Transaction commit result} 
    // In a multiple-OR transaction, the FE manager waits on the 
    // condition variable until phase 1 of the commit is complete.  
    // When a thread learns the result of the transaction, it should 
    // grab the mutex, set "result" and signal the condition variable.

    Condition *cond_commit;
    Mutex     *mutex_commit;
    int       result;
    Invalid_set *invalid_objs;   // Set of objects that need to be invalidated 
    Collection_num  max_message; // Highest # invalidation msg sent to FE
    Collection_num  current_message;  // Highest # invalidation msg that
				      // has been generated for this FE
    int           tno;	       // Number of transactions received

    void send_invalidation_message_if_needed();
    // effects - Send any invalid object set information that has not
    //           been sent to this frontend yet


    // Concurrency
    // The following methods lock and unlock the shared information execpt net
    void read_lock()     {lock->read_lock();};
    void read_unlock()   {lock->read_unlock();};
    void write_lock()    {lock->write_lock();};
    void write_unlock()  {lock->write_unlock();};

    // Statistics Interface
    Timer  stat_timer;    // Timer to get times that are later accumulated in fetch_stats
    static Fetch_Stats*  fetch_stats; // remove "static" to get per-fe stats/*

  private:

    RW_lock     *lock;        // Lock on shared info, except the net

    friend fe_recv_fetch_msg;

};

void Handle_stat_message(Network* net, Ubits32 msg_id);
// effects  Responds with OR performance numbers.  See "or_message.h"
//	    for the format and contents of the message sent back to the FE.
//          msg_id is the id of the message received by the OR


//bool forward_fetch (int segnum, Address helper, Address requester, 
//		    int request_id, bool append_mods);
// Requires that caller has not already locked mm->mutex.
// Sends forward-fetch msg to "helper", identifying the "requester" FE
// (zero if OR is the requester), the "request_id", and the segment requested.
// If "append_mods", also sends any pending modifications to the segment.
// Returns whether successfully sent the msg to helper.

//void send_objects(Network *net_, MM_HandleList *list);
// requires caller grabbed net_wlock associated with net_
// effects  - Send objects in "list" into the net_.


class Fetch_Stats {
public:
    // Fetch statistics
    Fetch_Stats();
    // initializes the statistics

    int total;                // Total number of fetches made
    int mods;           // FE fetches served by sending only mods
    int cache;          // FE fetches served from OR cache
    int forward;        // Requests to forward fetches to other FEs
    int forward_mods;   // Requests to forward fetches; with mods
    int forward_denied; // Denials for forwarded fetch requests
    int disk;           // FE fetches served from the disk
    int unreparable;    // fetches not forwarded because unreparable
    // invariant:
    // total fetches = mods+cache+forward+forward_mods+disk-forward_denied
    float disk_time;    // Time spent waiting for disk while fetching
                        // sum of the disk waiting times of all clients
    float tot_fetch_time;  // Total time spent servicing fetches 
                           // sum for all clients

    void reset();
    // resets the statistics to  initial values
    void print();
    // prints on the stdout
};


#endif /* _FE_MANAGER_H */
