// Copyright 1995 Barbara Liskov

//
//  \section{FE Table implementation}
//

#include "common/oref_set.h"
#include "common/th_assert.h"
#include "fe_table.h"
#include <assert.h>

implementOpenHashSet(Seg_set, Seg, SEG_HASH, SEG_EQUAL)

implementOpenHashMap(Segmap, int, Oref_set *, SEG_HASH, SEG_EQUAL)

//  \subsection{General operations}

FE_table::FE_table(void)
{
    used_segs = new Seg_set;
    fine_segs = new Segmap;
}

FE_table::~FE_table()
{
    delete_fine_orefs();
    delete used_segs;
    delete fine_segs;
}

//  \subsection{Used segment list operations}

void FE_table::add_segment(int segid)
{
    Seg new_seg = Make_Seg(segid, coarse);

    if (used_segs->contains(new_seg))
	return;
    
    used_segs->insert(new_seg);
}

void FE_table::set_segment_list(int count, int segments[])
{
    int i;
    Seg new_seg;

    used_segs->clear();
    delete_fine_orefs();
    fine_segs->clear();
    for (i=0; i < count; i++)
	{
	    new_seg = Make_Seg(segments[i], coarse);
	    used_segs->insert(new_seg);
	}
}

void FE_table::reset_segment_list(void)
{
    used_segs->clear();
    delete_fine_orefs();
    fine_segs->clear();
}

bool FE_table::is_object_used(Oref x)
{
    Seg table_seg, temp_seg;
    Oref_set *objs;
    bool used, found;

    // First see if x's segment is in table
    temp_seg = Make_Seg(OREF_SEGMENT(x), coarse);
    if (!used_segs->retrieve(temp_seg, &table_seg))
	{
	    return FALSE;
	}
    
    // If segment is coarse grain, we're done
    if (Get_Granularity(table_seg) == coarse)
	{
	    return TRUE;
	}

    // Otherwise, see if object is in segment's list of used objects.
    // (By rep invariant, segment should be in fine_segs)
    found = fine_segs->fetch(table_seg, objs);
    assert(found);

    used = objs->contains(x);

    return used;
}

void FE_table::add_object(Oref x)
{
    Seg temp_seg, table_seg;
    Oref_set *objs;
    bool used, found;

    temp_seg = Make_Seg(OREF_SEGMENT(x), coarse);
    used = used_segs->retrieve(temp_seg, &table_seg);

    // If segment unused, insert at coarse grain
    if (!used)
	{
	    used_segs->replace(temp_seg);
	    return;
	}

    // If stored at coarse grain, do nothing
    if (Get_Granularity(table_seg) == coarse)
	{
	    return;
	}

    // Otherwise, add to list of objects in this segment
    found = fine_segs->fetch(table_seg, objs);
    assert(found);

    objs->add(x, TRUE);
}

//  \subsection{Garbage collection operations}

void FE_table::store_fine_grain(int segid)
{
    Seg new_seg = Make_Seg(segid, fine);

    // If already stored at fine grain, do nothing
    if (fine_segs->contains(segid))
	{
	    return;
	}

    // Otherwise, add empty mapping to used objects
    used_segs->replace(new_seg);   // Mark as fine grain in used segments
    Oref_set *refs = new Oref_set(100); // XXX How much to specify for size
    fine_segs->store(new_seg, refs);
}

void FE_table::store_coarse_grain(int segid)
{
    Oref_set *objs;
    bool found;
    Seg new_seg = Make_Seg(segid, coarse);

    used_segs->replace(new_seg);  // Mark as coarse-grained

    // Get rid of list of used objects if segment was fine-grained.
    found = fine_segs->fetch(new_seg, objs);
    if (found)
	delete objs;
    fine_segs->remove(new_seg);
}

bool FE_table::is_fine_grain(int segid)
{
    Seg temp_seg, table_seg;
    bool used;

    temp_seg = Make_Seg(segid, coarse);
    used = used_segs->retrieve(temp_seg, &table_seg);

    if (!used)
	return FALSE;
    
    return Get_Granularity(table_seg) == fine;
}

Oref_set *FE_table::fine_grain_elements(int segid) {
    return new Oref_set(*(fine_segs->fetch(segid)));
}

// \subsection{ Private functions}
// Delete all lists of used objects from all segments in fine_segs.
// This is used to free memory when the fine segment table is cleared.
// Requires that mutex is held by caller.
void FE_table::delete_fine_orefs(void)
{
    // Iterate over all fine-grained segments and free associated list
    // of used objects.

    Segmap::Bindings bindings(fine_segs);

    for (; bindings.ok(); bindings.next())
	delete bindings.val();
}
