#include <iostream.h>
#include <pthread.h>

#include "or/mm/segment.h"
#include "or/mm/segtable.h"
#include "or/mm/logindex.h"
#include "or/mm/logrecord.h"
#include "utils/intset.h"
#include "utils/fail.h"
#include "utils/th_assert.h"
#include "common/modset.h"
#include "common/objrefs.h"
#include "common/obj_bitfield.h"
#include "common/null_or_ref.h"

#include "collector.h"
#include "inoutlists.h"
#include "translists.h"
#include "deltalists.h"
#include "gc.h"
#include "partition.h"
#include "mod_list.h"
#include "class_map.h"
#include "deltalists.h"
#include "scanlist.h"

Collector::Collector() {
    mods_to_be_freed = new Itable_Mods;
    mods_to_be_xscanned = new Itable_Mods;
    scanlist = new Scanlist;
}

Collector::~Collector() {
    delete mods_to_be_xscanned;
    delete mods_to_be_freed;
    delete scanlist;
}

Collector::Stats::Stats()
{on = FALSE;}

void Collector::Stats::start() {
    on = TRUE;
    objs_xscanned = 0;
    objs_scanned = 0;
}

void Collector::Stats::print(ostream *out) {
    *out << "Time taken to xscan mods: " << gc_timer.elapsed() << endl;
    *out << "Number of modified objects xscanned: " << objs_xscanned << endl;
    *out << "Total time MM mutex held: " << stopped_timer.elapsed() << endl;
}

void Collector::new_gc_object(Oref oref) {
    gc_objs.append(oref);
}

inline void Collector::free_mods() {
    int num_mods = mods_to_be_freed->size();
    if (num_mods < 128) 
	return; // wait until there is a bunch
    orx->mm->mutex->grab();
//    printf("Collector::free_mods grabbed orx->mm->mutex\n"); //DEBUG
    for (int i = 0; i < num_mods; i++)
	mods_to_be_freed->slot(i)->unref();
    orx->mm->mutex->release();
//    printf("Collector::free_mods released orx->mm->mutex\n"); //DEBUG
    mods_to_be_freed->clear();    
}

void Collector::init() {
    root_oref = orx->mm->directory(); // this is the actual root
    hidden_root_oref = orx->mm->super->root; // this is the hidden root
    root_partition = gc->partition_map->from_oref(root_oref).pid;
    hidden_root_partition = gc->partition_map->from_oref(hidden_root_oref).pid;
}

void Collector::xscan_mods() {
    gc_timer.start();
    int size = mods_to_be_xscanned->size();
    for (int i = 0; i < size; i++) {
	Itable_Mod *mod = mods_to_be_xscanned->slot(i);
	scan_object(mod->object(), mod->oref(), FALSE); // only xscan.
	mods_to_be_freed->append(mod);
    }
    free_mods();
    mods_to_be_xscanned->clear();
    gc_timer.stop(); 
}

void Collector::snapshot_xscan_mods() {
    gc_timer.start();
    int size = mods_to_be_xscanned->size();
    for (int i = 0; i < size; i++) {
	Itable_Mod *mod = mods_to_be_xscanned->slot(i);
	scan_object(mod->object(), mod->oref(), FALSE); // only xscan.
	mods_to_be_freed->append(mod);
    }
    int num_mods = mods_to_be_freed->size();
    for (int i = 0; i < num_mods; i++)
	mods_to_be_freed->slot(i)->unref();
    mods_to_be_freed->clear();    
    mods_to_be_xscanned->clear();
    gc_timer.stop(); 
}

void Collector::main() {
    do {
	gc->mod_list->mutex->grab();
	while (gc->mod_list->size() == 0)
	    gc->mod_list->condition->wait();
	mods_to_be_xscanned = gc->mod_list->extract(mods_to_be_xscanned);
	gc->mod_list->mutex->release();
	xscan_mods();
    } while (TRUE);
}

void Collector::scan_object(OR_obj *obj, Oref oref, bool scan) {
    if (st.on) scan ? st.objs_scanned++ : st.objs_xscanned++;
    Uint source_p = scan ? partition : gc->partition_map->from_oref(oref).pid;
    Oref class_oref = obj->class_oref();
    Class_info class_info;
    bool found = gc->class_map->find(class_oref, class_info);
    if (!found) {
	char s[255];
	sprintf(s, "\nCollector::scan_object: no info, oref=%d, \
class_oref=%d\n", oref, class_oref);
	fail(s);
    }
    OR_slot *fields = obj->fields();
    if (class_info.variable_size)
	class_info.size = (fields[0].value32 + 2*Slot_size - 1) / Slot_size;
    if (OBJ_BF_ISVARIABLE(class_info.ref_bits)) // varying bitfield
	class_info.ref_bits = fields[1].value32 & 0xFFFFFFFC;
    int ref_index;
    Object_references ref_gen(class_info.ref_bits, class_info.size);
    while (ref_gen.get(ref_index)) {
	Oref suboref = fields[ref_index].get_oref();
	if (Oref_equal(suboref, NULL_OREF))
	    continue;
	Partition_info pi = gc->partition_map->from_oref(suboref);
	Uint target_p = pi.pid;
	if (source_p != target_p) { // interpartition ref
	    gc->deltalists->mutex.grab();
	    gc->deltalists->include(source_p, target_p, suboref);
	    gc->deltalists->mutex.release();
	} else if (scan) {          // intrapartition ref
	    Uint orank = Oref_screate(pi.rank, Oref_sindex(suboref));
	    scanlist->enter_ranked(suboref, orank);
	}
    }
}

implementArray(Segments, Segment *);

void Collector::collect_partition() {
    printf("-> Begin GC phase:\n");
    select_partition(reduced_inset);
    snapshot();
    enter_roots();
    scan();
    sweep();
    replace_outlist();
    printf("<- Replaced outlist (done GC phase).\n");
}

void Collector::select_partition(int policy, IntSet* mod_segids = 0) {
    const float alpha = 0.6;
    
    float max_common = INT_MIN, candidate;
    int max_common_ind = 0;
    IntArray ranks;
    Partition_map *pmap = gc->partition_map;

    switch(policy) {
    case round_robin:
	th_assert(!mod_segids, "round_robin takes no mod_segids");
	if (partition == pmap->max_partition()) {
	    partition = root_partition;
	} else {
	    partition++;
	}
	break;
	
    case reduced_inset:
	th_assert(!mod_segids, "reduced_inset takes no mod_segids");
	for (Uint i = 0; i < pmap->max_partition(); i++) {
	    candidate = -pmap->inset_change(i);
	    if (candidate > max_common) {
		max_common_ind = i;
		max_common = candidate;
	    }
	}
	partition = max_common_ind;
	break;

    case segment_commonality:
	th_assert(mod_segids, "segment_commonality requires mod_segids");
	max_common = -1;
	max_common_ind = 0;
	th_assert(mod_segids, "Collector::select_partition: segment"); 
	for (Uint i = 0; i < pmap->max_partition(); i++) {
	    candidate = pmap->commonality_index(i, *mod_segids);
	    if (candidate > max_common) {
		max_common_ind = i;
		max_common = candidate;
	    }
	}
	partition = max_common_ind;
	break;

    case weighted_inset_commonality:
	th_assert(mod_segids, 
		  "weighted_inset_commonality requires mod_segids");
	max_common = -1;
	max_common_ind = 0;
	th_assert(mod_segids, "Collector::select_partition: segment"); 
	for (Uint i = 0; i < pmap->max_partition(); i++) {
	    candidate = alpha*pmap->commonality_index(i, *mod_segids) -
		(1.0 - alpha)*pmap->inset_change(i);
	    printf("Collector::select_partition: candidate(%d)=%f\n",
		   i, candidate);
	    if (candidate > max_common) {
		max_common_ind = i;
		max_common = candidate;
	    }
	}
	partition = max_common_ind;
	break;

    default:
	th_fail("Collector::select_partition: Bad selection policy");
    }
    pmap->reset_inset_change(partition);
    printf("Chose partition %d of %d.\n", partition, //DEBUG
	   gc->partition_map->max_partition()); //DEBUG
}

void Collector::fetch_segs() {
    printf(">>> fetch_segs, partition=%d\n", partition); //DEBUG
    segids = gc->partition_map->to_segments(partition);
    segs.clear();
    for (int i = 0; i < segids.size(); i++) {
	Segment *segment = orx->mm->fetch_segment(segids[i], FALSE, FALSE);
	// don't grab orx->mm->mutex; it's already grabbed by snapshot
	// so that applications are stopped.
	if (!segment) //DEBUG
	    th_fail("Collector::fetch_segs: segment not found"); //DEBUG
	segs.append(segment);
    }
    printf("fetch_segs: segs.size()=%d\n", segs.size()); //DEBUG
    printf("<<< fetch_segs\n"); //DEBUG
}

//  void Collector::snapshot() {
//      orx->mm->mutex->grab(); 
//      fetch_segs();

//      // Apply pending mods to segments.
//      for (int i = 0; i < segs.size(); i++) {
//  	Segment *seg = segs[i];
//  	seg->unpin(); // unpin to avoid self-deadlock
//  	orx->mm->modify_segment(seg, TRUE); // remove mods from itable
//  	seg->pin_segment();
//      }

//      // Extract mods to be xscanned.
//      gc->mod_list->mutex.grab();
//      mods_to_be_xscanned = gc->mod_list->extract(mods_to_be_xscanned);
//      gc->mod_list->mutex.release();

//      // XXX Include application roots before releasing mutex.
//      scanlist->clear();
//      orx->mm->mutex->release();

//      snapshot_xscan_mods();
//      enter_roots();

//      // Delete out-deltalists of the partition.
//      Deltalists::Elements dels(gc->deltalists, partition, TRUE);
//      Uint target_p;
//      Deltalist *delta;
//      while ((delta = dels.get(target_p)) != 0) {
//  	delta->clear();
//      }
//  }

void Collector::snapshot() {
    printf(">>> snapshot, partition=%d\n", partition); //DEBUG
    stopped_timer.start();  
    orx->mm->mutex->grab(); // halt applications
//    printf("Collector::snapshot grabbed orx->mm->mutex\n"); //DEBUG
    // (1) record the sequence number of the last log record so far.
    fetch_segs();
    Log_Index last_log_no = orx->log->high();
    // (2) inter-scan all objects in prepare records up to last_log_no.
    for (Log_Index cur_ind = orx->log->low(); 
	 cur_ind < last_log_no; cur_ind++) {
	Log_Record *log_rec = orx->log->fetch(cur_ind);
#if REPLICATION
	if (log_rec->type() == Prepared_Log_Record_Type) {
#else
	if (1) {
#endif
	    // only need to xscan the MOS and NOS, right?
	    Modset *mod_objs = 
		((Prepared_Log_Record *)log_rec)->modified_objects(),
		*new_objs = ((Prepared_Log_Record *)log_rec)->new_objects();
	    OR_obj *obj = 0;
	    Oref oref = 0;
	    int num_slots = 0;
	    // iterate over modified objs
	    Modset::Elements mod_elems(mod_objs);
	    while (mod_elems.get(obj, oref, num_slots))
		scan_object(obj, oref, FALSE); // only xscan
	    // iterate over new objs
	    Modset::Elements new_elems(new_objs);
	    while (new_elems.get(obj, oref, num_slots))
		scan_object(obj, oref, FALSE); // only xscan      
	}
    }
    printf("Inter-scanned prepare records.\n"); //DEBUG
    // (3) inter-scan all objects in the MOB.
    gc->mod_list->mutex->grab();
//    printf("Collector::snapshot grabbed gc->mod_list->mutex\n"); //DEBUG
    mods_to_be_xscanned = gc->mod_list->extract(mods_to_be_xscanned);
    gc->mod_list->mutex->release();
//    printf("Collector::snapshot released gc->mod_list->mutex\n"); //DEBUG
    snapshot_xscan_mods();
    printf("Inter-scanned MOB objects.\n"); //DEBUG
    // (4) scan current partition's objects in prepare records up
    //     to last_log_no.
    for (Log_Index cur_ind = orx->log->low(); 
	 cur_ind < last_log_no; cur_ind++) {
	Log_Record *log_rec = orx->log->fetch(cur_ind);
#if REPLICATION
	if (log_rec->type() == Prepared_Log_Record_Type) {
#else
	if (1) {
#endif
	    IntSet mod_segs;
	    ((Data_Log_Record *)log_rec)->get_modified_segments(&mod_segs);
	    // get segment indices of the current partition
	    UintArray segids = gc->partition_map->to_segments(partition);
	    for (int i = 0; i < segids.size(); i++)
		if (mod_segs.contains(segids[i])) {
		    // the current log record modifies the current 
		    // partition
		    Modset *mod_objs = 
			((Prepared_Log_Record *)log_rec)->modified_objects(),
			*new_objs = 
			((Prepared_Log_Record *)log_rec)->new_objects();
		    OR_obj *obj = 0;
		    Oref oref = 0;
		    int num_slots = 0;
		    // iterate over modified objs
		    Modset::Elements mod_elems(mod_objs);
		    while (mod_elems.get(obj, oref, num_slots))
			scan_object(obj, oref, TRUE); // scan.
		    // iterate over new objs
		    Modset::Elements new_elems(new_objs);
		    while (new_elems.get(obj, oref, num_slots))
			scan_object(obj, oref, TRUE); // scan.
		    break; // no need to examine further segment indices
		}
	}
    }
    // (5) install P's objects in the MOB into pages.  
    // modify_segment installs _committed_ mods to segments
    for (int i = 0; i < segs.size(); i++) {
	Segment *seg = segs[i];
	seg->unpin(); // unpin to avoid self-deadlock
	orx->mm->modify_segment(seg, TRUE); // remove mods from itable
	seg->pin_segment();
    }
    printf("Collector::snapshot about to release orx->mm->mutex\n"); //DEBUG
    orx->mm->mutex->release(); // resume applications
//    printf("Collector::snapshot released orx->mm->mutex\n"); //DEBUG
    stopped_timer.stop();

    // Delete out-deltalists of the partition.
    gc->deltalists->mutex.grab();
//    printf("Collector::snapshot grabbed gc->deltalists->mutex\n"); //DEBUG
    Deltalists::Elements dels(gc->deltalists, partition, TRUE);
    Uint target_p;
    Deltalist *delta;
    while ((delta = dels.get(target_p)) != 0) {
	delta->clear();
    }
    gc->deltalists->mutex.release();
//    printf("Collector::snapshot released gc->deltalists->mutex\n"); //DEBUG    
    printf("<<< snapshot\n"); //DEBUG
}

void Collector::enter_roots() {
    scanlist->clear();
    // Include persistent roots in scanlist.
    if (partition == root_partition)
	scanlist->enter(root_oref);
    if (partition == hidden_root_partition)
	scanlist->enter(hidden_root_oref);
    // include gc objects in the scanlist.
    for (int i = 0; i < gc_objs.size(); i++) {
	Oref o = gc_objs[i];
	if (gc->partition_map->from_oref(o).pid == partition) {
	    printf("Entering oref=%d into scanlist.\n", o); //DEBUG
	    scanlist->enter(o);
	}
    }
  // XXXXXXX include application roots in scanlist

  // Include in-translists.
    Inoutlists::Elements inelems(gc->inoutlists, TRUE, partition);
    Oref translist_oref;
    Uint source_p;
    while ((translist_oref = inelems.get(source_p)) != NULL_OREF) {
	// Include orefs in the translist.
	Translists::Elements transelems(translist_oref);
	Oref inref;
	while ((inref = transelems.get()) != NULL_OREF) {
	    scanlist->enter(inref);
	}
    }

    // Include in-deltalists.
    gc->deltalists->mutex.grab();
//    printf("Collector::enter_roots grabbed gc->deltalists->mutex\n"); //DEBUG
    Deltalists::Elements dels(gc->deltalists, partition, FALSE);
    Deltalist *delta;
    while ((delta = dels.get(source_p)) != 0) {
	// Include orefs in the deltalist.
	IntSet::Elements delelems(delta);
	int inref;
	while (delelems.get(inref)) {
	    scanlist->enter(inref);
	}
    }
    gc->deltalists->mutex.release();
//    printf("Collector::enter_roots released gc->deltalists->mutex\n"); //DEBUG
}

void Collector::scan() {
    // scan objects in scanlist until empty.
    printf(">>> scan: partition=%d\n", partition); //DEBUG
    Oref oref;
    while ((oref = scanlist->remove()) != NULL_OREF) {
	if (st.on) st.objs_scanned++;
	Uint seg_rank = gc->partition_map->from_oref(oref).rank;
	OR_obj *obj = segs[seg_rank]->find(Oref_sindex(oref));
	// No need to lock mm->mutex above because segments 
	// are already pinned.
	if (!obj)
	    fail("ERROR: Collector::scan: tried to scan null object, oref=%d, seg_rank=%d", oref, seg_rank);
	else {
	    putchar((int)'.');
	    scan_object(obj, oref, TRUE); // fully scan.
	}
    }
    printf("<<< scan\n"); //DEBUG
}

void Collector::sweep() {
    // Sweep the segments in the partition.
    printf(">>> sweep: partition=%d\n", partition); //DEBUG
    int i;
    for (i = 0; i < segs.size(); i++) {
	Segment *seg = segs[i];
	Uint num_pages = seg->num_pages();
	for (Uint j = 0; j < num_pages; j++) {
	    Page *page = seg->lookup_page(j);
	    Uint start_mark = (i*Max_pages_in_seg + j)*Max_objs_in_page;
	    printf("Sweeping partition %d, segment %d, page %d:\n", 
		   partition, i, j); //DEBUG
//	    int dead = page->retain(&(scanlist->markmap), start_mark, 
	    page->retain(&(scanlist->markmap), start_mark, 
				    segids[i], j);
//	    if (dead) 
		seg->mark_dirty();
	}
    }

    // Write segments back, if necessary
    for (i = 0; i < segs.size(); i++) {
	orx->mm->mutex->grab();
//	printf("Collector::sweep grabbed orx->mm->mutex\n"); //DEBUG
	Segment *seg = segs[i];
	seg->unpin(); // avoid self deadlock
	orx->mm->modify_segment(seg, TRUE); // might as well update it
	seg->write();
	orx->mm->mutex->release();
//	printf("Collector::sweep released orx->mm->mutex\n"); //DEBUG
    }
    printf("<<< sweep\n"); //DEBUG
}

void Collector::replace_outlist() {
    // For each translist in outlist of partition
    // overwrite with the corresponding deltalist.
    printf(">>> replace_outlist\n"); //DEBUG
    Inoutlists::Elements outelems(gc->inoutlists, FALSE, partition);
    Uint target_p;
    while (outelems.get(target_p) != NULL_OREF) {
	gc->deltalists->mutex.grab();
//	printf("Collector::replace_outlist grabbed gc->deltalists->mutex\n"); //DEBUG
	int size_change
	    = gc->deltalists->merge(partition, target_p, TRUE, TRUE);
	gc->partition_map->update_inset_change(target_p, size_change);
	// replace inoutlist, kill the deltalist.
	gc->deltalists->mutex.release();
//	printf("Collector::replace_outlist released gc->deltalists->mutex\n"); //DEBUG
    }
    
    // For each remaining out-deltalist, merge deltalist.
    gc->deltalists->mutex.grab();
//    printf("Collector::replace_outlist grabbed gc->deltalists->mutex\n"); //DEBUG
    Deltalists::Elements dels(gc->deltalists, partition, TRUE);
    IntSet victims;
    while (dels.get(target_p) != (Deltalist *)0) {
	printf("Merging deltalist from %d to %d.\n", partition, target_p);
	gc->deltalists->merge(partition, target_p, TRUE, FALSE);
	// replace inoutlist, don't kill the deltalist.
	victims.insert((int)target_p);
    }
    // done iterating over gc->deltalists
    IntSet::Elements e = &victims;
    int t;
    while (e.get(t)) {
	printf("Deleting deltalist from %d to %d.\n",  //DEBUG
	       partition, (Uint)t); //DEBUG
	gc->deltalists->remove(partition, (Uint)t);
    }
    gc->deltalists->mutex.release();
//    printf("Collector::replace_outlist released gc->deltalists->mutex\n"); //DEBUG
    segs.clear();
    printf("<<< replace_outlist\n"); //DEBUG
}
