#ifndef _COLLECTOR_H
#define _COLLECTOR_H

#include "or/thread.h"
#include "common/orefs.h"
#include "common/or_obj.h"
#include "utils/array.h"
#include "utils/Timer.h"
#include "partition.h"

class Scanlist;
class Itable_Mods;
class Segment;
declareArray(Segments, Segment *)

class Collector: public Thread {
public:
    Collector();
    ~Collector();
    
    void main();
    // Code executed by the GC thread.
    
    void init();

    void collect_partition();
    // collects a partition.

    void select_partition(int policy, IntSet* segments = 0);
    // effects: selects a partition to collect, using "policy"
    // to choose.

    void fetch_segs();
    // Fetch and pin segments of the partition.

    void snapshot();
    // Atomically, applies pending mods to segments in the partition,
    // xscans all pending mods, and finds roots for the partition.

    void enter_roots();
    // Enters persistent and secondary roots for partition in scanlist.

    void scan();
    // Scans objects in scanlist until empty.

    void sweep();
    // Sweep segs, writes them back, and unpins them.
    
    void replace_outlist();
    // Updates translists in the outlist at the end of a collection.

    void new_gc_object(Oref oref);
    // called when a new GC object is created; the collector remembers it in
    // order to avoid collecting itself.

    Uint partition; // Partition being scanned; NULL_PARTITION if none.

    // Statistics
    class Stats {
    public:
        Stats();
	int objs_xscanned; // number of modified objects xscanned
	int objs_scanned; // number of modified objects scanned
	void print(ostream *out); // print it on out
	void start(); // start collecting stats
	bool on; // whether collecting stats
    } st;

    // partition selection policies
    static const int round_robin = 1,
	reduced_inset = 2,
	segment_commonality = 3,
	weighted_inset_commonality = 4;
    
private:
   
    // SCANNING OBJECTS
    // Scanning an obj for interpartition references is called "xscan".
    Itable_Mods *mods_to_be_xscanned;
    Itable_Mods *mods_to_be_freed; // Mods that are xscanned not freed

    void xscan_mods();
    // effects: Inter-scans "mods_to_be_xscanned".

    void snapshot_xscan_mods();
    // effects: inter-scans "mods_to_be_xscanned", but does not immediately 
    // free them

    void scan_object(OR_obj *obj, Oref oref, bool scan);
    // Scans "obj" for interpartition (external) references.
    // and puts them in deltalists.
    // If "scan", also scans for intrapartition refs
    // and enters them in the scanlist.

    void free_mods();
    // Frees mods_to_be_freed if there are enough of them.

    // partition-level collection
    Segments segs;  // Array of segments in the partition.
    UintArray segids;  // Array of segment ids in the partition.
    Scanlist *scanlist; // Objects in the partition to scan.
    Oref hidden_root_oref;      // hidden root object
    Uint hidden_root_partition; // its partition
    Oref root_oref;      // root object
    Uint root_partition; // its partition
    Orefs gc_objs; // gc objects (i.e. translists, inoutlists, etc.)

    // site-level collection
    Uint site_phase;
};
#endif /* _COLLECTOR_H */
