#include <iostream.h>
#include "utils/fail.h"
#include "or/or.h"
#include "or/mm/mm.h"

#include "translists.h"
#include "deltalists.h"
#include "inoutlists.h"
#include "gc_transaction.h"
#include "gc.h"

#define IMPL_KEY Deltalists::Pair
#define IMPL_VALUE Deltalist*
#include "utils/impl_map.t"

#define GENERATOR_ELEMENT Deltalists::Pair
#include "utils/impl_generator.t"

declareArray(Pairs, Deltalists::Pair)
implementArray(Pairs, Deltalists::Pair)

implementArray(Merge_infos, Merge_info)

Deltalist::Deltalist(int predicted) {
    predict(predicted);
}

Deltalists::Deltalists() {
    usage = 0;
    deltamap.predict(512);
}

void Deltalists::remove(Uint source_p, Uint target_p) {
    st.refs++;
    Pair p(source_p, target_p);
    Deltalist* delta;
    if (deltamap.remove(p, delta))
	delete delta;
}

void Deltalists::include(Uint source_p, Uint target_p, Oref oref) {
    st.refs++;
    Deltalist *delta;
    Pair pair(source_p, target_p);
    bool found = deltamap.find(pair, delta);
    if (! found) {
	delta = new Deltalist(16); // predicted size 16
	usage += delta->memory_usage();
	deltamap.store1(pair, delta);
    }
    if (! delta->contains(oref)) {
	usage -= delta->memory_usage();
	delta->insert(oref);
	usage += delta->memory_usage();
	if (usage > gc->deltalists_size_hi) 
	    resize();
    }
}

void Deltalists::resize() {
    if (st.on) {
	st.resize_timer.start();    
	Merge_info mi;
	mi.usage = 0;
	mi.refs = 0;
	mi.deltalists = 0;
	mi.fetches = -gc->translists->st.fetches;
	mi.disk_fetches = -gc->translists->st.disk_fetches;
	mi.target_partitions = 0;
	st.mis->append(mi);
    }

    while (usage > gc->deltalists_size_low) {
	// find the biggest deltalist and merge it
	int maxsize = 0;
	Pair maxpair;
	MapGenerator<Pair, Deltalist*> gen(deltamap);
	Pair pair;
	Deltalist *delta;
	while (gen.get(pair, delta)) {
	    int size = delta->memory_usage();
	    if (size > maxsize) {
		maxsize = size;
		maxpair = pair;
	    }
	}
	if (maxsize == 0) {
	    cerr << "Usage = " << usage << 
		" low = " << gc->deltalists_size_low << endl;
	    fail("OR: GC: no delta list to merge. ");
	}
	if (gc->merge_delta_together) {
	    // Merge all delta lists with same target partition as maxpair
	    Pairs same_target_pairs;
	    MapGenerator<Pair, Deltalist*> gen2(deltamap);
	    while (gen2.get(pair, delta)) {
		if (pair.b == maxpair.b)
		    same_target_pairs.append(pair);
	    }
	    if (st.on) st.mis->high().target_partitions++;
	    int num_pairs = same_target_pairs.size();
	    for (int i = 0; i < num_pairs; i++) {
		Pair partitions = same_target_pairs[i];
		merge(partitions.a, partitions.b, FALSE, TRUE);
	    }
	} else {
	    Pair partitions = maxpair;
	    merge(partitions.a, partitions.b, FALSE, TRUE);
	}
    }
    if (st.on) 	{
	st.resize_timer.stop();
	st.mis->high().fetches += gc->translists->st.fetches;
	st.mis->high().disk_fetches += gc->translists->st.disk_fetches;
    }
    if (gc->evict_translist_cache) {
	// XXX evict all cache for experimentation only
	int cache_size = orx->mm->cache_size();
	orx->mm->resize_cache(100*1024); // leave space for permanent residents
	orx->mm->resize_cache(cache_size); // re-expand
    }
}

int Deltalists::merge(Uint source_p, Uint target_p, bool replace,
		       bool kill_deltalist) {
    Pair pair(source_p, target_p);
    Deltalist *delta = 0;
    bool found = deltamap.contains(pair);
    if (found && kill_deltalist)
	delta = deltamap.remove_fast(pair);
    else if (found)
	deltamap.find(pair, delta);
    if (! found) {
	if (! replace) 
	    // no deltalist from source_p to target_p and we're not
	    // clobbering the translist, so no work to do
	    return 0;
	delta = new Deltalist(0);
	usage += delta->memory_usage();
    }
    int delta_usage = delta->memory_usage();
    usage -= delta_usage;
    st.mis->high().refs += delta->size();
    st.mis->high().usage += delta_usage;
    st.mis->high().deltalists++;
    int size_change = 
	gc->translists->merge_delta(source_p, target_p, delta, replace);
    if (kill_deltalist || !found)
	delete delta;
    if (kill_deltalist)
	deltamap.allowAutoResize();
    printf("committing gc_transaction\n"); //DEBUG
    gc->transaction->commit(); // commit transaction.
    return size_change;
}

Deltalists::Stats::Stats() {
    on = FALSE;
}

void Deltalists::Stats::start() {
    on = TRUE;
    refs = 0;
    mis = new Merge_infos;
///////////////////////////////////////////////////////////////////////////////
    Merge_info mi;
    mi.usage = 0;
    mi.refs = 0;
    mi.deltalists = 0;
    mi.fetches = -gc->translists->st.fetches;
    mi.disk_fetches = -gc->translists->st.disk_fetches;
    mi.target_partitions = 0;
    mis->append(mi);
///////////////////////////////////////////////////////////////////////////////
}

void Deltalists::Stats::print(ostream *out) {
    *out << "Number of interpartition references added: " << refs << endl;
    *out << "Time to resize deltalists: " << resize_timer.elapsed() << endl;

    int num_mergers = mis->size();
    *out << "Number of mergers: " << num_mergers << endl;
    float avg_refs = 0;
    float avg_usage = 0;
    float avg_deltalists = 0;
    float avg_clumping = 0;
    *out << "deltas	refs	usage	fetches	disk	clumping" << endl;
    for (int i = 0; i < mis->size(); i++) {
	int deltalists = mis->slot(i).deltalists;
	int refs = mis->slot(i).refs;
	int usage = mis->slot(i).usage;
	float clumping = 0.0;
	if (mis->slot(i).target_partitions != 0.0)
	    clumping = float(deltalists)/mis->slot(i).target_partitions;
	*out << deltalists << "	" 
	     << refs << "	" 
	     << usage << "	" 
	     << mis->slot(i).fetches << "	"
	     << mis->slot(i).disk_fetches << "	" 
	     << clumping << "	"
	     << endl;
	avg_deltalists += deltalists;
	avg_refs += refs;
	avg_usage += usage;
	avg_clumping += clumping;
    }
    if (0 == num_mergers) return;
    *out << "Avg deltalists merged: " << avg_deltalists/num_mergers << endl;
    *out << "Avg usage merged: " << avg_usage/num_mergers << endl;
    *out << "Avg refs merged: " << avg_refs/num_mergers << endl;
    *out << "Avg clumping: " << avg_clumping/num_mergers << endl;
}

Deltalists::Elements::Elements(Deltalists * deltalists,
    Uint partition_, bool source_):
    gen(deltalists->deltamap),
    partition(partition_),
    source(source_)
{}

Deltalist* Deltalists::Elements::get(Uint &other_partition) {
    Pair pair;
    Deltalist *delta;
    while (gen.get(pair, delta)) {
	if (source) {
	    if (pair.a == partition) {
		other_partition = pair.b;
		return delta;
	    }
	} else {
	    if (pair.b == partition) {
		other_partition = pair.a;
		return delta;
	    }
	}
    }
    return 0;
}
