#ifndef _DELTALISTS_H
#define _DELTALISTS_H

#include "or/thread.h"

#include "utils/timed_log.h"
#include "utils/map.h"
#include "utils/intset.h"

struct Merge_info {
    int deltalists; // number of delta lists merged
    int usage; // reduction in usage bytes as a result of merger
    int refs; // references in delta lists that were merged
    int fetches; // number of translist block fetches during merger 
    int disk_fetches; // number of fetches from disk during merger 
    int target_partitions; // number of target partitions
};

declareArray(Merge_infos, Merge_info);
    
typedef class Deltalist: public IntSet {
  public:
    // A Deltalist is a set of orefs added recently to a translist.
    Deltalist(int predicted);
    int memory_usage() {return tsize*sizeof(int);}
};

//class Merge_infos;

class Deltalists {
  public:
    // Deltalists manages the various "Deltalist"s.
    Deltalists();

    void include(Uint source_partition, Uint target_partition, Oref o);
    // Adds "o" to the appropriate deltalist.
    // Resize "this" to fit within prescribed size.

    int merge(Uint source_p, Uint target_p, bool replace,
	      bool kill_deltalist);
    // Merges the deltalist from source_p to target_p into the corr translist. 
    // If "replace", empties the translist before merging.
    // Also updates gc->inoutlists if the translist is created or emptied.
    // If "kill_deltalist", removes the deltalist.
    // Commits transaction when done.  Returns the size difference between
    // the translist before and the translist after.

    void remove(Uint source_p, Uint target_p);
    // Removes the deltalist from source_p and target_p from the data
    // structure and deletes it if it exists; else does nothing.

    Mutex mutex; // controls concurrent access to deltalists

    // Statistics
    class Stats {
      public:
        Stats();
	Merge_infos *mis; // information about each merger
	int refs; // number of inter-partition refs included so far
	Timer resize_timer; // time to merge lists
	void print(ostream *out); // print it on out
	void start(); // start collecting stats
	bool on; // whether collecting stats
    } st;
    
  public:
    struct Pair {
	// A Pair stores two unsigneds (source and target partition ids).
	public:
	Pair() { }
	Pair(unsigned a, unsigned b) {this->a = a; this->b = b;}
	void operator=(Pair const &p) { this->a = p.a; this->b = p.b; }
	bool operator==(Pair const &p) const { return ((this->a == p.a) &&
						       (this->b == p.b)); }
	int hash() const { return hash_ubits64(((Ubits64)b << 32) | a); }
	void to_string() { printf("Pair: [%d,%d]\n", a, b); }
	Ubits32 a, b;
    };
    Map<Pair, Deltalist *> deltamap; 
    // maps [source, target partitions] to a "Deltalist"
    int usage; // memory usage of "this" in bytes

    void resize();
    // If the memory usage of "this" is above "gc->deltalists_size_hi",
    // merge some Deltalists until the usage drops below "low".

  public:
    class Elements {
      public:
	Elements(Deltalists * deltalists,
		 Uint partition, bool source);
	// If "source" generates all deltalists whose source ptn is partition.
	// Otherwise, generates all deltalists whose target ptn is partition.
	
	Deltalist* get(Uint &other_partition);
	// If no matching deltalist remains, returns 0.
	// Otherwise, returns next deltalist 
	// and sets other_partition to the corr. target/source partition.

      private:
	MapGenerator<Pair, Deltalist*> gen;
	Uint partition; // generate deltalists for this partition
	bool source; // if this partition is source or target
    };
    friend Elements;
};

#endif /* _DELTALISTS_H */
