#ifndef _INOUTLISTS_H
#define _INOUTLISTS_H

#include <iostream.h>
#include "common/oref.h"
#include "common/orefs.h"
#include "utils/Timer.h"

class OR_obj;
union OR_slot;
class Allocator;
class MM_Handle;

class Inoutlists {
    // An Inoutlists maps a partition to its inlist and outlist.
    // Also, it maps a source and a target partition to their translist.
public:
    Inoutlists();

    Oref get_translist(unsigned source_p, unsigned target_p) const;
    // Returns the oref for translist from source_p to target_p,
    // NULL_OREF if none exists.

    void add_translist(unsigned source_p, unsigned target_p, Oref trans_oref);
    // Requires that the translist from source to target_p is not already
    // included in in/outlist.
    // Sets the translist from source_p to target_p to trans_oref.
    // Puts modified and new in/outlist blocks in gc->transaction.

    void remove_translist(unsigned source_p, unsigned target_p);
    // Removes the translist from source_p to target_p from in/outlist.
    // Puts modified blocks in gc->transaction.

    class Elements {
    public:
	Elements(Inoutlists const* inoutlists, 
		 bool of_inlist, unsigned partition);
	// If "of_inlist", generates elements of the inlist for the partition.
	// Otherwise, generates elements in the outlist of partition.

	Oref get(unsigned &partition);
	// Returns oref of the contained translist,
	// and sets "partition" to the associated (source/target) partition.
	// Sets partition to NULL_PARTITION and returns NULL_OREF if none left.

	void release();
	// Must be called at the end if "get" is not repeated until normal
	// termination (when "get" returns a NULL_OREF).

    private:
	Oref next_oref;
	unsigned next_partition;
	MM_Handle *handle;
	OR_slot *fields;
	int i; // slot in fields to consider next
    };
    friend Elements;

    // Statistics
    class Stats {
    public:
        Stats();
	int blocks; // number of inoutlist blocks
	Timer get_translist_timer; // time to get translist
	Timer set_translist_timer; // time to set translist
	void print(ostream *out); // print it on out
	void start(); // start collecting stats
	bool on; // whether collecting stats
    } st;

private:
    Orefs inlists;
    Orefs outlists;
    Allocator *allocator; // To allocate in/outlistblocks
    
    void add_inoutlist(Orefs &inoutlists, unsigned partition1,
		       unsigned partition2, Oref translist_oref);
    // Sets (inoutlists[partition1])[partition2] = translist_oref.

    void remove_inoutlist(Orefs &inoutlists, unsigned partition1,
			  unsigned partition2);
    // Removes(inoutlists[partition1])[partition2]. 
};


#endif /* _INOUTLISTS_H */
