#ifndef _PARTITION_H
#define _PARTITION_H

#include "or/thread.h"
#include "gc.h"
#include "allocator.h"

#include "utils/array.h"
#include "utils/intarray.h"
#include "utils/intset.h"
#include "common/oref.h"
// #include "common/wk_xref.h"

Uint const NULL_PARTITION = Uint(-1);

// A "Partition_map" maps segment_id -> <partition_id, rank>.
// Each partition ranks its segments 0 onwards.
// The map also maps partition_id -> set of segments in it.

struct Partition_info {
    Uint pid; // id of containing partition
    Uint rank; // rank of containing partition
};

struct Partition_attrib {
    UintArray segs; // segids of contained segments
    Uint partition_phase; // phase counter for site-level collection
    bool partition_flag; // true iff site-marks have been fully propagated
};

declareArray(Partition_infos, Partition_info);
declareArray(Partitions, Partition_attrib);

class Partition_map {
  public:
    Partition_map();
    ~Partition_map();

    void add_segment(Uint segid);
    // requires: "segid" not added before.
    // effects: adds "segid" to a partition.

    void remove_segment(Uint segid);
    // effects: removes "segid"

    void update_inset_change(Uint partition, int delta);
    // effects: changes the "delta_inset" value associated with 
    // "partition" by "delta".

    void reset_inset_change(Uint partition);
    // effects: resets the "delta_inset" value associated with 
    // "partition" to zero.

    int commonality_index(Uint partition, IntSet& others);
    // effects: computes a heuristic measuring how many other segment ids in
    // "others" "partition" also has.  Used for partition selection
    // policies in GC.

    Uint remove_segments(Uint partition, IntSet& segments);
    // effects: removes all segment numbers in "segments" that are also
    // in partition "partition".
    
    Uint max_partition();
    // effects: returns the max partition number.

/*      Uint least_inset_change(); */
/*      // effects: returns the partition id with the least */
/*      // associated "delta_inset" value. */

    int inset_change(Uint partition);
    // effects: returns the amount by which "partition"'s inset has changed
    // since the last GC of "partition".

    Partition_info from_segment(Uint segid) const {
	// requires: "segid" has been added to "this".
	// effects: returns partition_info for segment "segid".
	printf(">>> from_segment, segid=%d\n", segid); //DEBUG
	mutex->grab();
	Partition_info p = partition_infos[segid];
	mutex->release();
	printf("<<< from_segment\n"); //DEBUG
	return p;
    }

    Partition_info from_oref(Oref oref) const {
	// requires: "segid" of "oref" has been added.
	// effects: returns partition_info for "oref".
	mutex->grab();
	Partition_info p = partition_infos[Oref_segment(oref)];
	mutex->release();
	return p;
    }

    UintArray & to_segments(Uint partition) const {
	// requires:  "partition" exists in "this". 
	// effects: returns array of segment ids in "partition".
	mutex->grab();
	UintArray& segids = partitions[partition].segs;
	mutex->release();
	return segids;
    }

    Uint partition_phase(Uint partition) const {
	// effects: returns "partition"'s phase counter status
	return partitions[partition].partition_phase;
    }

    bool flag(Uint partition) const {
	// effects: returns "partition"'s flag status
	return partitions[partition].partition_flag;
    }

    Uint oref_to_orank(Oref oref) const {
	// effects: returns rank of "oref" in its partition.
	mutex->grab();
	Uint seg_rank = partition_infos[Oref_segment(oref)].rank;
	Uint res = Oref_screate(seg_rank, Oref_sindex(oref));
	mutex->release();
	return res;
    }

    Oref orank_to_oref(Uint orank, Uint partition) {
	// effects: returns oref given the partition and object rank.
	mutex->grab();
	Uint seg_rank = Oref_segment(orank);
	Uint segid = (partitions[partition].segs)[seg_rank];
	Oref res = Oref_screate(segid, Oref_sindex(orank));
	mutex->release();
	return res;
    }

  private:
    Allocator* allocator;
    Mutex* mutex; // for controlling concurrent access to internal data
    Partition_infos partition_infos; // segid -> <partitionid, rank>
    Partitions partitions; // partitionid -> [segids]
    IntArray delta_insets; // partitionid -> [delta_insets]
};

#endif /* _PARTITION_H */
