#include <stdlib.h>
#include <math.h>
#include <iostream.h>
#include <unistd.h>

#include "utils/Timer.h"
#include "utils/random_permuter.h"
#include "utils/random.h"

#include "common/or_obj.h"
// #include "common/wk_xref.h"
#include "fe/boot/fe_wk_xref.h"
#include "common/page.h"
#include "or/mm/mm.h"
#include "or/mm/log.h"
#include "or/mm/itable.h"

#include "gc.h"
#include "test_scan_mods.h"
#include "mod_list.h"
#include "collector.h"
#include "class_map.h"
#include "partition.h"

Test_scan_mods::Test_scan_mods () {
    mods = new Itable_Mods;

    obj_slots = 6; // number of slots in each object
    objects_in_page = Page_slots/(obj_slots+2); // assume < 2 slot overhead
    num_segments = 8192;
    start_segment = 64;
    gather_or_stats = 1;
    create_segments_randomly = 1;
    reference_range = 32*1024;
    random = 0;
    float locality = 1.0;
    bool spiked_distribution = 1;
    float cross = 0; // fraction of cross references

    // read parameters from file, if present
    char *params_filename = getenv("THOR_GC_TEST");
    if (params_filename == 0) return;
    // (ifstream >> char*) does not work with g++.
    FILE *params_file = fopen(params_filename, "r"); 
    if (params_file == NULL) return;
    cerr << "Reading GC TEST params file " << params_filename << endl;
    int const length = 128;
    char line[length];
    while (fgets(line, length, params_file) != NULL) {
	char option[length];
	float value;
	if (sscanf(line, "%s %f", option, &value) != 2) {
	    cerr << "GC TEST params file contains an illegal line" << endl;
	    continue;
	}
	if (0 == strcmp(option, "num_segments"))
	    num_segments = unsigned(value);
	else if (0 == strcmp(option, "start_segment"))
	    start_segment = unsigned(value);
	else if (0 == strcmp(option, "objects_in_page"))
	    objects_in_page = unsigned(value);
	else if (0 == strcmp(option, "gather_or_stats"))
	    gather_or_stats = value;
	else if (0 == strcmp(option, "create_segments_randomly"))
	    create_segments_randomly = int(value);
	else if (0 == strcmp(option, "reference_range"))
	    reference_range = unsigned(value);
	else if (0 == strcmp(option, "locality"))
	    locality = value/100;
	else if (0 == strcmp(option, "cross"))
	    cross = value/100;
	else if (0 == strcmp(option, "uniform_distribution"))
	    random = new Uniform_random(0, 2*reference_range);
	else if (0 == strcmp(option, "exponential_distribution"))
	    random = new Exponential_random(reference_range);
	else if (0 == strcmp(option, "normal_distribution"))
	    random = new Normal_random(0, reference_range);
	else if (0 == strcmp(option, "spiked_distribution"))
	    spiked_distribution = value;
	else 
	    cerr << "Unrecognized GC TEST param: " << option << endl;
    }

    if (cross != 0) { // set locality
	float p = gc->segments_per_partition*Max_pages_in_seg*objects_in_page;
	float d = reference_range;
	locality = 1 - cross/(d/p*(1-exp(-p/d)));
    }
    if (random==0) { 
	random = new Exponential_random(reference_range);
    }
    if (spiked_distribution) {
	cerr << "locality = " << locality << endl;
	Random *spike = new Uniform_random(0, 0);
	random = new Composed_random(locality, spike, random);
    }

    // Enter obj in class_map
    gc->class_map->store("wk_part", WK_Part_OREF, obj_slots, 0x1);
}
 
void Test_scan_mods::create_db() {
    // add segments into the partition map
    for (unsigned i = 0; i < num_segments; i++) {
	gc->partition_map->add_segment(i+start_segment);
    }

    if (gather_or_stats) gc->start_stats();
    Timer timer;
    timer.start();
    if (0==create_segments_randomly) {
	// create pages randomly
	cerr << "Creating subpages :";
	Uint const num_pages = num_segments * Max_pages_in_seg;
	Uint const start_page = start_segment * Max_pages_in_seg;
	Uint const pages_per_group = 1;
	Uint const groups = num_pages/pages_per_group;

	Random_permuter permuter(groups);
	for (Uint i = 0; i < groups; i++) {
	    Uint group = permuter.get();
	    Uint start = start_page + group*pages_per_group;
	    // create pages starting at start
	    for (Uint j = 0; j < pages_per_group; j++) 
		create_page(start + j);
   	    if (((i+1)*pages_per_group) % (32*Max_pages_in_seg) == 0) {
		
		gc->collector->main();
		// unref the mods to delete them
		int num_mods = mods->size();
		orx->mm->mutex->grab();
		for (int m = 0; m < num_mods; m++)
		    mods->slot(m)->unref();
		mods->clear();    
		orx->mm->mutex->release();
		
		cerr << " " << (i+1)*pages_per_group;
	    }
	}
    } else {
	// create segments in groups
	Uint const segs_per_group = create_segments_randomly;
	Uint const groups = num_segments/segs_per_group;
	cerr << "Creating segments :";
	Random_permuter permuter(groups);
	for (Uint i = 0; i < groups; i++) {
	    Uint group = permuter.get();
	    Uint start = start_segment + group*segs_per_group;
	    // create segs segments starting at seg
	    for (Uint j = 0; j < segs_per_group; j++) 
		create_segment(start + j);
	    if (((i+1)*segs_per_group) % 32 == 0) {
		
		gc->collector->main();
		// unref the mods to delete them
		int num_mods = mods->size();
		orx->mm->mutex->grab();
		for (int m = 0; m < num_mods; m++)
		    mods->slot(m)->unref();
		mods->clear();    
		orx->mm->mutex->release();
		
		cerr << " " << (i+1)*segs_per_group;
	    }
	}
    }
    cout << "Total time: " << int(timer.elapsed()) << endl;
    gc->print_stats(&cout);
    OR_stat or_stat;
    orx->mm->disk->stat(or_stat);
    orx->log->stat(or_stat);
    report_stats(stdout, or_stat);
    cerr << "Exiting OR after GC test" << endl;
    exit(0);
}

void Test_scan_mods::create_segment(Uint segid) {
    for (Uint pagenum = 0; pagenum < Max_pages_in_seg; pagenum++)
	create_page(Pageid_from_segid(segid, pagenum));
}

void Test_scan_mods::create_page(Uint pageid) {
    for (Uint onum = 0; onum < objects_in_page; onum++)
	create_object(Oref_create(pageid, onum));
}

void Test_scan_mods::create_object(Oref oref) {
    OR_obj* obj = OR_obj::create(obj_slots);
    obj->set_class_oref(WK_Part_OREF);
    OR_slot* fields = obj->fields();
    fields[0].set_oref(create_link(oref));
    Itable_Mod *mod = Itable_Mod::alloc(oref, NULL, obj, obj_slots);
    mod->ref();
    mods->append(mod);

    delete [] obj;
    
    gc->mod_list->mutex->grab();
    gc->mod_list->insert(mod);
    gc->mod_list->mutex->release();
}

Oref Test_scan_mods::create_link(Oref from) {
    Uint from_index = oref_to_oindex(from);
    // Generate normal random integer with stddev ref range
    int r = int(random->next());
    static int total_objects = num_segments * Max_pages_in_seg
	* objects_in_page;
    Uint to_index = (from_index + r) % total_objects;
    return oindex_to_oref(to_index);
}

Oref Test_scan_mods::oindex_to_oref(Uint oindex) {
    Uint onum = oindex % objects_in_page;
    Uint pageindex = oindex / objects_in_page;
    Uint pagenum = pageindex % Max_pages_in_seg;
    Uint segindex = pageindex / Max_pages_in_seg;
    Uint segid = segindex + start_segment;
    Oref oref = Oref_seg_page_create(segid, pagenum, onum);
    return oref;
}

Uint Test_scan_mods::oref_to_oindex(Oref oref) {
    Uint segid = Oref_segment(oref);
    Uint segindex  = segid - start_segment;
    Uint pagenum =  Oref_pagenum(oref);
    Uint onum = Oref_onum(oref);
    Uint oindex = segindex*(Max_pages_in_seg*objects_in_page)
	+ pagenum*(objects_in_page)
	+ onum;
    return oindex;
}
