#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#ifdef __linux__
#include <errno.h>
#endif
#include "utils/basic.h"
#include "or/thread.h"
#include "or/or.h"
#include "or/or_config.h"
#include "AsynchIO.h"

extern void *memcpy (void *__restrict __dest,
                     __const void *__restrict __src, size_t __n) __THROW;

// This file contains both AsynchIO and AsynchIO_daemon code


//////////////////////////////////////////////////////////////////
// Methods for AsynchIO object : the interface to asynchronouse IO
//////////////////////////////////////////////////////////////////

AsynchIO::AsynchIO() {
  Init(this,1);
}

AsynchIO::AsynchIO(int num) {
  Init(this,num);
}

void AsynchIO::Init(AsynchIO *aio, int num) {
  int i;

  if (num < 1) num = 1;
  if (num > MAX_AIO_THREADS) num = MAX_AIO_THREADS;
                        IF_DEBUG(5) printf("         initing AIO (%d)\n",num);
  aio->threads    = num;
  aio->daemon     =(AsynchIO_daemon **)malloc(sizeof(AsynchIO_daemon *)*num);
  aio->current    =(aio_completion_t **)malloc(sizeof(aio_completion_t *)*num);
  aio->pending    = new AIO_Queue;
  aio->completed  = new AIO_Queue;
  aio->mtx        = new Mutex;
  aio->Message    = new Condition(mtx);
  aio->should_exit= 0;
  aio->Response   = new Condition(mtx);
  aio->Finished   = new Condition(mtx);
  aio->exited     = num;
  aio->errors     = 0;
  aio->requests   = 0;

  // spawn threads to handle AsynchIO requests
  
  for (i = 0; i < num; i++) {
    aio->current[i] = NULL;
    aio->daemon[i] = new AsynchIO_daemon(aio,i);
    aio->daemon[i]->start();
  }
}

AsynchIO::~AsynchIO() {
  // kill thread handling AsynchIO (once IO is done)
  mtx->grab();
  should_exit = 1;
  while (exited < threads) {
    Message->broadcast();
    Finished->wait();
  }
  free(current);
  free(daemon);
  delete pending;
  delete completed;
  delete mtx;
  delete Message;
  delete Finished;
  delete Response;
  
}

void AsynchIO::add_request_to_pending(aio_completion_t *foo, int MessageSignal) {
  mtx->grab();
    foo->reqID = requests;
    pending->enqueue(foo);
    requests++;
    if (MessageSignal == AIO_YES_SIGNAL)    Message->signal();
    if (MessageSignal == AIO_YES_BROADCAST) Message->broadcast();
  mtx->release();
}

void AsynchIO::init_completion_t(aio_completion_t *foo, aiocb_t *ap) {
  foo->aio_aiocb = ap;
  foo->aio_result = 0;
  foo->aio_error = 0;
  foo->next = NULL;
  foo->completed = 0;
}

int AsynchIO::partials_completed(aio_completion_t **partials, int nent) {
  int still_pending = 0;
  mtx->grab();
    still_pending = 0;
    for (int i = 0; i < nent; i++)
      if (partials[i]->completed == 0) { still_pending = 1; break; }
  mtx->release();
  return (still_pending);
}

int AsynchIO::read(aiocb_t *ap) {
  IF_DEBUG(5) printf("   AsynchIO::read\n");

  aio_completion_t *result = (aio_completion_t *)malloc(sizeof(aio_completion_t));
  ap->aio_lio_opcode = LIO_READ;
  init_completion_t(result,ap);
  add_request_to_pending(result,AIO_YES_SIGNAL);
  
  IF_DEBUG(5) printf("   Request %d (read %d: %ld offset %d bytes) submitted\n",
		     requests, ap->aio_fildes, ap->aio_offset, ap->aio_nbytes);
  return (0);
}

int AsynchIO::write(aiocb_t *ap) {
  IF_DEBUG(5) printf("   AsynchIO::write\n");

  aio_completion_t *result = (aio_completion_t *)malloc(sizeof(aio_completion_t));
  ap->aio_lio_opcode = LIO_WRITE;
  init_completion_t(result,ap);
  add_request_to_pending(result,AIO_YES_SIGNAL);

  IF_DEBUG(5) printf("   Request %d (write %d: %ld offset %d bytes) submitted\n",
		     requests, ap->aio_fildes, ap->aio_offset, ap->aio_nbytes);
  return (0);
}

int AsynchIO::listio(int mode,aiocb_t *list[],int nent,struct sigevent *sig){
  int i;
  aiocb_t *notify;
  aio_completion_t *notify_r;
  aio_completion_t **partials;

  //  XXX - if mode is LIO_WAIT do we also ignore all individual signals
  //        in the list, or just the overall sig?
  partials = (aio_completion_t **)malloc(sizeof(aio_completion_t *) * nent);
  int num_partials = 0;
  for (i = 0; i < nent; i++) {
    partials[i] = NULL;
    if (list[i]->aio_lio_opcode != LIO_NOP) {
      notify_r = (aio_completion_t *)malloc(sizeof(aio_completion_t));
      partials[i] = notify_r;
      num_partials++;
      init_completion_t(notify_r,list[i]);
      add_request_to_pending(notify_r,AIO_NO_SIGNAL);

      IF_DEBUG(5) printf("               Request %d 0x%x added to pending\n",
			 requests,(int)list[i]);
    }
  }

  IF_DEBUG(1) printf("AsynchIO::listio (0x%x):%d\n",(int)list,pending->size());
  
  if (notify_r == NULL) return 0;                  // return if no IO queued

  if (mode == LIO_NOWAIT) {                                // queue a notify request and return
    notify = (aiocb_t *)malloc(sizeof(aiocb_t));
    notify->aio_lio_opcode = LIO_COMPOSITE;
    notify->signal = sig->sigev_notify;
    notify->signo = sig->sigev_signo;
    memcpy(&notify->aio_sigevent,sig,sizeof(struct sigevent));

    notify_r = (aio_completion_t *)malloc(sizeof(struct aio_result));
    init_completion_t(notify_r, notify);
    notify_r->partials = partials;
    notify_r->num_partials = num_partials;
    add_request_to_pending(notify_r,AIO_YES_BROADCAST);
    
  } else {                                      // wait until last IO is done
    mtx->grab();                              // and then free partials and return
    
    int still_pending = 1;
    while (still_pending == 1) {
      Message->signal();
      Response->wait();
      still_pending = 0;
      for (int j = 0; j < nent; j++) {
	if (partials[j]->completed == 0) {
	  still_pending = 1;
	  break;
	}
      }
      mtx->release();
    }
    free(partials);
  }
  IF_DEBUG(1) printf("  AsynchIO::listio (0x%x) done\n",(int)list);
  return 0;
}

size_t AsynchIO::value(aiocb_t *ap) {
  aio_completion_t *result;
  int r,i;
  IF_DEBUG(5) printf("   AsynchIO::value (0x%x)\n",(int)ap);
  
  mtx->grab();

  result = completed->find_aiocb(ap);
  if (result != NULL) {
    completed->remove(result);
  } else {
    result = pending->find_aiocb(ap);
    if (result != NULL) pending->remove(result);
  }
  // at this point, if ap was either completed or pending,
  // it has been removed and result contains it.

  if (result == NULL) {                        // if it is not pending or done
    r = -1;
    errno = EINVAL;
    for (i = 0; i < threads; i++)              // it is either invalid
      if (current[i] != NULL) 
	if (current[i]->aio_aiocb == ap){      // or it is in progress
	  errno = EINPROGRESS;
	  IF_DEBUG(1) printf("ERROR  AsynchIO::value InProgress\n");
	}
    if (errno == EINVAL)
      IF_DEBUG(1)   printf("ERROR  AsycnhIO::value Invalid aiocb_t (0x%x)\n",(int)ap);

  } else {                                     // if we found it, report
    
    if (result->completed == 0) {                // Pending, or
      errno = EINPROGRESS;
      IF_DEBUG(1) printf("ERROR  AsynchIO::value Pending on request %d\n",result->reqID);
      r = -1;
    } else if (result->completed == -1) {         // Canceled, or
      IF_DEBUG(1) printf("ERROR  AsynchIO::value Canceled on request %d\n",result->reqID);
      errno = ECANCELED;
      r = -1;
    } else {
      if (result->aio_error != 0) {              // Error, or
	IF_DEBUG(1) printf("ERROR  AsynchIO::value OpError (%d) on request %d\n",
			   result->aio_error,result->reqID);
	errno = result->aio_error;
	r = -1;
      } else {
	r = result->aio_result;                  // Result
      }
      free(result);                            // Free aio_result storage
    }
  }
  mtx->release();
  return (r);
}

int AsynchIO::error(const aiocb_t *ap) {
  aio_completion_t *result;
  int i,r = 0;
  mtx->grab();
  result = completed->find_aiocb(ap);
  if (result == NULL) result = pending->find_aiocb(ap);
 
  if (result == NULL) {                        // if not pending or done, it
    r = EINVAL;                                // is invalid or in progress
    for (i = 0; i < threads; i++)
      if ((current[i] != NULL) && (current[i]->aio_aiocb == ap)) {
	r = EINPROGRESS;  break; }
  } else                                       // return in progress or
    if (result->completed == 0)       r = EINPROGRESS;
    else if (result->completed == -1) r = ECANCELED;
    else                              r = (result->aio_error);
  mtx->release();
  return (r);
}

int AsynchIO::cancel_fd(int fd) {
  // Cancel all pending requests of a file descriptor fd.  If some are
  // in progress, return AIO_NOTCANCELED.  If all are completed, return
  // AIO_ALLDONE.  If none pending, in progress or done, return EBADF.
  int i,p = 0;
  aio_completion_t *temp;
  int r = 0;

  mtx->grab();                            // remove all pending requests
  temp = pending->find_fd(fd);              // with that fd, set them as
  while (temp != NULL) {                    // canceled and add them to
    temp->completed = -1;                   // the completed queue
    p++;
    pending->remove(temp);
    completed->enqueue(temp);
    temp = pending->find_fd(fd);
  }
  mtx->release();
  for (i = 0; i < threads; i++) {
    if (current[i] != NULL)
      if (current[i]->aio_aiocb->aio_fildes == fd) // if in progress,
	return (AIO_NOTCANCELED);                  // return NOTCANCELED
  }
  
  if (p > 0)                                // if any were canceled, return
    r = (AIO_CANCELED);                     // such
  else {
    mtx->grab();                          // otherwise, if any were
    temp = completed->find_fd(fd);          // completed, return ALLDONE
    mtx->release();
    if (temp != NULL)
      r = (AIO_ALLDONE);
    else {                                  // else error EBADF
      errno = EBADF;
      r = -1;
    }
  }
  return r;
}


int AsynchIO::cancel_aiocb(aiocb_t *ap) {
  // Cancel a request of a specific aiocb.  If it is being worked on,
  // return AIO_NOTCANCELED.  If it is done, return AIO_ALLDONE.  If it
  // is neither pending, in progress, or done, return EBADF.
  int i;
  aio_completion_t *temp;

IF_DEBUG(5) printf("   AsynchIO::cancel_aiocb\n");
  mtx->grab();
  temp = pending->find_aiocb(ap);
  if (temp != NULL) {                       // if pending, cancel and move
    pending->remove(temp);                  // to the completed queue
    temp->completed = -1;
    completed->enqueue(temp);
    mtx->release();
    return (AIO_CANCELED);
  }
  
  for (i = 0; i < threads; i++) {
    if (current[i] != NULL) 
      if (current[i]->aio_aiocb == ap) {           // if in progress, say so
	mtx->release();                         // release mutex and return
	return (AIO_NOTCANCELED);
      }
  }
  
  temp = completed->find_aiocb(ap);
  mtx->release();
  if (temp != NULL)                         // if already done, say so
    return (AIO_ALLDONE);
  else {                                    // else, error EBADF
    errno = EBADF;
    return (-1);
  }
}


int AsynchIO::cancel(int fd, aiocb_t *ap) {
IF_DEBUG(5) printf("   AsynchIO::cancel\n");
  if (ap == NULL)                   // if canceling all IO of an fd
    return cancel_fd(fd); 
  else                              // if canceling a singled IO request
    return cancel_aiocb(ap);
}


///////////////////////////////////////////////////////////////////////////
// Methods for AsynchIO_daemon object : the background IO processing thread
///////////////////////////////////////////////////////////////////////////

AsynchIO_daemon::AsynchIO_daemon() {
  aio = NULL;
  ID = -666;
}
AsynchIO_daemon::AsynchIO_daemon(AsynchIO *aiop, int id) {
  if (aiop == NULL){ ID = -666; aio = NULL;}
  else {             ID = id;   aio = aiop; }
}

void AsynchIO_daemon::main() {
  int running = 1;
  int processed = 0;
  aiocb_t *job;
  
  if (ID == -666) {
    printf("Daemon failed...invalid initialization\n");
    return;
  }
  
  aio->mtx->grab();  aio->exited--; aio->mtx->release();  // register thread is running
  
  while (running) {
    aio->mtx->grab();                         // wait until something to do
    while ((aio->pending->size() == 0) && (aio->should_exit == 0)) {
      aio->Message->wait();
    }
    processed = 0;
    while (aio->pending->size() != 0) {        // pluck off first IO request
      processed++;
      aio->current[ID] = aio->pending->dequeue();
      aio->mtx->release();                     // release mutex and do job
      job = aio->current[ID]->aio_aiocb;
      if      (job->aio_lio_opcode == LIO_READ)      do_read();
      else if (job->aio_lio_opcode == LIO_WRITE)     do_write();
      else if (job->aio_lio_opcode == LIO_COMPOSITE) do_composite();
      else                                           do_error();
      aio->mtx->grab();                         //grab mutex for next
    }                                           //while-loop iteration
    
    if (aio->should_exit == 1) {     // complete, and die if appropriate
      running = 0;                   // enable us to fall out of while loop
      aio->exited++;                 // increment exited counter
      aio->Finished->signal();       // signal cycle completion
      aio->Response->signal();
      IF_DEBUG(4) printf(" %d Signaled Finished\n",ID);
    }
    aio->mtx->release();
  }                               // loop back to wait state if not exited
  IF_DEBUG(4) printf(" %d Exited\n",ID);
}


void AsynchIO_daemon::do_read() {
  aiocb_t *job = aio->current[ID]->aio_aiocb;
  
  lseek(job->aio_fildes, job->aio_offset, SEEK_SET);
  aio->current[ID]->aio_result = read(job->aio_fildes,
				      (void *)job->aio_buf,
				      job->aio_nbytes);
  
  if (aio->current[ID]->aio_result == -1)  aio->current[ID]->aio_error = errno;
  else                                     aio->current[ID]->aio_error = 0;
 
  aio->current[ID]->next = NULL;  
  aio->current[ID]->completed = 1;
  queue_completed_request(0);
  aio->current[ID] = NULL;
}

void AsynchIO_daemon::do_write() {
  aiocb_t *job = aio->current[ID]->aio_aiocb;
  
  lseek(job->aio_fildes, job->aio_offset, SEEK_SET);
  aio->current[ID]->aio_result = write(job->aio_fildes,
				       (const void *)job->aio_buf,
				       job->aio_nbytes);
  
  if (aio->current[ID]->aio_result == -1)  aio->current[ID]->aio_error = errno;
  else                                     aio->current[ID]->aio_error = 0;
 
  aio->current[ID]->next = NULL;  
  aio->current[ID]->completed = 1;
  queue_completed_request(0);
  aio->current[ID] = NULL;
}

void AsynchIO_daemon::do_composite() {
  aiocb_t *job = aio->current[ID]->aio_aiocb;
  aio_completion_t *cur = aio->current[ID];
  aio->mtx->grab();
  int still_pending = 0;
  for (int i = 0; i < cur->num_partials; i++)     // test completion of parts
    if (cur->partials[i]->completed == 0) { still_pending = 1; break;}
  while (still_pending == 1) {                    // while some parts aren't done
    aio->Message->signal();
    aio->Response->wait();
    still_pending = 0;
    for (int i = 0; i < cur->num_partials; i++)   // retest completion of parts
      if (cur->partials[i]->completed == 0) { still_pending = 1; break;}
    aio->mtx->release();
  }
  aio->current[ID]->next = NULL;
  aio->current[ID]->completed = 1;
  queue_completed_request(0);

  if (job->signal != 0) { // send appropriate signal when done
    printf("Ack! We should generate signal here but do not\n");
  }
  aio->current[ID] = NULL;
}

void AsynchIO_daemon::do_error() {
  aiocb_t *job = aio->current[ID]->aio_aiocb;
  IF_DEBUG(1) printf(" Bad Opcode (%d) on request %d\n",
		     job->aio_lio_opcode,aio->current[ID]->reqID);
  aio->current[ID]->aio_result = -1;       
  aio->current[ID]->aio_error  = EINVAL;
  queue_completed_request(1);
  aio->current[ID] = NULL;
}

void AsynchIO_daemon::queue_completed_request(int error) {
  aio->mtx->grab();
  aio->completed->enqueue(aio->current[ID]);
  aio->current[ID]->completed = 1;
  aio->Response->signal();
  if (error) aio->errors++;
  aio->mtx->release();
}

////////////////////////////////////////////////////////////////////
// AIO_Queue
////////////////////////////////////////////////////////////////////

AIO_Queue::AIO_Queue() {
  head = NULL;
  tail = NULL;
}

AIO_Queue::~AIO_Queue() {
}

void AIO_Queue::enqueue(aio_completion_t *ap) {
  // Effects : Adds ap to the tail of the queue
  if (head == NULL) {  head = ap;       tail = ap;}
  else              { tail->next = ap;  tail = ap;}
}

aio_completion_t *AIO_Queue::dequeue() {
  // Effects : Removes first element from the queue and returns it
  //           or NULL if empty
  aio_completion_t *temp;
  if (head == tail) { temp = head; head = NULL;       tail = NULL;}
  else              { temp = head; head = head->next;             }
  return (temp);
}

int AIO_Queue::size() {
  // Effects: Returns # of elements on queue
  int i = 0;
  aio_completion_t *temp = head;
  while (temp != NULL) { i++; temp = temp->next;}
  return (i);
}

aio_completion_t *AIO_Queue::find_aiocb(const aiocb_t *ap) {
  // Effects : returns ptr to first elm whose (aio_aiocb == ap), NULL if none
  aio_completion_t *temp = head;
  while ((temp != NULL) && (temp->aio_aiocb != ap))    temp = temp->next;
  return temp;
}

aio_completion_t *AIO_Queue::find_fd(int fd) {
  // Effects: returns ptr to first elm whose (fildes == fd), NULL if none
  aio_completion_t *temp = head;
  while ((temp != NULL) && (temp->aio_aiocb->aio_fildes != fd))  temp = temp->next;
  return temp;
}


void AIO_Queue::remove(aio_completion_t *ap) {
  // Effects : removes all elements in queue equal to ap
  aio_completion_t *temp = head;
  if (ap == NULL) return;                      // Can't remove empty set
  while (head == ap) head = head->next;        // remove all initial ap's
  if (head == NULL) { tail = NULL;  return; }

  while (temp->next != NULL) {
    if (temp->next == ap)  temp->next = temp->next->next;
    else                   temp = temp->next;
  }
  if (temp->next == NULL) tail = temp;
}
