#ifndef _LDFORMAT_H
#define _LDFORMAT_H

// Disk format for log-structured layout policy.

#include "dformat.h"

#define REGION_MAGIC  0xAE51
#define FRAG_MAGIC    0xAE52

// XXX Each region size is fixed.  The first region
// starts right after the super-blocks.
static const int region_size = 512 * 1024;

// The shift amount needed to efficiently divide/multiply by the region size.
static const int region_shift = 19;

// Byte address of zeroeth region on disk
static const int region_start = (DISK_SB_2 + DISK_SB_SIZE) << DISK_UNIT_SHIFT;

inline int a2r(int byte_address) {
    // effects	Return region number that contains specified byte address
    return (byte_address - region_start) >> region_shift;
}

inline int r2a(int rnum) {
    // effects	Return starting byte address of named region
    return (rnum << region_shift) + region_start;
}

inline int s2r(int slot_address) {
    // effects	Return region number that contains specified slot address
    return a2r(slot_address * sizeof(OR_slot));
}

inline int r2s(int rnum) {
    // effects	Return starting slot address of named region
    return r2a(rnum) / sizeof(OR_slot);
}

inline int b2r(int blk) {
    // effects	Return region number that contains specified block
    return a2r(blk << DISK_UNIT_SHIFT);
}

inline int r2b(int rnum) {
    // effects	Return starting block number of named region
    return r2a(rnum) >> DISK_UNIT_SHIFT;
}

inline int s2b(int slot_address) {
    // effects	Return block number for specified slot address
    return (slot_address * sizeof(OR_slot)) >> DISK_UNIT_SHIFT;
}

inline int b2s(int blk) {
    // effects	Return slot address for specified block number
    return (blk << DISK_UNIT_SHIFT) / sizeof(OR_slot);
}

// Region header information.
struct Ldisk_Region {
    ubits32	magic;		// Region magic number
    ubits32	stamp;		// version stamp incremented per region write
    ubits32	frags;		// Number of fragments in region
    ubits32	size;		// Combined size of all fragments (in slots)
    ubits32	sum;		// Check sum of region contents?

    // Pad to slot boundary.
    ubits32	pad[1];

    // Rest of the region
    OR_slot	buf[1];
};

// Info kept per object in a fragment header table.
struct Ldisk_Entry {
    ubits16	id;		// Object index
    ubits16	offset;		// Offset in # of slots from start of fragment
};

// Format of segment fragments.
struct Ldisk_Fragment {
    ubits16	magic;		// Magic number
    ubits16	id;		// Segment number
    ubits16	slots;		// Fragment size
    ubits16	count;		// Number of objects in segment
    ubits32	stamp;		// Segment age

    // Pad to slot boundary.
    ubits32	pad[1];

    // Structure has at least two header entries (for padding).
    Ldisk_Entry	header[2];	// Variable length object location table

    // Object data occurs after the header.  The information
    // before the header occupies exactly one slot.
    // For "K" objects, the header occupies ceil(K/2) slots.
    // Therefore, the first object starts at slot "ceil(K/2) + 1".
};

#endif /* _LDFORMAT_H */
