// Copyright 1995 Barbara Liskov

// \section{Memory Manager Implementation}

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <iostream.h>

#include "utils/fail.h"
#include "utils/th_assert.h"

#include "common/class_code.h"
#include "common/obj_bitfield.h"
#include "common/or_obj.h"
#include "common/modset.h"
#include "common/tid.h"
#include "common/transaction.h"
// #include "common/wk_xref.h"
#include "fe/boot/fe_wk_xref.h"

#include "or/or.h"
#include "or/or_config.h"
#include "or/runtime.h"
#include "or/thread.h"

#include "or/gc/partition.h"
#include "or/gc/gc.h"
#include "or/gc/collector.h"

#include "dformat.h"
#include "handle.h"
#include "itable.h"
#include "log.h"
#include "logrecord.h"
#include "mm.h"
#include "mm_stats.h"
#include "scache.h"
#include "segment.h"
#include "segtable.h"

#include "config/vdefs/SMALL_SLOTS.h"

static inline Uint Blocks_per_segment() {
    // effects: Returns the no. of blocks in a segment
    // Assumes that Page_size_bits >= DISK_UNIT_SHIFT
    int result = Max_pages_in_seg << (Page_size_bits - DISK_UNIT_SHIFT);
    return result;
}


MM::MM(bool init_flag) {
    orx->mm = this;
    mutex = new Mutex;
    new_arena = -1;
    gc_on = FALSE;
    cache = new Cache(orx->config->cache_size);
    itable = new Itable;
    stamp_counter = 0;
    retry = new MM_Handle;
    segment_request_id = -1;
    segment_requested = 0;

    mods	= 0;
    absorbed	= 0;
    installs	= 0;
    fetches	= 0;
    misses	= 0;
    frag_writes	= 0;
    frag_sizes	= 0;
    clean_count	= 0;
    clean_live	= 0;

    if (init_flag)
	init();
    else
	recover();

#if PAGING
    orx->pages = new Page_Table(or, segtable);
#endif

    stats = new MM_Stats();
}

/* \subsection{Reading Objects} */

MM_Handle* MM::fetch(Oref oref) {
    MM_Handle* h;
    mutex->grab(); {
	h = do_fetch(oref, FALSE);
    } mutex->release();
    return h;
}

MM_Handle* MM::fast_fetch(Oref oref) {
    MM_Handle* h;
    mutex->grab(); {
	h = do_fetch(oref, TRUE);
    } mutex->release();
    return h;
}

void MM::release(MM_Handle* h) {
    mutex->grab(); {
	delete h;
    } mutex->release();
}

void MM::release_list(MM_HandleList* list) {
    mutex->grab(); {
	int num = list->size();
	for (int i = 0; i < num; i++)
	    delete list->slot(i);
    } mutex->release();
}

MM_Handle* MM::do_fetch(Oref oref, bool fast) {
    // Remember the initial oref to place into the handle
    Oref orig = oref;

    // We need a loop to follow forwarding pointers.
    bool missed = FALSE;
    while (1) {
//  #ifdef __linux__
//  	pth_yield(NULL);
//  #endif
	OR_obj* obj;
	MM_Handle* result;

	// We first look for the named object in the "itable", and if
	// that does not succeed, we examine the cache.
	Itable_Mod* mod = itable->lookup(oref);
	if (mod != 0) {
	    result		= new MM_Handle;
	    result->in_itable	= TRUE;
	    result->oref_	= orig;
	    result->obj_	= mod->object();
	    result->mod		= mod;
	    mod->ref();
	}
	else {
	    // Object is not in the itable.  Look in the cache or on disk.
	    result = read_object(oref, fast);
	    if (result == retry) {
		missed = TRUE;
		continue;
	    }
	    if (result == 0) return 0;
	}

	// Forwarding code deleted
	obj = result->obj();

	// We increment miss count only at end to avoid multiple misses
	// for a single fetch.
	if (!fast) {
	    fetches++;
	    if (missed) misses++;
	}

	// Found it.  Make sure we report the requested oref.
	result->oref_ = orig;
	result->from_disk = missed;
	return result;
    }
}

Oref MM::directory() {
    MM_Handle* h = fetch(super->root);
    th_assert(h != (MM_Handle *)0x0, "could not fetch hidden OR root");
    Oref result = (((OR_hidden_root*) (h->obj()))->root.get_oref());
    mutex->grab(); {
	delete h;
    } mutex->release();
    return result;
}

// \subsection{Modifications}

void MM::set_gc(bool status=TRUE) {
    gc_on = status;
}


// Convert bytes to slots
inline int b2s(int b) {
    return (b + sizeof(OR_slot) - 1) / sizeof(OR_slot);
}

int MM::alloc_new_segment() {
    mutex->grab();
    // Assuming that pagesize is bigger than disk block size
    int blocks = Blocks_per_segment(), segid = -1;
    if (!segtable->add(segid, blocks))
	warn("could not allocate space for a big object segment");
    if (gc)
	gc->partition_map->add_segment(segid);
    mutex->release();
    return segid;
}

long MM::new_stamp() {
    stamp_counter++;
    return stamp_counter;
}

void MM::stat(OR_stat& stat) {
    mutex->grab(); {
	disk->stat(stat);
	stat.mods	  = mods;
	stat.absorbed	  = absorbed;
	stat.installs	  = installs;
	stat.logsize	  = orx->log->current_size();
	stat.fetches	  = fetches;
	stat.misses	  = misses;
	stat.frag_writes  = frag_writes;
	stat.frag_sizes	  = frag_sizes;
	stat.clean_count  = clean_count;
	stat.clean_live   = clean_live;
    } mutex->release();
}

void MM::resize_cache(int bytes) {
    mutex->grab(); {
	cache->resize(bytes);
    } mutex->release();
}

int MM::cache_size() {
    return cache->get_max_size();
}

void MM::resize_log(int bytes) {
    // Log has its own mutex.  No need to grab mm->mutex.
    orx->log->resize(bytes);
}

Oref MM::reserve(int num_slots) {
    static int cur_segment = -1; // Cur seg where objects are allocated
    static Uint cur_page = 0;    // Page on seg where objs are allocated
    static Uint cur_onum = 0;    // Next onum to be used on cur_page

    // Used for checking if an allocation can be made on a page
    static Page dummy_page(orx->config->ornum);

    int loop_count = 0;
    while (1) {
//  #ifdef __linux__
//  	pth_yield(NULL);
//  #endif
	th_assert(++loop_count <= 2, "Looping in reserve");

	if (cur_segment >= 0 && cur_onum <= Max_onum &&
	    dummy_page.allocate(cur_onum, num_slots)) {
	    //  Allocated object on the current page
		break;
	}
	// Could not allocate in the current page
        dummy_page = Page(orx->config->ornum);
        if (cur_segment < 0 || cur_page == Max_pages_in_seg - 1) {
	    // Need to create a new segment
	    cur_segment = alloc_new_segment(); // XXX Lock grabbed inside
	    th_assert(cur_segment >= 0, "Segment allocation failed");
	    cur_page = 0; cur_onum = 0;
	} else {
	    // cur_segment >= 0 AND cur_page != Max_pages_in_seg - 1
	    // AND no space or no onums on cur_page. Therefore, move to
	    // the next page
	    cur_page++; cur_onum = 0;
	}
    }
    Oref oref = Oref_seg_page_create(cur_segment, cur_page, cur_onum);
    cur_onum++;
    return oref;
}


// XXX Speed this up to avoid creating one log record per object?

Oref MM::make_object(OR_obj* obj,int num_slots, Oref class_oref) {
    Transaction trans;
    trans.mos = new Modset;
    trans.nos = new Modset;

    Oref oref = reserve(num_slots); // Reserve num_slots and get an oref

    trans.nos->add_object(oref, obj, num_slots, class_oref);

    // XXX Currently the TID value does not matter, so we do not initialize it
    Log_Record* rec = new Data_Log_Record(&trans);
    Log_Index index = orx->log->append(rec);

    orx->log->flush(index);
    rec->install();
    orx->log->installed(index);

    return oref;
}

/* \subsection{Initialization and Recovery} */


void MM::init() {
    // Create disk
    int fd = ::open(orx->config->disk, O_RDWR|O_SYNC|O_CREAT, 0666);
    if (fd < 0) sysfail(orx->config->disk);
    disk = Disk::open_fd(fd);

    // Check for disk vs. file
    struct stat prop;
    if (fstat(fd, &prop) < 0) sysfail(orx->config->disk);
    th_assert((S_ISCHR(prop.st_mode) || S_ISREG(prop.st_mode)),
	      "specified file is neither a disk, nor a regular file");
    if (S_ISCHR(prop.st_mode) && orx->config->debug_level > 0) {
	// Report actual statistics
	fprintf(stderr, "size    = %5ld\n", (long) prop.st_size);
	fprintf(stderr, "blksize = %5d\n", (int) prop.st_blksize);
	fprintf(stderr, "blocks  = %5d\n", (int) prop.st_blocks);

	// XXX _How do I get the actual disk size?_
    }

    // Create superblock

    super			= new Disk_SB;
    super->magic		= DISK_SB_MAGIC;
    super->checksum		= 0;
    super->timestamp		= 0;
    super->size			= orx->config->init->size;
    super->segprefsize		= Blocks_per_segment();
    super->segtable.address	= 0;
    super->segtable.count	= 0;
    super->root                 = Oref_screate(0, 0);
    sprintf(super->log, "logfile");

    // The following code tries to fit at least two non-big objects
    // into normal sized segments to avoid too much space loss
    // due to fragmentation.  We can achieve this by making
    // "big_threshold" be just less than half the preferred segment
    // size.

    int big_blocks = (super->segprefsize - 1) / 2;
    big_threshold = big_blocks << DISK_UNIT_SHIFT;

    create_seg_table();

    // Create hidden root for the OR
    super->root = make_root();

    // Now that the root object is known, write the superblocks to disk
    mutex->grab(); {
	Disk_Range r;
	r.count = DISK_SB_SIZE;
	r.address = DISK_SB_1;
	if (! disk->write(super, r)) sysfail(orx->config->disk);
	r.address = DISK_SB_2;
	if (! disk->write(super, r)) sysfail(orx->config->disk);
    } mutex->release();

    // Make sure everything is written out
    orx->log->flush(orx->log->high());
    orx->log->drain();
    if (gc)
	gc_on = TRUE;
}

void MM::recover() {
    disk = Disk::open(orx->config->disk);
    if (disk == 0) sysfail(orx->config->disk);

    mutex->grab(); {
	super = new Disk_SB;
	bool result = Disk_ReadSuper(disk, super);
	if (result == 0)
	  th_fail("could not read disk superblock");

        // Set big threshold (See init() for details)
        int big_blocks = (super->segprefsize - 1) / 2;
        big_threshold = big_blocks << DISK_UNIT_SHIFT;

    } mutex->release();

    recover_seg_table();
}

// effects	Creates an object of the specified number of field slots
static OR_obj* make_obj(int num_slots) {
    OR_obj* x = ((OR_obj*) (new OR_slot[OR_obj_headers + num_slots]));
    return x;
}

// The code here depends on the rep of the directory class
// defined in "fe/types-th/orDirectory.th".

Oref MM::make_root() {
    // XXX The object formats here have to be checked
    OR_obj* x;

    // XXX Make a rootDir object the apparent root
    Oref expected = Oref_seg_page_create(5, 0, 0);

    x = make_obj(3);
    OR_slot* fields = x->fields();
    fields[0].set_oref(expected);
    fields[1].set_oref(expected);
    fields[2].value32 = -1;
    // ZZZ: num_slots (=3) has to be kept in sync
    Oref rootDir = make_object(x, 3, WK_thor_Directory_OREF);
    cout << "WK_thor_Directory_OREF=" << WK_thor_Directory_OREF << "\n";
    th_assert(expected == rootDir, "Wrong expected root oref");
    delete [] x;

    // Now make the real hidden root object
    x = make_obj(1);
    fields = x->fields();
    fields->set_oref(rootDir);
    Oref root = make_object(x, 1, 0);
    delete [] x;

    return root;
}


// Oref MM::make_root() {
//     // XXX The object formats here have to be checked
//     OR_obj* x;
// 
//     // XXX Make a counter object the apparent root
//     Oref expected = Oref_seg_page_create(5, 0, 0);
// 
//     x = make_obj(2);
//     OR_slot* fields = x->fields();
//     fields[0].value32      = 0;
//     fields[1].set_oref(expected);
//     Oref counter = make_object(x, 2, WK_Counter_OREF);
//     th_assert(expected == counter, "Wrong expected root oref");
//     delete [] x;
// 
// // XXX Following removed for testing with counter objects
// #if 0 
//     // Make an empty vector of nodes
//     x = make_obj(0);
//     Oref vec = make_object(x, 0, WK_vec_OF_node_OREF);
//     delete [] x;
// 
//     // Make an array of nodes that uses the preceding vector for its rep
//     x = make_obj(5);
//     OR_slot* fields = x->fields();
//     fields[0].oref         = Oref_to_ivar(vec);
//     fields[1].value32      = 0;
//     fields[2].value32      = 0;
//     fields[3].value32      = 0;
//     fields[4].value32      = 1; // Taken from the old code XXX
//     Oref rep = make_object(x, 5, WK_array_OF_node_OREF);
//     delete [] x;
// 
//     // Make the directory object that uses the preceding array as its rep
//     x = make_obj(1);
//     fields = x->fields();
//     fields->oref = Oref_to_ivar(rep);
//     Oref dir = make_object(x, 1, WK_simple_directory_OREF);
//     delete [] x;
// 
// #endif
// 
//     // Now make the real hidden root object
//     x = make_obj(1);
//     fields = x->fields();
//     fields->set_oref(counter);
//     Oref root = make_object(x, 1, 0);
//     delete [] x;
// 
// 
// 
//     return root;
// }
// 
// 
// 
// 
// 
// 
// 
// 
// 
