// Copyright 1995 Barbara Liskov

/*
\section{Segment Table Interface}

A segment table maps from segment identifiers to disk ranges
occupied by the corresponding segments.

Concurrent access to the segment table is controlled via "mm->mutex".
All of the segment table methods require that the caller holds "mm->mutex".
*/

#ifndef _SEGTABLE_H
#define _SEGTABLE_H

#include "utils/basic.h"
#include "logrecord.h"
#include "disk.h"
#include "config/vdefs/REPLICATION.h"

struct	Disk_Range;
class	Disk_RootObj;
class	Disk_LeafObj;

class Segment_Table {
  public:
    Segment_Table();
    // requires	 Caller holds "mm->mutex".
    // effects	 create empty segment table

    void recover();
    // requires  Called once during recovery or initialization.
    //		 Caller holds "mm->mutex".
    // effects   Recover segment table from persistent storage.
    //		 Crashes OR if there is an error.

    bool find(int segid, Disk_Range& range) const;
    // requires  Caller holds "mm->mutex".
    // modifies	 range
    // effects	 If segment named by "segid" exists, then sets
    //		 "range" to its address on disk and returns TRUE.
    //		 Else, returns FALSE and does not modify "range".

    bool add(int& newid, int blocks);
    // requires  Caller holds "mm->mutex".
    // modifies  this, newid
    // effects   Allocates unmapped segment id and range for a segment
    //		 occupying the specified number of blocks, adds
    //		 mapping (id->range) and sets "newid" to the new
    //		 segment id.  Returns true iff new segment is added.
    //		 "newid" is modified iff a new segment is added.


    void install_mapping(int segid, Disk_Range range);
    // requires  Caller holds "mm->mutex".
    // modifies	 this
    // effects	 Adds mapping (segid->range).

    int container(int segid);
    // requires  Caller holds "mm->mutex".
    // effects	 Returns segment id for the segment that contains
    //		 the mapping for segid.

    int free_space();
    // requires  Caller holds "mm->mutex".
    // effects	 Return the number of bytes of free space on disk.
    //		 This free-space might be fragmented and therefore
    //		 compaction might be required to access all of
    //		 this space.

	 bool valid(int segid) {
      return (segid <= last) ? TRUE : FALSE;
    }


    void print(FILE *fp = NULL);
    // modifies: fp
    // effects:  Prints human readable contents on "fp"

  private:
/*
\subsection{Representation}

The segment table is represented as a two-level mapping structure built
out of segments.  The top-level entry is a segment with a well-known name
stored at a fixed location on disk.  The top-level entry contains disk
ranges of the second-level segments.  Each second-level segment stores
the disk ranges of other segments.
*/
    Disk_RootObj*   root;	// The root object
    Disk_LeafObj**  leaf;	// Array of leaf objects

// Information maintained for allocation without a linear search
    int		last;		// Last id allocated
    Uint	blk_max;	// Largest allocated block number

    bool alloc(int& s, Disk_Range& r, int n);
    // modifies	"leaf", "s", "r"
    // effects	Searches for free space for a new segment with "n" blocks.
    //		Sets "s" to index for the segment, "r" to its range.
    //		Modifies the appropriate leaf segment to contain
    //		the correct mapping.

    void recompute_allocation_info();
    // effects	Reconstruct allocation info from root and leaves.
};


// \subsection{Log Record for Segment Allocation}

class Seg_Alloc_Record : public Log_Record {
  public:
    Seg_Alloc_Record(int seg, Disk_Range range);
    // requires	"mm->mutex" is held by the caller

    Seg_Alloc_Record();
    // sets up the record so that the data can be added later

    virtual ~Seg_Alloc_Record();

    virtual int  size();

#if REPLICATION
    virtual int type();
#endif

    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
  private:
    int		segment;	// Segment number
    Disk_Range	range;		// Disk range assigned to segment
    long	stamp;		// Time stamp of modification to seg table
};



#endif /* _SEGTABLE_H */






