// Copyright 1995 Barbara Liskov

#include "common/compat.h"
#include "common/fail.h"
#include "common/fe_or_msg.h"
#include "common/hostname.h"
#include "common/network.h"
#include "common/or_obj.h"
#include "common/or_set.h"
#include "common/prefetch.h"
#include "common/ros.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/transaction.h"
#include "common/tstampgen.h"
#include "common/xrefs.h"
#include "common/uids.h"

#include "server.h"

// Error handler for OR connection.
static void err_handler(Device*, char const* msg) {
    fail(msg);
}

static const int OR_NUMBER = 1;

Server::Server(char const* spec) {
    struct sockaddr_in address;
    if (!findport(spec, OR_DEFAULT_FE_PORT, &address))
	fail("%s: invalid OR specifier", spec);

    int sock;
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	sysfail("opening socket");

    if (connect(sock, (struct sockaddr*) &address, sizeof(address)) < 0)
	sysfail("connecting to OR");

    or = new Network(sock);
    or->set_handler(err_handler);
    or->set_nodelay();

    id = new fe_num(gethostid(), getpid(), time(0), 0);
    id->encode(or);
    or->flush();
}

Server::~Server() {
    delete id;
    delete or;
}

Oref Server::root() {
    fe_message msg;
    msg.msgtype = FE_FETCHROOT;
    msg.encode(or);
    or->flush();

    // Wait for reply
    or_message reply;
    while (1) {
	reply.decode(or);
	if (reply.msgtype == OR_ROOT) break;
	skip_extra_data(reply);
    }

    return reply.u.root;
}

void Server::skip_extra_data(or_message const& msg) {
    // Read any extra data that follows the message header.
    or_objdesc* descs;
    OR_slot* slots;
    Xrefs xrefs;
    Uids uids;

    switch (msg.msgtype) {
      case OR_OBJECTS:
	read_objects(msg.u.objects.number, descs, slots);
	delete [] descs;
	delete [] slots;
	break;
      case OR_COMMITTED:
	// Read and discard the new object info.
	xrefs._enlarge_by(msg.u.commit.count);
	uids._enlarge_by(msg.u.commit.count);
	or->recv_buffer(xrefs.as_pointer(), sizeof(Xref)*msg.u.commit.count);
	or->recv_buffer(uids.as_pointer(), sizeof(OR_uid)*msg.u.commit.count);
	break;
      case OR_INVALIDATION:
	// XXX Cannot handle it
	fprintf(stderr, "cannot handle invalidation messages\n");
	exit(1);
	break;
      case OR_STAT:
	// XXX Cannot handle it
	fprintf(stderr, "cannot handle stat messages\n");
	break;
      default:
	// These should have no extra data
	break;
    }
}

void Server::read_objects(int num, or_objdesc*& descs, OR_slot*& slots) {
    //fprintf(stderr, "  %5d objects\n", num);
    descs = new or_objdesc[num];
    or->recv_buffer(descs, num * sizeof(or_objdesc));

    // Get number of objects slots that follow the descriptor array
    int count = 0;
    for (int i = 0; i < num; i++)
	count += descs[i].objsize;

    slots = new OR_slot[count];
    or->recv_buffer(slots, count * sizeof(OR_slot));
}

OR_obj* Server::fetch_one(Oref o, prefetch_hint* h) {
    fe_message msg;
    msg.msgtype = FE_FETCH;
    msg.u.fetch.o = o;
    msg.u.fetch.prefetch = *h;
    msg.encode(or);
    or->flush();

    // Wait for reply
    or_message reply;
    or_objdesc* descs;
    OR_slot* slots;
    int i, offset;
    OR_obj* result = 0;

    while (result == 0) {
	reply.decode(or);
	switch (reply.msgtype) {
	  case OR_FETCH_DENY:
	    if (OREF_EQUAL(reply.u.denied, o))
		return 0;
	    break;
	  case OR_OBJECTS:
	    read_objects(reply.u.objects.number, descs, slots);

	    // Search for requested object
	    offset = 0;
	    for (i = 0; i < reply.u.objects.number; i++) {
		int size = descs[i].objsize;
		//offset += DEFAULT_NULLSPACE;

		if (OREF_EQUAL(descs[i].o, o)) {
		    // Found it
		    result = (OR_obj*) new OR_slot[size];
		    memcpy(result, slots+offset, size * sizeof(OR_slot));
		    break;
		}
		offset += size;
	    }

	    delete [] descs;
	    delete [] slots;
	    break;
	  default:
	    skip_extra_data(reply);
	}
    }

    return result;
}

ObjectMap* Server::fetch(Oref o, prefetch_hint* h) {
    ObjectMap* result = new ObjectMap;
    or_message reply;
    or_objdesc* descs;
    OR_slot* slots;

    // Read pending message before sending request
    while (or->can_read()) {
	reply.decode(or);
	switch (reply.msgtype) {
	  case OR_OBJECTS:
	    read_objects(reply.u.objects.number, descs, slots);
	    parse_objects(reply.u.objects.number, descs, slots, result);
	    delete [] descs;
	    delete [] slots;
	    break;
	  default:
	    skip_extra_data(reply);
	    break;
	}
    }

    // Do not bother sending request if object has already arrived
    if (result->contains(o)) return result;

    // Send fetch request
    fe_message msg;
    msg.msgtype = FE_FETCH;
    msg.u.fetch.o = o;
    msg.u.fetch.prefetch = *h;
    msg.encode(or);
    or->flush();

    // Wait for reply
    while (!result->contains(o)) {
	reply.decode(or);
	switch (reply.msgtype) {
	  case OR_FETCH_DENY:
	    if (OREF_EQUAL(reply.u.denied, o)) return result;
	    break;
	  case OR_OBJECTS:
	    read_objects(reply.u.objects.number, descs, slots);
	    parse_objects(reply.u.objects.number, descs, slots, result);
	    delete [] descs;
	    delete [] slots;
	    break;
	  default:
	    skip_extra_data(reply);
	    break;
	}
    }

    return result;
}

void Server::parse_objects(int num, or_objdesc* descs, OR_slot* slots,
			   ObjectMap* result) {
    int offset = 0;
    for (int i = 0; i < num; i++) {
	int size = descs[i].objsize;
	//offset += DEFAULT_NULLSPACE;

	if (! result->contains(descs[i].o)) {
	    // Copy object into map
	    OR_obj* obj = (OR_obj*) new OR_slot[size];
	    memcpy(obj, slots+offset, size * sizeof(OR_slot));
	    result->store(descs[i].o, obj);
	}

	offset += size;
    }
}

void Server::commit(Ros* ros, Mos* mos, Nos* nos) {
    Ros mros(0);
    Mos mmos;
    Nos mnos;
    OR_set parts;

    if (ros == 0) ros = &mros;
    if (mos == 0) mos = &mmos;
    if (nos == 0) nos = &mnos;

    ros->sort();

    Transaction t;
    t.tid = Tid(*id, gen.generate());
    t.ros = ros;
    t.mos = mos;
    t.nos = nos;
    t.participants = &parts;
    t.participants->add(OR_NUMBER);

    fe_message msg;
    msg.msgtype = FE_PREPARE_COORD;
    msg.encode(or);
    t.encode(or);
    or->flush();

    while (1) {
	or_message reply;
	reply.decode(or);

	// XXX We do not do anything with returned data
	skip_extra_data(reply);

	switch (reply.msgtype) {
	  case OR_COMMITTED:
	    return;
	  case OR_STALEABORT:
	    fail("transaction accessed stale data");
	    break;
	  case OR_FAILABORT:
	    fail("transaction failed");
	    break;
	  default:
	    break;
	}
    }
}

implementOpenHashMap(ObjectMap,Oref,OR_obj*,OREF_HASH,OREF_EQUAL)
