// Copyright 1995 Barbara Liskov

// Traverse entire object graph at OR.

#include <stdio.h>
#include <stdlib.h>

#include "common/fail.h"
#include "common/or_obj.h"
#include "common/prefetch.h"
#include "common/transaction.h"
#include "common/xref.h"
#include "common/ros.h"
#include "common/mos.h"
#include "server.h"
#include "fetch.h"

// Proc called on each object
struct State;
typedef void (*Obj_Proc)(State*, Oref, OR_obj*);

// Traversal state
struct State {
    ObjectMap*	objects;

    // Transaction info
    Ros*	ros;
    Mos*	mos;
    orefs*	mod_orefs;

    // Summary state
    int slots;			// Non-header slots
};

static void usage() {
    fprintf(stderr,"traverse [-nrws]\n");
    fprintf(stderr, "   -n         ; do nothing to each object\n");
    fprintf(stderr, "   -r         ; read each object.  Commit at end\n");
    fprintf(stderr, "   -w         ; write each object.  Commit at end\n");
    fprintf(stderr, "   -s         ; print object summary\n");
    exit(1);
}

// Read proc
void read_proc(State* state, Oref o, OR_obj* obj) {
    state->ros->add_obj(o);
}

// Flip list element pointers
void flip_proc(State* state, Oref o, OR_obj* obj) {
    if (OR_OBJ_SIZE(obj) != 2) return;
    if (OR_OBJ_BITFIELD(obj) != 3) return;

    Xref tmp = obj->slot[0].xref;
    obj->slot[0].xref = obj->slot[1].xref;
    obj->slot[1].xref = tmp;

    state->mod_orefs->append(o);
}

void generate_mos(State* state) {
    // Sort by oref #
    int num = state->mod_orefs->size();
    sort_orefs(state->mod_orefs->as_pointer(), num);

    state->mos->clear();
    for (int i = 0; i < num; i++) {
	Oref o = state->mod_orefs->slot(i);
	OR_obj* obj = 0;
	if (! state->objects->fetch(o, obj)) continue;

	Obj_Handle h = state->mos->add_object(o, OR_OBJ_SIZE(obj));
	state->mos->init_object(h,
				OR_OBJ_CLASS(obj),
				OR_OBJ_BITFIELD(obj),
				OR_OBJ_SIZE(obj),
				obj->slot);
    }
}

// Summary proc
void summary_proc(State* state, Oref o, OR_obj* obj) {
    state->slots += OR_OBJ_SIZE(obj);
}

main(int argc, char** argv) {
    Obj_Proc p = 0;

    int opt;
    while ((opt = getopt(argc, argv, "nrws")) != EOF) {
	switch (opt) {
	  case 'n':
	    p = 0;
	    break;
	  case 'r':
	    p = read_proc;
	    break;
	  case 'w':
	    p = flip_proc;
	    break;
	  case 's':
	    p = summary_proc;
	    break;
	  default:
	    usage();
	    break;
	}
    }

    // Get all objects
    Server* server = new Server(getenv("THOR"));
    prefetch_hint* hint = new prefetch_hint;
    hint->max_prefetch = 32;

    State* state = new State;
    state->ros = new Ros(0);
    state->mos = new Mos;
    state->mod_orefs = new orefs;
    state->slots = 0;
    state->objects = fetch_all(server, server->root(), hint);

    // Process all objects
    if (p != 0) {
	for (ObjectMap::Bindings x = state->objects; x.ok(); x.next())
	    p(state, x.key(), x.val());

	// Commit the transaction
	generate_mos(state);
	if ((p == read_proc) || (p == flip_proc))
	    server->commit(state->ros, state->mos, 0);

	// Summary
	if (p == summary_proc) {
	    printf("objects: %6d\n", state->objects->size());
	    printf("headers: %6d\n", state->objects->size() * OR_obj_headers);
	    printf("data:    %6d\n", state->slots);
	}
    }

    // Clean out object storage
    for (ObjectMap::Bindings x = state->objects; x.ok(); x.next())
	delete [] ((OR_slot*) x.val());

    delete server;
    delete hint;
    delete state->objects;
    delete state->ros;
    delete state->mos;
    delete state->mod_orefs;
    delete state;
    return 0;
}
