// Copyright 1995 Barbara Liskov

// \section{Pthread-based Thread Implementation}

#include <pthread.h>
#include <errno.h>
#include <stdio.h>
#ifdef __linux__
#include <sys/time.h>
#define DST_NONE 0
#endif

#include "utils/fail.h"
#include "thread.h"
#include "utils/th_assert.h"

// Wrapper for thread creation.
static void* thread_wrapper(void*);

Thread::Thread() {
}

Thread::~Thread() {
}

void Thread::start(int prio) {
    pthread_t pt;

    pthread_attr_t attr;
    struct sched_param par;
    
    if (prio > 0) {
      th_assert(prio <= TH_MAX_PRIO && prio >=  TH_MIN_PRIO, "Invalid priority");
      pthread_attr_init(&attr);
      par.sched_priority = prio;
      if (pthread_attr_setschedparam(&attr, &par)) 
	th_fail("pthread_attr_setschedparam failed");
      if (pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED))
	th_fail("pthread_attr_setinheritsched failed");
    }


    // Try creating thread several times to deal with "EAGAIN"
    int count = 0;
    while (count < 5) {
	if (pthread_create(&pt, ((prio > 0) ? &attr : NULL), thread_wrapper,
			   (void*) this) == 0)
	    break;
	if (errno != EAGAIN) sysfail("pthread_create");
	count++;
    }
    if (count == 5) sysfail("pthread_create");

    if (pthread_detach(pt) != 0) {
	sysfail("pthread_detach");
    }
}

static void* thread_wrapper(void *arg) {
    Thread* thread = (Thread*) arg;

    thread->main();
    delete thread;
    return 0;
}

struct Mutex_Rep {
    pthread_mutex_t mutex;
};

Mutex::Mutex() {
    rep = new Mutex_Rep;
    if (pthread_mutex_init(&(rep->mutex), NULL) != 0) {
	sysfail("pthread_mutex_init");
    }
}

Mutex::~Mutex() {
    if (pthread_mutex_destroy(&(rep->mutex)) != 0) {
	sysfail("pthread_mutex_destroy");
    }
    delete rep;
}

void Mutex::grab() {
    if (pthread_mutex_lock(&(rep->mutex)) != 0) {
	sysfail("pthread_mutex_lock");
    }
}

void Mutex::release() {
    if (pthread_mutex_unlock(&(rep->mutex)) != 0) {
	sysfail("pthread_mutex_unlock");
    }
}

struct Condition_Rep {
    pthread_cond_t   cond;
    pthread_mutex_t* mutex;
};

Condition::Condition(Mutex* mutex) {
    rep = new Condition_Rep;
    if (pthread_cond_init(&(rep->cond), NULL) != 0) {
	sysfail("pthread_cond_init");
    }
    rep->mutex = &(mutex->rep->mutex);
}

Condition::~Condition() {
    if (pthread_cond_destroy(&(rep->cond)) != 0) {
	sysfail("pthread_cond_destroy");
    }
    delete rep;
}

void Condition::wait() {
    if (pthread_cond_wait(&(rep->cond), rep->mutex) != 0) {
	sysfail("pthread_cond_wait");
    }
}

bool Condition::timedwait(float seconds) {
    struct timespec delta, deadline;
    struct timeval now;
    struct timezone here;

    // Used to use
    //     pthread_get_expiration_np(&delta, &deadline);
    // but it's not portable.

    here.tz_minuteswest = 0;
    here.tz_dsttime = DST_NONE;
    gettimeofday(&now,&here);
    delta.tv_sec = time_t(seconds);
    delta.tv_nsec = long(seconds*1000*1000*1000) % (1000*1000*1000);
    deadline.tv_sec = now.tv_sec + delta.tv_sec;
    deadline.tv_nsec = (now.tv_usec * 1000) + delta.tv_nsec;

    if (pthread_cond_timedwait(&(rep->cond), rep->mutex, &deadline) != 0) {
        fprintf(stderr, "timedwait errno=%d\n", errno);
	if (errno == EAGAIN) return TRUE;       // timeout
	else sysfail("pthread_cond_timedwait"); // error
    }
    return FALSE;
}

void Condition::signal() {
    if (pthread_cond_signal(&(rep->cond)) != 0) {
	sysfail("pthread_cond_signal");
    }
}

void Condition::broadcast() {
    if (pthread_cond_broadcast(&(rep->cond)) != 0) {
	sysfail("pthread_cond_broadcast");
    }
}
