// Copyright 1995 Barbara Liskov

/*
\section{Transaction Manager}

The TM interface is used for transaction management.

\subsection{Todo}

\begin{itemize}
\item arguments to commit interface
\item What happens if fe connection dies while commit is in progress?
\item Destructor?
\item Everything ...
\end{itemize}
*/

#ifndef _TM_H
#define _TM_H

#include "common/xref.h"
#include "common/tid.h"
#include "common/modset.h"
#include "common/or_stat.h"
#include "mm/logindex.h"

class FE_manager;
class MM;
class Mutex;
class Transaction;
class Validate_queue;
class Log;
class Modset;
class Installer;
class Prepared_Log_Record;
class Coord_set;
class OR_set;

struct or_stat;

class TM {
  public:
    TM();
    // effects - create a transaction manager from configuration information

    ~TM ();
    // Destructor

    Ubits32 commit_local(FE_manager* fe, Transaction const* tx);
    // requires - Transaction involves only this OR. mutex_validation has
    //            been grabbed
    // returns  - The result of the commit (see common/or_message.h)
    // effects  - Commit a transaction tx obtained from FE fe locally (either
    //            single-OR transaction or read-only) and returns the outcome.  
    //            This call is allowed to return before the commit is 
    //            finished as long as the outcome message is delivered 
    //            to the fe eventually.
    //            If the fe connection dies while the commit is being
    //            done, the commit work still proceeds
    //            Adds appropriate records to log.

    void TM::commit_exclusive(Transaction* transaction);
    // requires - Transaction involves only this OR and requires no validation
    //            or invalidation, say, because only one thread ever reads
    //            or writes these objects. E.g., internal state managed by 
    //            the OR (root objects, seg table, GC objects, etc.) may be
    //            maintained this way.
    // effects    Sets transaction.tid and commits the transaction.

    Ubits32 prepare_coord(FE_manager* fe, Transaction const* tx);
    // requires - This OR is the coordinator for tx.
    //            mutex_validation has been grabbed
    // returns  - The result of the prepare (see common/or_message.h"
    //            Returns 0 if prepare was successful.
    // effects  - Performs validation of the multiple-OR transaction tx.
    //            Adds appropriate records to log.

    Ubits32 prepare_part(FE_manager* fe, Transaction const* tx, bool &force);
    // requires - This OR is not the coordinator for tx.
    //            mutex_validation has been grabbed
    // returns  - The result of the prepare (see common/or_message.h"
    // effects  - Performs validation of the multiple-OR transaction tx.
    //            Adds appropriate records to log.
    //            Sets force: true iff transaction prepared but prepare
    //            record was forced to disk (for Stamos optimization).

    void abort(Tid const& tid);
    // effects  - Abort given transaction.  If we are coordinator, send 
    //            abort messages to participants who have voted for 
    //            transaction to commit.
    //            Remove entry for transaction from or->tstatus.

    void commit(Tid const& tid);
    // requires - This OR is coordinator for tid.
    //            All participants have voted OK for transaction.
    // effects  - Commit given multiple-site transaction.

    void truncate_vqueue_if_needed();
    // effects: Remove "old" entries from the VQ and update threshold.

    void stat(OR_stat& s);
    // modifies	s
    // effects	Store the stats for the TM in "s".
    //          and reset the relevant timers for it.

  private:
    Global_tstamp  threshold;       // No trans is allowed below this ts.
    Global_tstamp  stable_threshold; 
    // The stable copy of threshold.  Will not need to modify now.
    // Invariant : read_threshold <= stable_threshold.
    Global_tstamp  to_install_stable_threshold;
    // This timestamp is the threshold value that has to be put on
    // stable storage. *After* this value has been flushed to the
    // log, the stable_threshold value is updated.
    // Invariant: stable_threshold <= to_install_stable_threshold

    Mutex *mutex_validation;        // Mutex during the whole validation
				    // process. 
    Validate_queue*     vq;         // Validation Queue.
    Coord_set*          coord_set;  // Set of coordinators used in Stamos'
				    // optimization.
    Tstamp              last_trunc; // Last time coordinator set was truncated

    Ubits32             xactions;   // Number of transactions committed so far.

    friend Installer;   // Let update thread use internal methods
    
    // Messages that are friends of the TM. They should not use
    // the private variables. Only the private methods should be used

    friend class FE_recv_commit_msg;


    // \subsection{Private methods}

    void commit_single_phase(FE_manager *fe, Transaction *tx, Ubits32 req_id);
    // requires: tx is a global read-only or a single site transaction
    // effects: Validates tx on fe's behalf and sends the approproate reply
    //          to fe using the request_id req_id


    void commit_multiple(FE_manager *fe, Transaction *tx, Ubits32 req_id);
    // requires: tx is a read-write multi-OR transaction
    // effects: Commits tx and sends the result to fe using request_id "req_id"

    void finish_transaction(FE_manager *fe, Tid const& tid, Ubits32 result,
			    Ubits32 reqid);
    // effects  - Send "result" of given transaction "tid" back to FE "fe"
    //            using the request id "reqid"
    //            For use only with multiple OR transactions.

    bool send_commit_result(FE_manager *fe, Ubits32 result, Ubits32 req_id);
    // effects: Sends the commit reply message to the FE given by fe
    //          using the request_id "req_id" Returns TRUE iff succeeds


    Ubits32 validate (FE_manager* fe, Transaction const* tx,
		      Global_tstamp& tx_tstamp);
    // requires    mutex_validation to have been grabbed
    // effects --- Validate the transaction and return the commit result (see
    //             common/or_message.h") 
    //             tx->mos and tx->nos have not been deleted.
  
    bool threshold_check (Global_tstamp& tstamp);
    // requires    mutex_validation to have been grabbed
    // effects --- Checks if tstamp has a timestamp greater than threshold.

    bool invalid_obj_check (FE_manager* fe, Transaction const* tx); 
    // requires    mutex_validation to have been grabbed
    // effects --- Checks if the transaction has read any stale objects.
    //             using the FE tables of fe.

    bool reserve_object_space (Transaction const* tx);
    // requires    mutex_validation to have been grabbed
    // effects --- Reserve space for all the MOS/NOS objects.
    // returns --- TRUE iff succeeds

    bool update_threshold(Global_tstamp &tstamp);
    // requires    mutex_validation to have been grabbed
    // effects     Sets threshold to maximum of current value and value of
    //             given timestamp.  If stable_threshold is updated, returns
    //             true; otherwise returns false.

    void set_stable_threshold(bool mutex_held);
    // requires  mutex_validation has been grabbed iff mutex_held is TRUE
    //           Called after the log has been flushed
    // effects   Sets the stable_threshold to the value that is on disk
    //           If mutex is not held, will grab it and then release it.
    //           Otherwise it will not relase the mutex

    void add_to_invalid_sets (FE_manager *fe, const Oref* wset, int wsize);
    // requires    mutex_validation to have been grabbed
    // modifies    or->fe_map
    // effects     Enters into the FE_tables of other FEs, all of
    //		   the objects in wset (wsize is the size of the array).

    void add_to_FE_table (FE_manager *fe, Modset *nos);
    // modifies    or->fe_map
    // effects     Enters the pages for nos objects that are not present in
    //             the FE to be complete (Assumes that the FE had allocation
    //             rights to the whole page)

    void truncate_coordinators();
    // requires    Caller holds validation mutex.
    // effects     Removes old coordinators from coordinator set for
    //             Stamos' optimization.
    
    void free_trans_space(Transaction const *tx);
    // effects: Frees the space assoicated with modified and newly persistent
    //          objects 

    // \subsection{Logging operations}
    // These operations create log records, add them to the log, and
    // flush the log if necessary.

    void mark_installed(Global_tstamp* ts);
    // requires: Transaction corresponding to ts exists
    //           hould not be called by an internal routine of TM
    // effects:  Marks the corresponding transaction as
    //           installed

    void log_commit_single(FE_manager* fe, Transaction const* tx);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction commit information for single OR trans.
    //		   and install any necessary changes.
    //             fe refers to the FE whose transaction is being committed
    //             Releases mutex_validation so that it is not held
    //             during logging.

    Log_Index log_prepared(Transaction const* tx, bool &force);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction prepare information at participant.
    //             Releases mutex_validation so that it is not held
    //             during logging.
    //             If force is true, record will be forced to disk.
    //             If force is false, force will be set to true if record
    //               is forced to disk.
    // returns --- Log index of prepare record.

    Log_Index log_committing(Transaction const* tx);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction prepare information at coordinator.
    //             Releases mutex_validation so that it is not held
    //             during logging.
    // returns --- Log index of committing record.

  public:
    void log_committed(Tid const&, OR_set* participants);
    // effects --- Log a committed record for given transaction.
    
    void log_abort(Tid const &tid);
    // effects --- Log an abort record for given transaction.

    void log_done(Tid const &tid);
    // effects --- Log a done record for given transaction.

    void log_participant(Tid const &tid, OR_num participant, 
			 Prepared_Log_Record *rec, Log_Index index);
    // effects --- Log participant's prepare record in coordinator log 
    //             protocol.
};

extern float total_time, recv_time, send_time, validate_time;
#include "utils/Timer.h"
extern Timer total_timer, recv_timer, send_timer, validate_timer;
extern Timer cumm_total_timer, cumm_recv_timer, cumm_send_timer, cumm_validate_timer;

#endif /* _TM_H */
