/*
\section{Address}

This file desribes an abstraction for addresses used to uniquely name
processes in a distributed system. These addresses are 64 bits long and
typically contain a host identifier and a uniquifier within the host.  In a
system like Thor, the host address is replaced by an OR number (for FEs it
is the OR that the FE first talks to).

             ****** This is an immutable stack abstraction ****** 
*/

#ifndef _ADDRESS_H
#define _ADDRESS_H

#include "basic.h"
#include "bits.h"
#include "hashfuncs.h"

#include <stdio.h>

class Device;

class Address {
  public:
    Address();
    // effects: Uninitialized fe number

    Address(Ubits32 host_id, Ubits32 uniquifier);
    // effects: Creates an address with the given hostid and uniquifier

    Address(Address const &a) : host_id(a.host_id), unique(a.unique) {}
    // effects: Copy constructor

    void operator=(Address const &addr) {host_id = addr.host_id; unique = addr.unique;}
    // Assigns addr to this.

    Ubits32 hostid() { return host_id;}
    // effects: Returns the host_id of this

    Ubits32 uniquifier() { return unique;}
    // effects: Returns the host_id of this

    bool operator==(Address const& a)  const;
    // effects: returns true iff this "=" a.

    bool operator<(Address const& a)  const;
    // effects: returns true iff this "<" a.

    bool operator>(Address const& a)  const;
    // effects: returns true iff this ">" a.
    
    int hash() const { return host_id ^ unique; }
    // effects: Returns a hashed value (Not using the host_id here. Should we?)

    bool encode(Device*) const;
    bool decode(Device*);

    void print(FILE *fp = NULL) const;

    static const Address Error_address;

  private:
    Ubits32 host_id;
    Ubits32 unique;
};

inline Address::Address() {
}

inline Address::Address(Ubits32 hid, Ubits32 uniquifier):host_id(hid),
    unique(uniquifier)  {}

inline bool Address::operator==(Address const& a)  const {
    return (host_id == a.host_id && unique == a.unique);
}

#endif /* _ADDRESS_H */
