/*
\section{Generic Resizable Array}

"array.h" provides a generic resizable array built out of preprocessor
macros.  These arrays can be stack allocated as long as you are careful
that they do not get copied excessively.  (One reason for stack allocation
of arrays is so that you can use the nice [] syntactic sugar.)

You can declare an array of "<Element>" and name the resulting
class "<ArrayClass>" by saying
\begin{verbatim}
    declareArray(<ArrayClass>, <Element>)
\end{verbatim}

You can also provide an implementation for a previously declared
array by saying
\begin{verbatim}
    implementArray(<ArrayClass>, <Element>)
\end{verbatim}

Example (an array of integers)
\begin{verbatim}
    declareArray(IntList, int)
    implementArray(IntList, int)
\end{verbatim}


Arrays grow and shrink at the high end.  The low end always has index
0.

The following specifications assume that you have delared an 	array
named "Array" with element types "T" by saying "declareArray(Array, T)".
*/

// \subsection{Constructor Specifications}

    /*
    Array()
	effects  - Creates empty array

    Array(int predict)
	requires - predict >= 0
	effects  - Creates empty array.  Extra storage is preallocated
		   under the assumption that the array will be grown
		   to contain predict elements.  This constructor form
		   helps avoids unnecessary copying.

    Array(T const* x, int count)
	requires - count >= 0 and x is a pointer to at least count elements.
	effects  - creates an array with a copy of the count elements pointed
		   to by x.  This constructor is useful for initializing an
		   array from a builtin array.

    Array(Array const& x)
	effects  - creates an array that is a copy of x.

    Array(T x, int count)
	requires - count >= 0
	effects  - creates an array with count copies of x.
    */

// \subsection{Destructor Specifications}

    /*
    ~Array()
	effects  - releases all storage for the array.
    */

// \subsection{Operation Specifications}

    /*
    Array& operator=(Array const& x)
	effects  - copies the contents of x into *this.

    T& operator[](int index) const
	requires - index >= 0 and index < size().
	effects  - returns a reference to the indexth element in *this.
		   Therefore, you can say things like --
			Array a;
			...
			a[i] = a[i+1];

    T& slot(int index) const
	requires - index >= 0 and index < size().
	effects  - returns a reference to the indexth element in *this.
		   This operation is identical to operator [].  It is
		   just more convenient to use with an Array*.

			Array* a;
			...
			a->slot(i) = a->slot(i+1);

    int  size() const
	effects	 - returns the number of elements in *this.

    T& high() const
	effects  - returns a reference to the last element in *this.

    void append(T v)
	modifies - *this
	effects  - grows array by one by adding v to the high end.

    void append(T v, int n)
	requires - n >= 0
	modifies - *this
	effects  - grows array by n by adding n copies of v to the high end.

    void concat(T const* x, int n)
	requires - n >= 0, x points to at least n Ts.
	modifies - *this
	effects	 - grows array by n by adding the n elements pointed to by x.

    void concat(Array const& x)
	modifies - *this
	effects  - append the contents of x to *this.

    T remove()
	requires - array is not empty
	modifies - *this
	effects  - removes last element and return a copy of it.

    void remove(int num)
	requires - num >= 0 and array has at least num elements
	modifies - *this
	effects  - removes the last num elements.

    void clear()
	modifies - *this
	effects  - removes all elements from *this.

    void reclaim()
	effects  - reclaim unused storage.

    T* as_pointer() const
	requires - returned value is not used across changes in array size
		   or calls to reclaim.
	effects  - returns a pointer to the first element in the array.
		   The returned pointer is useful for interacting with
		   code that manipulates builtin arrays of T.

    void predict(int new_alloc);
         effects - Does not change the abstract state. If the allocated 
                   storage is smaller than new_alloc elements, enlarges the
                   storage to new_alloc elements.
 
    void _enlarge_by(int n);
	requires - n >= 0
	effects  - appends "n" UNITIALIZATED entries to the array.
		   This is an unsafe operation that is mostly useful
		   when reading the contents of an array over the net.
		   Use it carefully.

    */

#ifndef _ARRAYH
#define _ARRAYH

#include "th_assert.h"

// \subsection{Array Class}
#define _declareArrayClass(ArrayType,Element)				      \
class ArrayType {							      \
  public:								      \
									      \
    /* Constructors */							      \
    ArrayType();			/* Empty array */		      \
    ArrayType(int predict);		/* Empty array with size predict */   \
    ArrayType(Element const*, int);	/* Initialized with C array */	      \
    ArrayType(ArrayType const&);	/* Initialized with another Array */  \
    ArrayType(Element, int);		/* Fill with n copies of element */   \
									      \
    /* Destructor */							      \
    ~ArrayType();							      \
									      \
    /* Assignment operator */						      \
    ArrayType& operator=(ArrayType const&);				      \
									      \
    /* Array indexing */						      \
    Element& operator[](int index) const;				      \
    Element& slot(int index) const;					      \
									      \
    /* Other Array operations */					      \
    int  size() const;			/* Return size; */		      \
    Element& high() const;		/* Return last element */	      \
    Element* as_pointer() const;	/* Return as pointer to base */	      \
    void append(Element v);		/* append an element */		      \
    void append(Element, int n);	/* Append n copies of element */      \
    void concat(Element const*, int);	/* Concatenate C array */	      \
    void concat(ArrayType const&);	/* Concatenate another Array */	      \
    Element remove();			/* Remove and return last element */  \
    void remove(int num);		/* Remove last num elements */	      \
    void clear();			/* Remove all elements */	      \
    void predict(int new_alloc);        /* Increase allocation */	      \
									      \
									      \
    /* Storage stuff */							      \
    void reclaim();			/* Reclaim all unused space */	      \
    void _enlarge_by(int n);		/* Enlarge array by n */	      \
  private:								      \
    Element*	store_;			/* Actual storage */		      \
    int		alloc_;			/* Size of allocated storage */	      \
    int		size_;			/* Size of used storage */	      \
									      \
    /* Storage enlargers */						      \
    void enlarge_allocation_to(int s);	/* Enlarge to s */		      \
    void enlarge_to(int s);		/* Enlarge to s if necessary */	      \
};									      \


// \subsection{Inline Operations}
#define _declareArrayInlines(ArrayType,Element)				      \
									      \
inline ArrayType::ArrayType() {						      \
    alloc_ = 0;								      \
    size_  = 0;								      \
    store_ = 0;								      \
}									      \
									      \
inline int ArrayType::size() const {					      \
    return size_;							      \
}									      \
									      \
inline Element& ArrayType::operator[](int index) const {		      \
    th_assert((index >= 0) && (index < size_), "array index out of bounds");  \
    return store_[index];						      \
}									      \
									      \
inline Element& ArrayType::slot(int index) const {			      \
    th_assert((index >= 0) && (index < size_), "array index out of bounds");  \
    return store_[index];						      \
}									      \
									      \
inline Element& ArrayType::high() const {				      \
    th_assert(size_ > 0, "array index out of bounds");			      \
    return store_[size_-1];						      \
}									      \
									      \
inline Element* ArrayType::as_pointer() const {				      \
    return store_;							      \
}									      \
									      \
inline void ArrayType::append(Element v) {				      \
    if (size_ >= alloc_)						      \
	enlarge_allocation_to(size_+1);					      \
    store_[size_++] = v;						      \
}									      \
									      \
inline Element ArrayType::remove() {					      \
    if (size_ > 0) size_--;						      \
    return store_[size_];						      \
}									      \
									      \
inline void ArrayType::remove(int num) {				      \
    th_assert((num >= 0) && (num <= size_), "invalid array remove count");    \
    size_ -= num;							      \
}									      \
									      \
inline void ArrayType::clear() {					      \
    size_ = 0;								      \
}									      \
									      \
inline void ArrayType::_enlarge_by(int n) {				      \
    th_assert(n >= 0, "negative count supplied to array operation");	      \
    enlarge_to(size_ + n);						      \
}									      \


// \subsection{Non-Inline Operations}
#define _declareArrayNonInlines(ArrayType,Element)			      \
									      \
ArrayType::ArrayType(int predict) {					      \
    th_assert(predict >= 0, "negative count supplied to array operation");    \
    alloc_ = 0;								      \
    size_ = 0;								      \
    store_ = 0;								      \
    enlarge_to(predict);						      \
    size_ = 0;								      \
}									      \
									      \
ArrayType::~ArrayType() {						      \
    if (alloc_ > 0) delete [] store_;					      \
}									      \
									      \
ArrayType::ArrayType(Element const* src, int s) {			      \
    th_assert(s >= 0, "negative count supplied to array operation");	      \
    alloc_ = 0;								      \
    size_  = 0;								      \
    store_ = 0;								      \
    enlarge_to(s);							      \
    for (int i = 0; i < s; i++)						      \
	store_[i] = src[i];						      \
}									      \
									      \
ArrayType::ArrayType(ArrayType const& d) {				      \
    alloc_ = 0;								      \
    size_  = 0;								      \
    store_ = 0;								      \
    enlarge_to(d.size_);						      \
    for (int i = 0; i < size_; i++)					      \
	store_[i] = d.store_[i];					      \
}									      \
									      \
ArrayType::ArrayType(Element element, int num) {			      \
    th_assert(num >= 0, "negative count supplied to array operation");	      \
    alloc_ = 0;								      \
    size_ = 0;								      \
    store_ = 0;								      \
    enlarge_to(num);							      \
    for (int i = 0; i < num; i++)					      \
	store_[i] = element;						      \
}									      \
									      \
ArrayType& ArrayType::operator=(ArrayType const& d) {			      \
    size_ = 0;								      \
    enlarge_to(d.size_);						      \
    for (int i = 0; i < size_; i++)					      \
	store_[i] = d.store_[i];					      \
    return (*this);							      \
}									      \
									      \
void ArrayType::append(Element element, int n) {			      \
    th_assert(n >= 0, "negative count supplied to array operation");	      \
    int oldsize = size_;						      \
    enlarge_to(size_ + n);						      \
    for (int i = 0; i < n; i++)						      \
	store_[i + oldsize] = element;					      \
}									      \
									      \
void ArrayType::concat(ArrayType const& d) {				      \
    int oldsize = size_;						      \
    enlarge_to(size_ + d.size_);					      \
    for (int i = 0; i < d.size_; i++)					      \
	store_[i+oldsize] = d.store_[i];				      \
}									      \
									      \
void ArrayType::concat(Element const* src, int s) {			      \
    th_assert(s >= 0, "negative count supplied to array operation");	      \
    int oldsize = size_;						      \
    enlarge_to(s + size_);						      \
    for (int i = 0; i < s; i++)						      \
	store_[i+oldsize] = src[i];					      \
}									      \
									      \
void ArrayType::predict(int new_alloc) {				      \
    if (new_alloc > alloc_)						      \
        enlarge_allocation_to(new_alloc);				      \
}									      \
 									      \
void ArrayType::enlarge_to(int newsize) {				      \
    if (newsize > alloc_)						      \
	enlarge_allocation_to(newsize);					      \
   size_ = newsize;							      \
}									      \
									      \
void ArrayType::enlarge_allocation_to(int newsize) {			      \
    int newalloc = alloc_ * 2;						      \
    if (newsize > newalloc) newalloc = newsize;				      \
									      \
    Element* oldstore = store_;						      \
    store_ = new Element[newalloc];					      \
									      \
    for (int i = 0; i < size_; i++)					      \
	store_[i] = oldstore[i];					      \
									      \
    if (alloc_ > 0) delete [] oldstore;					      \
    alloc_ = newalloc;							      \
}									      \
									      \
void ArrayType::reclaim() {						      \
    if (alloc_ > size_) {						      \
	/* Some free entries that can be reclaimed */			      \
	if (size_ > 0) {						      \
	    /* Array not empty - create new store */			      \
	    Element* newstore = new Element[size_];			      \
	    for (int i = 0; i < size_; i++)				      \
		newstore[i] = store_[i];				      \
	    delete [] store_;						      \
	    alloc_ = size_;						      \
	    store_ = newstore;						      \
	}								      \
	else {								      \
	    /* Array empty - delete old store */			      \
	    if (alloc_ > 0) {						      \
		delete [] store_;					      \
		alloc_ = 0;						      \
	    }								      \
	}								      \
    }									      \
}									      \



#define declareArray(ArrayType,Element)					      \
_declareArrayClass(ArrayType,Element)					      \
_declareArrayInlines(ArrayType,Element)					      \


#define implementArray(ArrayType,Element)				      \
_declareArrayNonInlines(ArrayType,Element)				      \


#endif /*_ARRAYH */
