// It should be possible to include this file multiple times.

// Bitmaps are the stack-abstraction equivalent of bool_vector.
// A bitmap does not involve any heap allocation.
// Its use is suitable when the size of the bitmap is a relatively small 
// compile-time constant.

// To instantiate a Bitmap of say 32 bits called Bitmap32, do the following
// #define BITMAP_CLASS Bitmap32
// #define BITMAP_SIZE 32
// #include "bitmap.h"
//
// IMPORTANT: the value of BITMAP_SIZE must be a valid constant-expression for the
//   cpp #if directive. It cannot contain names of C++ constants.
//         

// XXX is_all_true, is_all_false are broken or say that all bitmap sizes
// must be 32-bit aligned. num_true assumes that

#ifndef BITMAP_CLASS
#error BITMAP_CLASS not defined
#endif
#ifndef BITMAP_SIZE
#error BITMAP_SIZE not defined
#endif

#include "basic.h"
#include "bits.h"
#include "th_assert.h"
#include "network.h"

typedef unsigned int ChunkInt; 
typedef unsigned long ChunkLong;

#define Ntrue_bits 4
#define Ntrue_size (1 << (Ntrue_bits))
#define Ntrue_mask ((Ntrue_size) - 1)

extern Ubits8 num_true_arr[]; // No. of 1s in ith number from 0 to Ntrue_size -1

// Chunk is the unit of manipulation in the implementation
#if BITMAP_SIZE > INT_BITS

#if BITMAP_SIZE > LONG_BITS
#define Chunk ChunkInt
#else
#define Chunk ChunkLong
#endif

#else 
#define Chunk ChunkInt
#endif

// No. of bits in Chunk
#define Chunk_bits (sizeof(Chunk)*byte_bits)


class BITMAP_CLASS {

  public:
    BITMAP_CLASS(bool value=false);
    /* Creates a bitmap with all booleans to "value" */

    void clear();
    /* Clears the bitmap. */

    bool test(unsigned int i) const;
    /* requires: "i" is within bounds.      */
    /* effects:  returns ith boolean.       */

    int size() const;
    /* effects:  returns the size of the bitmap */

    void assign(unsigned int i, bool value);
    /* requires: "i" is within bounds. value is true or false      */
    /* effects:  Sets ith boolean to val.  */

    void set(unsigned int i);
    /* requires: "i" is within bounds.      */
    /* effects:  Sets ith boolean to true.  */
    // Note: this is faster than "assign"ing it to true.

    void reset(unsigned int i);
    /* requires: "i" is within bounds.      */
    /* effects:  Sets ith boolean to false. */

    void set_range(Uint min, Uint max);
    // requires: min is within bounds, max is within bounds+1.
    // effects: sets all bits in [min, max[.  

    void reset_range(Uint min, Uint max);
    // requires: min is within bounds, max is within bounds+1.
    // effects: resets all bits in [min, max[.  

    int num_true() const;
    // effects: Returns the number of bits in this that are TRUE */

    bool is_all_false() const;
    /* effects: returns true of all bits are false */

    bool is_all_true() const;
    /* effects: returns true of all bits are true */

    BITMAP_CLASS operator| (BITMAP_CLASS other) const;
    // Returns the boolean OR of this and other.

    BITMAP_CLASS operator& (BITMAP_CLASS other) const;
    // Returns the boolean AND of this and other.

    BITMAP_CLASS operator~ () const;
    // Returns the boolean NOT of this.

    bool operator== (BITMAP_CLASS const other) const;
    // Returns true if this and other are equal, false if not.

    void print (FILE *fp = NULL);
    // effects: Prints a human readable of"this" on fp

    bool encode(Network *net);
    /* Sends this to the net. Returns true iff successful.*/

    bool decode(Network *net);
    /* Reads this from the net. Returns true iff successful. */

    class Iter {
	/* An iterator for yielding the booleans in a bool bitmap
	   Once created, an iterator must be used before any
	   changes are made to the iterated object. The effect is undefined
	   if an iterator method is called after such a change.
	 */

      public:
	Iter(BITMAP_CLASS *bitmap, bool val);
	/* Creates iterator to yield indices and values of the bitmap in order.
	   Only those booleans set to value are yielded.
	   */

	bool get(unsigned int& index);
	/* modifies: index
	   effects: Sets "index" to the next index that has the value
                    specified in the constructor
	            Returns false iff there is no such index.
        */
      private:
	BITMAP_CLASS *bitmap;  /* The bitmap being yielded. */
	bool  value;           /* The value being searched. */
	unsigned int index;    /* Index of next boolean to be tested */
    };

  private:
    friend Iter;

#if BITMAP_SIZE > LONG_BITS
    Chunk chunks[(BITMAP_SIZE+Chunk_bits-1)/Chunk_bits]; /* Chunk of bits */
#else
    Chunk chunk;
#endif

    inline int num_true_chunk(Chunk chunk) const;
    // effects: Returns the number of entries true in the chunk

    inline Chunk bitselector(unsigned int i) const;
    /* requires: "Chunk_bits" is a power of 2.
       effects:  Computes the position p of the ith boolean within its
                 chunk and returns a chunk with pth bit set
    */

    inline Chunk val_bitselector(unsigned int i, bool val) const;
    /* requires: "Chunk_bits" is a power of 2. val is true or false
       effects:  Computes the position of the ith boolean within its
       chunk, say p, and returns a chunk with ith bit set to val
                 and other bits to 0
    */
};


inline int BITMAP_CLASS::num_true_chunk(Chunk chunk) const {
    int const num_subchunks = sizeof(chunk)*8/Ntrue_bits; // XXX
    int result = 0;
    for (int i = 0; i < num_subchunks; i++) {
	int subchunk = chunk & Ntrue_mask;
	result += num_true_arr[subchunk];
	chunk = chunk >> Ntrue_bits;
    }
    return result;
}

#if BITMAP_SIZE > LONG_BITS

inline BITMAP_CLASS::BITMAP_CLASS(bool value) {
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    for (int i = 0; i < num_chunks; i++) {
	chunks[i] = value? (~0) : 0;
    }
}


inline Chunk BITMAP_CLASS::bitselector(unsigned int i) const{
    int p = i & (Chunk_bits - 1);
    return 1 << p;
}


inline Chunk BITMAP_CLASS::val_bitselector(unsigned int i, bool val) const{
    int p = i & (Chunk_bits - 1);
    return val << p;
}

inline void BITMAP_CLASS::clear() {
  int const num_chunks = sizeof(chunks)/sizeof(Chunk);
  for (int i = 0; i < num_chunks; i++) chunks[i] = 0;
}
    
inline bool BITMAP_CLASS::test(unsigned int i) const {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    return (chunks[i/Chunk_bits] & bitselector(i)) != 0;
}


inline int BITMAP_CLASS::size() const {
    return BITMAP_SIZE;
}


inline int BITMAP_CLASS::num_true() const {
    int const num_chunks = sizeof(chunks)/sizeof(Chunk); // XXX
    int result = 0;
    for (int i = 0; i < num_chunks; i++)
	result += num_true_chunk(chunks[i]);
    return result;
}


inline void BITMAP_CLASS::set(unsigned int i) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunks[i/Chunk_bits] |= bitselector(i);
}


inline void BITMAP_CLASS::assign(unsigned int i, bool val) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunks[i/Chunk_bits] &= (~ bitselector(i));
    chunks[i/Chunk_bits] |= val_bitselector(i, val);
}


inline void BITMAP_CLASS::reset(unsigned int i) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunks[i/Chunk_bits] &= (~ bitselector(i));
}

inline void BITMAP_CLASS::set_range(Uint min, Uint max) {
  for (Uint i = min; i < max; i++) set(i);  
}

inline void BITMAP_CLASS::reset_range(Uint min, Uint max) {
  for (Uint i = min; i < max; i++) reset(i);  
}

inline bool BITMAP_CLASS::is_all_false() const {
    int extra = BITMAP_SIZE%sizeof(Chunk);
    int num_chunks = sizeof(chunks)/sizeof(Chunk);
    if (extra) num_chunks--;
		
    int i;	      
    for (i = 0; i < num_chunks; i++) {
	if (chunks[i] != 0)
	  return (false);
    }
    if (extra) {
      if ((chunks[i+1] & ((1 << extra) - 1)) != (Chunk)0)
	return false;
    }
    return (true);
}

inline bool BITMAP_CLASS::is_all_true() const{
    int extra = BITMAP_SIZE%sizeof(Chunk);
    int num_chunks = sizeof(chunks)/sizeof(Chunk);
    if (extra) num_chunks--;

    int i;
    for (i = 0; i < num_chunks; i++) {
      if (chunks[i] != (Chunk)(~0))
	return (false);
    }
    if (extra) {
      if ((chunks[i+1] | ~((1 << extra) - 1)) != ~((Chunk)0))
	return false;
    }
    return (true);
}

inline void  BITMAP_CLASS::print(FILE *fp) {
    if (!fp) fp = stderr;
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    for (int i = 0; i < num_chunks; i++) {
	fprintf(fp, "%x", chunks[i]);
    }
}


inline BITMAP_CLASS BITMAP_CLASS::operator| (BITMAP_CLASS other) const {
    BITMAP_CLASS result;
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    for (int i = 0; i < num_chunks; i++)
	result.chunks[i] = chunks[i] | other.chunks[i];
    return result;	
}

inline BITMAP_CLASS BITMAP_CLASS::operator& (BITMAP_CLASS other) const {
    BITMAP_CLASS result;
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    for (int i = 0; i < num_chunks; i++)
	result.chunks[i] = chunks[i] & other.chunks[i];
    return result;	
}

inline BITMAP_CLASS BITMAP_CLASS::operator~ () const {
    BITMAP_CLASS result;
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    for (int i = 0; i < num_chunks; i++)
        result.chunks[i] = ~chunks[i];
    return result;	
}

inline bool BITMAP_CLASS::operator== (BITMAP_CLASS const other) const {
    int const num_chunks = sizeof(chunks)/sizeof(Chunk); // XXX
    for (int i = 0; i < num_chunks; i++) 
        if (chunks[i] != other.chunks[i]) return (false);
    return (true);
}

inline bool BITMAP_CLASS::encode(Network *net) {
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    return net->send_buffer(chunks, sizeof(Chunk)*num_chunks, true);
}


inline bool BITMAP_CLASS::decode(Network *net) {
    int const num_chunks = sizeof(chunks)/sizeof(Chunk);
    return net->recv_buffer(chunks, sizeof(Chunk)*num_chunks);
}

#else

inline BITMAP_CLASS::BITMAP_CLASS(bool value) { 
  th_assert(BITMAP_SIZE <= LONG_BITS, 
	    "Preprocessor could not translate BITMAP_SIZE into a constant\n");
  chunk = value ? ~((Chunk)0) : (Chunk)0; 
}

inline Chunk BITMAP_CLASS::bitselector(unsigned int i) const { 
  return ((Chunk)1UL) << i;
}

inline Chunk BITMAP_CLASS::val_bitselector(unsigned int i, bool val) const {
  return ((Chunk)val) << i;
}

inline int BITMAP_CLASS::num_true() const {
    return num_true_chunk(chunk);
}

inline void BITMAP_CLASS::clear() { chunk = (Chunk)0; }
    
inline bool BITMAP_CLASS::test(unsigned int i) const {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    return (chunk & bitselector(i)) != 0;
}


inline int BITMAP_CLASS::size() const {
    return BITMAP_SIZE;
}


inline void BITMAP_CLASS::set(unsigned int i) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunk |= bitselector(i);
}

inline void BITMAP_CLASS::set_range(Uint min, Uint max) {
  Chunk mask = ~((1UL << min) - 1) & ((1UL << max) - 1);
  chunk |= mask; 
}

inline void BITMAP_CLASS::reset_range(Uint min, Uint max) {
  Chunk mask = ~(((1UL << max) - 1) & ~((1UL << min) - 1));
  chunk &= mask; 
}

inline void BITMAP_CLASS::assign(unsigned int i, bool val) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunk &= (~ bitselector(i));
    chunk |= val_bitselector(i, val);
}


inline void BITMAP_CLASS::reset(unsigned int i) {
    th_assert(i < BITMAP_SIZE, "Index out of bounds\n");
    chunk &= (~ bitselector(i));
}

inline bool BITMAP_CLASS::is_all_false() const { 
  Chunk tmp = chunk & ((1 << BITMAP_SIZE) - 1);
  return tmp == (Chunk)0; }

inline bool BITMAP_CLASS::is_all_true() const { 
  Chunk tmp = chunk | ~((1 << BITMAP_SIZE) - 1);
  return tmp == ~((Chunk)0); 
}

inline void  BITMAP_CLASS::print(FILE *fp) {
  if (!fp) fp = stderr;
  fprintf(fp, "%x", (int)chunk);
}


inline BITMAP_CLASS BITMAP_CLASS::operator| (BITMAP_CLASS other) const {
  BITMAP_CLASS result;
  result.chunk = chunk | other.chunk;
  return result;
}

inline BITMAP_CLASS BITMAP_CLASS::operator& (BITMAP_CLASS other) const {
  BITMAP_CLASS result;
  result.chunk = chunk & other.chunk;
  return result;
}

inline BITMAP_CLASS BITMAP_CLASS::operator~ () const {
  BITMAP_CLASS result;
  result.chunk = ~chunk;
  return result;
}

inline bool BITMAP_CLASS::operator== (BITMAP_CLASS const other) const {
    return chunk == other.chunk;
}

inline bool BITMAP_CLASS::encode(Network *net) {
    return net->send_buffer(&chunk, sizeof(Chunk), true);
}


inline bool BITMAP_CLASS::decode(Network *net) {
    return net->recv_buffer(&chunk, sizeof(Chunk));
}

#endif

inline BITMAP_CLASS::Iter::Iter(BITMAP_CLASS *bitmap_, bool value_):
    bitmap(bitmap_), value(value_), index(0) {}

inline bool BITMAP_CLASS::Iter::get(unsigned int& ind) {
    while (index < BITMAP_SIZE) {
	bool this_value = bitmap->test(index);
	if (this_value == value) {
	    ind = index++;
	    return true;
	}
	index++;
    }
    return false;
}

#undef BITMAP_CLASS
#undef BITMAP_SIZE
#undef Chunk_bits
#undef Chunk
