#ifndef _COMMUNICATION_H
#define _COMMUNICATION_H

// Overview: This abstraction stores handlers for various message types
// as well as specific request messages.
//
// The type of a received message identifies whether it is synchronous,
// i.e., should be expected at the receiver.
// If a synchronous message is received that has no handler registered 
// for its id, its skip method is invoked.
// For other messages received, their decode method is invoked.

#include "basic.h"
#include "map.h"
#include "valuekey.h"
#include "bits.h"

#include "message.h"
#include "send_message.h"
#include "recv_message.h"

#include "address.h"

// Message ids and message types are in the same name space. The ids from 0 to
// Max_msg_type are reserved for message types

typedef Ubits32    Request_id;
typedef Ubits32    Message_type;
class   Network_set;
const   Request_id Null_request_id      = 0;
const   Ubits32 Max_msg_type    = 1000;

// Structure that keeps track of the received messages info
struct Recv_msg_info {
    Recv_msg_info() {}
    Recv_msg_info(Recv_message *rm): rmsg(rm), to_delete(FALSE) {}
    Recv_message* rmsg; // The message handler 
    bool to_delete;  // Whether r must be deleted by the comm. abstraction
};

typedef Map<UIntKey, Recv_msg_info> Handler_map;
typedef Map<Address, Network *> Address_map;

//   Messages are sent and received using the following functions.
class Communication {
  public:

    Communication();
    // Creates an initialized object.

    ~Communication();

    void include(Network *net);
    // Includes "net" in the set of connected networks.

    bool send(Address a, Send_message *sm, 
	      Recv_message *rm, Request_id& id);
    // Modifies id.
    // If the request id is 0, generates a new one and assigns it to id.
    // Uses id to send the message "sm" to address "a".
    // Registers "rm" as the handler for message received with given id.
    // If "rm" is NULL, no receive handler is installed. 
    // Returns TRUE iff succeeds in sending message.

    int handle_message(float timeout, bool &unblock);
    // modifies: unblock
    // Blocks until timeout to receive one message from the connected networks.
    // (A negative timeout implies infinite timeout.)
    // Executes decode/skip method for the message as described above.
    // Returns -1 if there was a message but failed to handle it,
    // 0 if no message was found, and if  a message was successfully handled,
    // returns its id
    // Sets unblock to TRUE, unless there is a handler for the msgid received
    // and the handler's decode message indicates that caller should be blocked

    bool handled(Request_id id);
    // Requires that id is a valid request id allocated earlier.
    // Returns whether the response to given request id has been handled.

    int handle_messages_until_done(Request_id id);
    // Requires that request "id" was sent earlier with an associated handler
    // or the "id" is Null.
    //
    // Handles messages from the connected networks until the response 
    // for the request is handled. (Skips this if id is Null.)
    // All messages available without wait after this are also handled. 
    // Returns the number of messages handled, -1 if handling of any failed. 

    void send_recv(Address a, Send_message *sm, Recv_message *rm);
    // Execute "send" followed by "handle_messages_until_done" using above.
    // Fail if either returns error.

    void unregister_handler(Request_id id);
    // Unregisters the message handler associated with id.
    // (Nothing happens if no such handler exist.)

    void register_handler(Recv_message *handler);
    // Registers "handler" as the handler for asynchronous messages
    // of its type. 

  private:
    Network_set *nets;  // Networks that are used. There is redundancy here
    // with Addr_map since a Network_set is needed for a select call
    Address_map *amap;  // Maps addresses to networks
    Handler_map *hmap;  // Table to map msg types/ids to handlers.
    Ubits32 next_id;    // Next request id to be issued on a send
};

#endif /* _COMMUNICATION_H */
