#include <errno.h>
#include <string.h>
#include "device.h"
#include "fail.h"
#include "th_assert.h"

// Standard error handler.
static void default_error_handler(Device*, char const* msg) {
    fail("device stream: %s\n", msg);
}

Device::Device(int bufsize) {
    handler = &default_error_handler;
    had_error = FALSE;

    outbuf = new char[bufsize];
    outbuf_size = bufsize;
    outbuf_used = 0;
}

Device::~Device() {
    delete [] outbuf;
}

// Control operations

Device::Error_Handler Device::set_handler(Device::Error_Handler h) {
    Error_Handler old = handler;
    handler = h;
    return old;
}

bool Device::ok() const {
    return (! had_error);
}

void Device::shutdown() {
    flush();
    had_error = TRUE;
}

void Device::error(int err) {
    had_error = TRUE;
    handler(this, strerror(err));
}

// Output operations

bool Device::flush() {
    if (outbuf_used > 0) {
	bool result = write_bytes(outbuf, outbuf_used);
	outbuf_used = 0;
	return result;
    }
    else {
	return TRUE;
    }
}

bool Device::flush(void const* buf, int size) {
    if (outbuf_used > 0) {
	// write outbuf and buf using one write_vector
	iovec iov[2];
	iov[0].iov_base = outbuf;
	iov[0].iov_len = outbuf_used;
	outbuf_used = 0;
	iov[1].iov_base = (char *) buf;
	iov[1].iov_len = size;
	return write_vector(iov, 2);
    }
    return write_bytes(buf, size);
}

bool Device::send_buffer(void const* ptr, int size, bool buffer) {
    if (buffer && size <= outbuf_size - outbuf_used) {
	memcpy(outbuf+outbuf_used, ptr, size);
	outbuf_used += size;
	return TRUE;
    }
    return flush(ptr, size);
}

bool Device::send_ubits32(Ubits32 val, bool buffer) {
    if (buffer && 4 <= outbuf_size - outbuf_used) {
	memcpy(outbuf+outbuf_used, (char *)&val, 4); 
	// Caution: outbuf+outbuf_used may not be 4byte aligned 
	outbuf_used += 4;
	return TRUE;
    }
    return flush(&val, 4);
}

bool Device::send_vector(struct iovec* vec, int count, bool) {
    // The current implementation never buffers this
    return (flush() && write_vector(vec, count));
}

bool Device::force() {
    return(TRUE);
  }

// Input operations

bool Device::recv_buffer(void* ptr, int size) {
    return (read_bytes(ptr, size));
}

bool Device::recv_vector(struct iovec* vec, int count) {
    return (read_vector(vec, count));
}

bool Device::recv_ubits32(Ubits32* ptr) {
    return (read_bytes(ptr, 4));
}
