#include <errno.h>
#include <string.h>

#include "fail.h"
#include "fdevice.h"
#include "th_assert.h"

FDevice::FDevice(int f, int bufsize) : Device(bufsize) {
    fd = f;
}

FDevice::~FDevice() {
    shutdown();
}

void FDevice::shutdown() {
    flush();
    ::close(fd);
    Device::shutdown();
}

void FDevice::error(int err) {
    ::close(fd);
    Device::error(err);
}

bool FDevice::write_bytes(void const* ptr, int size) {
    char const* buffer = (char const*) ptr;

    while (size > 0) {
	// XXX - Some prototypes are messed up and think write() modifies buf
	int count = write(fd, (char*)buffer, size);

	if (count < 0) {
	    error(errno);
	    return FALSE;
	}

	// XXX - Can we really get a zero from write()?
	assert(count != 0);

	assert(count <= size);
	size -= count;
	buffer += count;
    }

    return TRUE;
}

bool FDevice::write_vector(struct iovec* vec, int count) {
    assert(count >= 0);

    while (count > 0) {
	// writev may write less than what we asked for...
	// It also does not like more than UIO_MAXIOV entries at a time.
	int entries = (count > UIO_MAXIOV) ? UIO_MAXIOV : count;
	int result = writev(fd, (struct iovec*) vec, entries);
	if (result < 0) {
	    error(errno);
	    return FALSE;
	}

	advance(vec, count, result);
    }

    return TRUE;
}

bool FDevice::read_bytes(void* ptr, int size) {
    char* buffer = (char*) ptr;

    while (size > 0) {
	int count = read(fd, buffer, size);
	if (count == 0) {
	    // EOF
	    return FALSE;
	}
	if (count < 0) {
	    error(errno);
	    return FALSE;
	}
	assert(count <= size);
	size -= count;
	buffer += count;
    }

    return TRUE;
} 

bool FDevice::read_vector(struct iovec* vec, int count) {
    th_assert(count >= 0, "Read_vector called with negative array size");

    // readv() blocks on empty vectors in Linux 2.4.16
    // do this check to prevent hanging when all vectors have zero length
    // (from Rodrigo)
        int all_zeroes = 1;
        for (int i = 0; i<count; i++) {
          if (vec[i].iov_len != 0) {
            all_zeroes = 0;
            break;
          }
        }

        if (all_zeroes)
          return TRUE;


// #ifdef __linux__
// #define MAX_BYTES_IN_READV 8*1024
//     // Count the number of bytes that have been requested
//     int initial_count = count;
//     struct iovec* initial_vec = vec;
//     int i = 0;
//     while (i < initial_count) {
// 	// Put less than MAX_BYTES_IN_READV bytes in each call
// 	int total_bytes = 0;
// 	int cur_i = i;
// 	vec = &initial_vec[i];
// 	for (; i < initial_count; i++) {
// 	    if (initial_vec[i].iov_len > MAX_BYTES_IN_READV)
// 		th_fail("Cannot handle this byte size on linux");
// 	    if (total_bytes >= MAX_BYTES_IN_READV) {
// 		break;
// 	    }
// 	    total_bytes += initial_vec[i].iov_len;
// 	}
// 	count = i - cur_i;
// #endif

	while (count > 0) {
	    // Readv may read less than what we asked for...
	    // It also does not like more than UIO_MAXIOV entries at a time.
	    int entries = (count > UIO_MAXIOV) ? UIO_MAXIOV : count;
	    int result = readv(fd, vec, entries);

	    if (result < 0) {
		error(errno);
		return FALSE;
	    }
	    advance(vec, count, result);
	}
// #ifdef __linux__
//     }
// #endif
    return TRUE;
}

void FDevice::advance(struct iovec*& vec, int& count, int num) {
    // Update the vector to indicate that "num" bytes have been processed.
    while (count > 0) {
	if ((signed)vec[0].iov_len > num) {
	    // vec[0] was only partially filled in
	    vec[0].iov_len -= num;
	    (char*)vec[0].iov_base += num;
	    break;
	}

	// vec[0] was completely filled in
	num -= vec[0].iov_len;
	(signed)vec[0].iov_len = -1;
	vec[0].iov_base = 0;
	vec++;
	count--;
    }

    if (count == 0) {
	th_assert(num == 0, "more bytes processed than indicated in iovec");
    }
}
