// \section{Device built on File Descriptor}
//
// This is an interface to a device built on a file descriptor.

#ifndef _FDEVICE_H
#define _FDEVICE_H

#include "device.h"

class FDevice : public Device {
  public:
    FDevice(int fd, int bufsize=1024);
    // requires - "fd" is a file descriptor valid for reading and writing.
    // effects  - Create a transmission channel for the connection
    //		  inherent in sock.
    //            Sets the byte size of output buffer to bufsize.

    virtual ~FDevice();
    // effects  - Closes the descriptor.

// \subsection{Overridden Device Operations}
    virtual void shutdown();
    virtual void error(int err);

// \subsection{Implementations of Abstract Device Operations}
    virtual bool write_bytes(void const* ptr, int size);
    virtual bool write_vector(struct iovec* vector, int count);
    virtual bool read_bytes(void* ptr, int size);
    virtual bool read_vector(struct iovec* vec, int count);

  protected:
    int descriptor();
    // effects	- Return underlying descriptor.

    void advance(struct iovec*& vec, int& count, int number_of_bytes);
    // requires	- count >= 0
    // modifies	- vec, vec[0..count-1], count
    // effects	- Modifies vec/vec[0..count-1]/count to indicate that
    //		  the first "number_of_bytes" specifed by the vector have
    //		  been handled.  On return, vec/count can be passed again to
    //		  readv/writev to continue processing the IO operation.

// \subsection{The Rep}
    int fd;				// Underlying connection

// \subsection{Rep Invariant}
//
// \begin{itemize}
// * "fd" is a file-descriptor.
// * "fd" is open iff "!ok()".
// \end{itemize}
};

inline int FDevice::descriptor() {
    return fd;
}

#endif /* _FDEVICE_H */
