#include <errno.h>
#include <string.h>
#include <fcntl.h>

#include "th_assert.h"
#include "fail.h"
#include "network.h"
#include "other_unix.h"

extern "C" {
#include <netinet/tcp.h>
}

Network::Network(Address a, Ubits32 host_addr, int port, 
		 bool& success, int bufsize): 
    FDevice(-1, bufsize), address(a) {
    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	perror("opening stream socket");
	return;
    }
    struct sockaddr_in address;
    address.sin_family = AF_INET;
    address.sin_addr.s_addr =  host_addr;
    address.sin_port = htons(port);

    if (connect(fd, (struct sockaddr*) &address, sizeof(address)) < 0) {
	perror("Performing connect call");
	return;
    }
    success = true;
}


Network::Network(Address a, int sock, int bufsize):
    FDevice(sock, bufsize), address(a) {
}

Network::~Network() {
}

void Network::set_nodelay() {
    int val = 1;
    if (0 > setsockopt(descriptor(), IPPROTO_TCP, TCP_NODELAY, 
		       (char*)&val, sizeof(val))) {
	error(errno);
    }
}


void Network::set_buffsizes(int send_bufsize, int recv_bufsize) {
    if (setsockopt(descriptor(), SOL_SOCKET, SO_SNDBUF, (char *) &send_bufsize,
		   sizeof(send_bufsize)) < 0) {
        perror("setsockopt :");
	exit(-1);
    }

    if (setsockopt(descriptor(), SOL_SOCKET, SO_RCVBUF, (char *) &recv_bufsize,
		   sizeof(recv_bufsize)) == -1) {
	perror("setsockopt :");
	exit(-1);
    }
}


void Network::allow_sigio() {
    int fd = descriptor();
    fcntl(fd, F_SETOWN, getpid());
    fcntl(fd, F_SETFL, FASYNC);
}

bool Network::has_buffered_data() {
    return FALSE;
}

bool Network::can_read() {
    // Check for data by making a "select(2)" call with a zero timeout.

    int result = wait_or_timeout(0);
    // Report all errors except for interrupts
    if (result < 0)
	error(errno);

    return (result > 0);
}

bool Network::sloppy_read_bytes(void* ptr, int size, int* count) {
    // Just do one read of up to size bytes, fill in count of bytes
    // actually read.  Count is meaningless unless TRUE is returned.

    char* buffer = (char*) ptr;

    int cnt = read(descriptor(), buffer, size);
    if (cnt == 0) {
        // EOF
        return FALSE;
    }
    if (cnt < 0) {
        error(errno);
        return FALSE;
    }
    *count = cnt;
    return TRUE;
} 

int Network::wait_or_timeout(float sec) {

    struct timeval tmval;
    tmval.tv_sec = (int)sec;
    tmval.tv_usec = ((unsigned long)sec*1000*1000) % (1000*1000);
    
    fd_set readfs;     
    FD_ZERO (&readfs);
    int fd = descriptor();
    FD_SET(fd, &readfs);

    int retval = 0;
    while (retval <= 0) {
	if (!ok()) return -1;
	retval = select (fd+1, &readfs, NULL, NULL, &tmval);
	if (retval < 0 && errno == EINTR) continue; // signal received
	if (retval <= 0) return retval;
    }
    th_assert(FD_ISSET(fd, &readfs), "Connection did not time out but "
	      "fd of connection not set after select call");
    return retval;
}

int Network::get_descriptor() {
    return descriptor();
}
