#include <errno.h>
#include "array.h"
#include "hostname.h"
#include "network_set.h"

implementArray(Networks, Network*)

extern int errno;

Network_set::Network_set() {
    net_list = new Networks;
    remove_all();
}

Network_set::~Network_set() {
    delete net_list;
}

void Network_set::remove_all() {
    net_list->clear();
    FD_ZERO(&fds);
    FD_ZERO(&read_fds);
    maxfd = 0;
    nfds = 0;
}

void Network_set::remove(Network *net) {
    int fd = net->get_descriptor();

    th_assert(FD_ISSET(fd, &fds), "No-existent network being removed");
    th_assert(net_list->slot(fd) == net, "Network fd may been changed");

    net_list->slot(fd) = NULL;
    FD_CLR(fd, &fds);
    nfds--;
    if (maxfd != fd) return;

    // The largest fd is being removed. Find highest fd that is in the set now 
    maxfd = 0;
    for (int i = fd-1; i >= 0; i--) {
	if (FD_ISSET(i, &fds)) {
	    maxfd = i;
	    break;
	}
    }
    net_list->remove(fd - maxfd);
}

void Network_set::insert(Network *net) {
    int fd = net->get_descriptor();

    bool found = FD_ISSET(fd, &fds);
    if (found) return;

    FD_SET(fd, &fds);
    nfds++;
    if (maxfd < fd) maxfd = fd;

    // insert in net_list
    if (fd >= net_list->size())
	net_list->append(NULL, fd + 1 - net_list->size());
    net_list->slot(fd) = net;
}

Network_set::Iter::Iter(Network_set *net_set_, float timeout) {
    net_set = net_set_;
    yield_fdsp = &net_set->read_fds; // set bits for nets with data here
    fd = -1;

    struct timeval tmval, *tmvalp;
    tmval.tv_sec = (unsigned int)timeout;
    tmval.tv_usec = ((unsigned long)timeout*1000*1000) % (1000*1000);
    tmvalp = (timeout < 0) ? 0: &tmval;

    FD_ZERO(yield_fdsp);
    memcpy(yield_fdsp, &net_set->fds, sizeof(net_set->fds)); // expensive!

    int ready = 0;

    // First check whether any buffered data is available.
    for (int fd = 0; fd <= net_set->maxfd; fd++) {
	    if (!FD_ISSET(fd, &(net_set->fds))) continue;
	    Network *net = net_set->net_list->slot(fd);
	    if (net->has_buffered_data()) {
		FD_SET(fd, yield_fdsp);
		ready++;
	    }
    }
    if (ready > 0) return;

    while (1) {
	// Try the select call
	ready = select (net_set->maxfd+1, yield_fdsp, NULL, NULL, tmvalp);
	if (ready < 0) {
	    if (errno == EINTR) continue; // signal received
	    th_assert(true, "select failed");
	}
	break;
    }

}


