/* Copyright (c) 1996 Andrew C. Myers */

#ifndef _SET_H
#define _SET_H

#include "basic.h"
#include "generator.h"
#include "bhash.h"
#include "buckets.h"

#include "th_assert.h"
#include <stddef.h>

/*
   \section{Sets}

    This section defines an easy-to-use implementation of
    a mutable set type, based on the same underlying hash-table
    technology as the maps in "map.h". Elements in the set are
    never equivalent to each other.

    "Set<ELEM>": an efficient implementation using chained buckets.
        Has the weakness that each bucket is long-word aligned, possibly
        wasting space. It also generates new code for every instantiation.

    "PtrSet<ELEM>": an efficient implementation of Set<ELEM *>, where
        element equivalence is defined by pointer equality.
    
    The interface is the following:

    int size() const;
      // Returns the number of (k, v) pairs in the table

    void add(ELEM const &);
      // Checks: the table does not already contain a mapping for the key.
      // Effects: Adds a new mapping to the hash table.

    bool find(ELEM &e) const;
      // Place the contained element equivalent to "e" in "e",
      //    and return true. Return false if no such element exists.

    bool contains(ELEM const &e) const;
      // Return whether there is an element equivalent to "e".

    bool add1(ELEM const &);
      // same as "add" except allowAutoResize() is called before store

    bool remove(ELEM &e);
      // If "e" is in the set, return true and place the
      // equivalent element into "e". Otherwise, return false.

    void clear();
      //   modifies - this
      //    effects - All bindings are removed from this.

    void predict(int size);
      //    Indicate that you would like the hash table to contain "size"
      //    buckets. This is useful for obtaining optimal
      //    performance when many new mappings are being added at once. It
      //    may be an expensive operation, though it has no semantic effect.
      //    For best performance, "size" should be at least as large as the
      //    number of elements. This operation is O(size()).
        
    void allowAutoResize();
      //    Allow the hash table to rehash its mappings so that optimal
      //    performance is achieved. This operation is O(size()).
      //    The hash table will not rehash the mappings if it is
      //    at a reasonable size already.

    float estimateEfficiency() const;
      //    Return the average expected number of buckets in a bucket chain.
      //    If this number is "n", the expected number of buckets that will be
      //    looked at when a "fetch" or "remove" operation is performed is
      //    roughly "1 + n/2". Efficiency should be 1 for maximal speed.

    float estimateClumping() const;
      //    Evaluate how well the current hash function is working. This number
      //    should be close to one if the hash function is working well. If
      //    the clumping is less than 1.0, the hash function is doing better
      //    than a perfect hash function would be expected to, by magically
      //    avoiding collisions. A clumping of K means that the hash function
      //    is doing as badly as a hash function that only generates a
      //    fraction 1/K of the hash indices.

    int memory_usage() const;
      //    Returns number of bytes used by this data structure.

   To iterate over the elements in a set, use a generator: a data
   structure that successively produces each of the keys of the map.
   A generator is used in the following fashion:

	Set<ELEM> s;
	...
        SetGenerator<ELEM> g(s);
        ELEM e;
        while (g->get(e)) { ... }

   The "remove" method of a "SetGenerator" may be used to remove the
   current.
*/

/*
    The "Set" class is a easy-to-use set of ELEM.
    It supports all the operations described above.
    See "bhash.h" for implementation details.
*/
#define _superclass BHash<ELEM, Buckets<ELEM>, BucketsGenerator<ELEM> >
template<class ELEM> class SetGenerator;

template <class ELEM> class Set : private _superclass {
public:
    friend class SetGenerator<ELEM>;
    Set(int sizehint) : _superclass(sizehint) {}
    Set() : _superclass()        {}
    int size() const             {  return _superclass::size(); }
    void add(ELEM const &e)    	 { _superclass::add(e); }
    void add1(ELEM const &e)   	 { allowAutoResize();
				   _superclass::add(e); }
    bool find(ELEM &e) const 	 {  ELEM *e2 = _superclass::find(e);
				    if (!e2) return false;
				    e = *e2;
				    return true; }
    bool contains(ELEM const &e) const {  return 0 !=_superclass::find(e); }
    bool remove(ELEM &e) 	 {  return _superclass::remove(e); }
    void clear() 		 {  _superclass::clear(); }
    void predict(int size) {  _superclass::predict(size); }
    void allowAutoResize() {  _superclass::allowAutoResize(); }
    float estimateEfficiency() const
				 {  return _superclass::estimateEfficiency(); }
    float estimateClumping() const
    				 {  return _superclass::estimateClumping(); }
    int memory_usage() {return _superclass::memory_usage();}
};

#define BHG BHashGenerator<ELEM, Buckets<ELEM>, BucketsGenerator<ELEM> >

template<class ELEM>
class SetGenerator: public Generator<ELEM>, private BHG {
public:
    SetGenerator(Set<ELEM> const &s) :
	BHG((BHash<ELEM, Buckets<ELEM>, BucketsGenerator<ELEM> > &)s) {}
    virtual bool get(ELEM &e) { return BHG::get(e); }
    virtual void remove() { BHG::remove(); }
};

#undef BHG
#undef HP
#undef _superclass
/*
    The class "ELEM" must conform to the following declaration:

class ELEM {
    // An ELEM must be a hashable value. Often, an ELEM is a wrapper class
    // that provides an inlined "hash" implementation. Ideally, the hash
    // function does not lose information about the value that is being
    // hashed: a good implementation of a hash function for an integer key
    // would be the identity function.

    ELEM();				  // default constructor
    ELEM(ELEM const &);                   // Elements can be copied
    ~ELEM();                              // Elements can be destroyed. 
    operator=(ELEM const &);              // Elements can be overwritten
    bool similar(ELEM const &);		  // The equivalence relation
    int hash() const;                     // Returns a hash key for this value 
      
};

   See "valuekey.h" for a macro that creates new element classes out of
   primitive types such as int and long.

   A "PtrElem" is a useful class that provides a key for any pointer type.
   T can be any type.
*/

template <class T>
class PtrElem {
public:
    PtrElem() : val(0) {}
    PtrElem(T *a) : val(a) {}
    void operator=(PtrElem<T> const &x) { val = x.val; }
    int hash() const { return int(*((ptrdiff_t *)&val) >> 2); }
    bool similar(PtrElem<T> const &x)
        { return (x.val == val) ? true : false; } 
    T * val;
};

/*
    The "PtrSet" class is a easy-to-use set of ELEM *, with additional
    performance advantages.
*/
#define _superclass BHash<PtrElem<void>, Buckets<PtrElem<void> >, \
			BucketsGenerator<PtrElem<void> > >
#define E PtrElem<void>((void *)e)

template <class ELEM> class PtrSet : private _superclass {
public:
    PtrSet() : _superclass(0)    {}
    PtrSet(int sizehint)         : _superclass(sizehint) {}
    int size() const             {  return _superclass::size(); }
    void add(ELEM *e)            { _superclass::add(E); }
    bool contains(ELEM *e) const {  return 0 != _superclass::find(E); }
    bool remove(ELEM *e)         {  PtrElem<void> tmp = e; return _superclass::remove(tmp); }
    void clear()                 { _superclass::clear(); }
    void predict(int size)       { _superclass::predict(size); }
    void allowAutoResize()       { _superclass::allowAutoResize(); }
    float estimateEfficiency() const
				 {  return _superclass::estimateEfficiency(); }
    float estimateClumping() const
    				 {  return _superclass::estimateClumping(); }
};
#undef _superclass
#undef E

#endif /* _SET_H */
