#ifndef _Stack_refs_h
#define _Stack_refs_h 1

#include "bool_vector.h"

class Stack_refs {
  // Overview: Keeps track of granules that may be pointed to
  //   by stack or registers. All suitably aligned pointer-sized 
  //   words in the stack or registers are conservatively
  //   assumed to be pointers. The size of the granules must be a power of 2.

public: 
  Stack_refs(void *st, unsigned no, unsigned sz);
  // Requires: "sz" is a power of 2 
  // Effects: Creates a Stack_refs object to keep track of potential
  //   pointers in the stack (or registers) to elements of the array 
  //   that starts at "st" and has "no" elements of "sz" bytes.

  ~Stack_refs();
  // Effects: Destroys this and all associated storage.

  void compute(void **stack_bottom);
  // Effects: Updates this to reflect the current state of the stack and 
  //   registers

  bool is_referenced(unsigned index) const;
  // Effects: Returns true iff the granule "start[index]" was potentially 
  //   referenced by the stack (or registers) the last time "compute" 
  //   was called.

  void reset(unsigned index);

  static void compute(Stack_refs **list, Uint size, void **stack_bottom);
  // Requires: "list" is an array of pointers to Stack_refs 
  //   objects with at least "size" elements. 
  // Effects: Same as calling compute on the first "size" elements of "list".
  //   (This is significantly more efficient than separately calling compute
  //   on each element.) 


private:
  void *start, *end;    // Boundaries of region of interest (start <= end)
  int shift_bits;       // Amount by which value should be shifted to emulate
                        // division by the size of the elements in the region
                        // of interest (the size of these elements must be 
                        // a power of 2)      
 
  Bool_vector ref_map;  // Bitmap with a bit for each element in the region
                        // of interest.
 
  // Invariant: A bit in ref_map is set iff the corresponding element 
  //   was potentially referenced by the stack (or registers) the last 
  //   time compute was called.
};

inline bool Stack_refs::is_referenced(unsigned index) const {
  return ref_map.test(index);
}


inline  void Stack_refs::reset(unsigned index) {
  ref_map.reset(index);
}
#endif // _Stack_refs_h


