#include <stdlib.h>
#include <stdio.h>
#include "utils/bool_vector.h"

#define BITMAP_CLASS Bmap7
#define BITMAP_SIZE 7
#include "utils/bitmap.h"

#define BITMAP_CLASS Bmap32
#define BITMAP_SIZE 32
#include "utils/bitmap.h"

#define BITMAP_CLASS Bmap63
#define BITMAP_SIZE 63
#include "utils/bitmap.h"

#define BITMAP_CLASS Bmap65
#define BITMAP_SIZE 65
#include "utils/bitmap.h"

void test_assert(bool val, char const *str) {
    // effects: If val is TRUE, does nothing, else prints str  with a newline
    if (!val)
	fprintf(stderr, "%s\n", str);
}

// Macro for testing that the Bmap bitmap is correctly implemented 

#begin define RUN_TEST(Bmap, IS_HEAP)

static bool Bmap##_compare(Bmap a, Bmap b) {
    // effects: Returns TRUE if both a and b are the same

    int bsize = a.size();
    int result = TRUE;
    for (int i = 0; i < bsize; i++) {
	result = result && a.test(i) == b.test(i);
    }
    return result;
}

static bool Bmap##_check_all_val(Bmap bmap, bool val) {
    // effects: Checks that all values in bmap are val. If so returns TRUE,
    //          else returns FALSE

    // This routine actually tests the iterator and the test method also
    bool result = TRUE;
    Uint bsize = bmap.size();
    int count = 0;
    for (Uint i = 0; i < bsize; i++) {
	if (bmap.test(i) != val) {
	    count++;
	    result = FALSE;
	}
    }

    if (result) {
	// The following iterator should result in all indexes being yielded
	Bmap::Iter iter1(&bmap, val);
	Uint index = 0;
	Uint next_index = 0;
	while (iter1.get(index)) {
	    test_assert(index == next_index, "All vals not same in iter");
	    next_index = index + 1;
	}
	test_assert(next_index == bsize, "After iteration, index is not at end");
    }

    // Check that there are "count" number of !val values
    Bmap::Iter iter2(&bmap, !val);
    Uint index = 0;
    int icount = 0;
    while (iter2.get(index)) icount++;
    test_assert(icount == count, "Different number of !val in iter and for");
    return result;
}

static void Bmap##_run_test() {

    // Some sample bitmaps

    Bmap a(TRUE);
    Bmap b(FALSE);

    int size = a.size();
    test_assert(Bmap##_check_all_val(a, TRUE), "All values not TRUE in a");
    test_assert(Bmap##_check_all_val(b, FALSE), "All values not FALSE in b");
    
    // Check assignment. Inner element. Boundary: 0 and size-1
    a.assign(size/2, FALSE);
    test_assert(!Bmap##_check_all_val(a, TRUE), "All values are TRUE in a");
    test_assert(!a.test(size/2), "Size/2 entry in a is not FALSE");
    b.assign(0, TRUE);
    test_assert(!Bmap##_check_all_val(b, FALSE), "All values are FALSE in b");
    test_assert(b.test(0), "0 entry in b is not TRUE");
    a.assign(size/2, TRUE); a.assign(size-1, FALSE);
    test_assert(!Bmap##_check_all_val(a, TRUE), "All values are TRUE in a");
    test_assert(!a.test(size-1), "size-1  entry in a is not FALSE");

    // Check set and reset. Set/reset 0, size-1 entry
    a.set(size-1);
    test_assert(a.test(size-1), "size-1  entry in a is FALSE");
    a.reset(size/4);
    test_assert(!a.test(size/4), "size/4  entry in a is TRUE");
    a.set(size/4);
    b.set(0);
    test_assert(b.test(0), "0  entry in b is FALSE");
    b.reset(0);
    test_assert(!b.test(0), "0  entry in b is TRUE");
    test_assert(Bmap##_check_all_val(b, FALSE), "All values are not FALSE in b");
    test_assert(Bmap##_check_all_val(a, TRUE), "All values are not TRUE in a");
    
    // Iter and test are checked in check_all_val quite well!

    // Operators |, &, and ~
    Bmap c = a;
    Bmap d = b;
    Bmap e;
    
    e = c & d;
    test_assert(Bmap##_check_all_val(e, FALSE), "All values are not FALSE in e");
    e = c | d;
    test_assert(Bmap##_check_all_val(e, TRUE), "All values are not TRUE in e");
    c.set(size/3);
    e = a & c;
    test_assert(Bmap##_compare(e, c), "Bitmaps a and c are not same");
    e = b | c;
    test_assert(Bmap##_compare(e, c), "Bitmaps a and c are not same");

    e = ~c;
    test_assert(Bmap##_check_all_val(e, FALSE), "All values are not FALSE in e");
    e = ~d;
    test_assert(Bmap##_check_all_val(e, TRUE), "All values are not TRUE in e");
}
#end // of RUN_TEST

// Instantiate the run_test for various sizes
RUN_TEST(Bmap7, 0)
RUN_TEST(Bmap32, 0)
RUN_TEST(Bmap63, 0)
RUN_TEST(Bmap65, 0)

int main() {
    fprintf(stderr, "Testing bitmaps ...");
    Bmap7_run_test();
    Bmap32_run_test();
    Bmap63_run_test();
    Bmap65_run_test();
    fprintf(stderr, " done\n");
}
