/*
\section{Timestamp}

The timestamp abstraction is used to generate local timestamps.  These
timestamps have to be augmented with a globlly unique address to
make them unique.  _This is a stack abstraction._
*/

#ifndef _TSTAMP_H
#define _TSTAMP_H

extern "C" {
#include <sys/time.h>
}

#include <stdio.h>

#include "basic.h"
#include "bits.h"
#include "th_assert.h"

const Ubits64 Microsecs =  1000000; // Each second has this many microseconds.

class Device;

// \subsection{The Interface}
class Tstamp {
  public:
    Tstamp() {}
    // effects - Uninitialized timestamp.

    Tstamp(bool generate);
    // effects - if generate is TRUE, returns a new timestamp that is larger
    //           than all timestamps generated before. Else returns a null
    //           timestamp
  
    Tstamp (Tstamp const& t) { usecs = t.usecs;}
    // Copy constructor.

    Ubits32 coarse_time() { return usecs/Microsecs;}
    // effects: Returns "coarse" information about the time stored in this

    void operator=(Tstamp const& t) { usecs = t.usecs;}
    // Assignment operator. Copies the contents of t into this.

    bool operator<(Tstamp const& t)  const { return usecs < t.usecs;}
    // effects --- returns true iff this "<" t.
    
    bool operator==(Tstamp const& t)  const { return usecs == t.usecs;}
    // effects --- returns true iff this "==" t.
    
    bool operator!=(Tstamp const& t)  const { return usecs != t.usecs;}
    // effects --- returns true iff this "!=" t.

    bool operator>(Tstamp const& t) const  { return usecs > t.usecs;}
    // effects --- returns true iff this ">" t.

    Tstamp operator+(Bits32 increment) const;
    // requires: this + increment is not < 0
    // effects --- incrments the timestamp by "increment" microseconds.

// \subsection{Device Transmission Operations}
    bool encode(Device* net) const;
    bool decode(Device*);

    void print (FILE *fp = NULL);

/*
\subsection{Representation}
*/    
  private:
    // Representation
    Ubits64 usecs;	// No. of microseconds elapsed since the epoch

    static Ubits64 last_issued; // Keeps track of the last timestamp
    // that was issued by this abstraction
};


inline Tstamp Tstamp::operator+(Bits32 increment) const {
    Tstamp res; 
    res.usecs = usecs + increment;
    return res;
}


#endif /* _TSTAMP_H */

