#ifndef _MODSET_H
#define _MODSET_H

// Abstraction for a set of OR_objects. Used by the NOS and MOS.
// This is a set of objects maintained for every participant in a transaction
// commit 

#include <stdio.h>
#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/intarray.h"
#include "oref.h"
#include "orefs.h"
#include "or_obj.h"

class Device;
class OR_obj;
typedef int Obj_handle;

class Modset {
  public:
    Modset();		// Create empty modified object set
    ~Modset();		// Delete all storage

    // The following are common operations that may be used at either
    // the FE or the OR.

    int  rep_size() const;		// Size of encoded rep
    bool encode(Device* dev) const;
    bool decode(Device* dev);
    void print(bool whole_print, FILE* fp = NULL);
    // effects: Prints the modset. whole_print indicates if the whole
    // object has to be displatyed

    // The following operations are called at the FE when building
    // up information about a committing transaction.

    Obj_handle add_object(Oref oref, OR_obj* obj, int num_slots,
			  Oref class_oref);
    // requires: obj has num_slots number of fields
    // effects:	Allocate space for an OR object with num_slots fields and
    //          copies those slots from "obj". It assigns the class oref of
    //          the object to be "class_oref" and its oref to be
    //          "oref". Returns a handle for the allocated object that can be
    //          used to call set_oref later object.

    void add_surrogate(Oref oref, OR_num dest_or, Oref dest_oref);
    // effects: Adds a surrogate at "oref" for destination <dest_or,dest_oref>,
    // i.e. the surrogate at "oref" contains the value <dest_or,dest_oref>

    void set_oref(Obj_handle h, int slot_number, Oref oref);
     // requires: "h" is a handle for an object allocated in this modset.
     //           "slot_number" is a legal index into the data slots for "h".
     // effects:   Stores "oref" into the slot named by "slot_number".
     //            This routine is used during unswizzling.

    // Generate all elements of the modified object set.
    class Elements {
      public:
        Elements(Modset* set);
        // requires     set is not modified for the lifetime of generator.
        // effects      Generate each element of set.  Each element is
	//		generated exactly once in the order it was
	//		inserted

	bool get(OR_obj*& o, Oref& oref, int& num_slots);
	// modifies this, o, oref, size
	// effects  If more objects are available, modify o to
	//          contain the pointer to the next object, oref to be its
	//          oref and num_slots to contain the number of slots in o
	//          (excluding the headers), and return TRUE.  Else do not
	//          modify o, oref, size and return FALSE.
      private:
	int count;            // Number of object being yielded
	OR_slot_array* data;  // The actual contents of the Modset
	int  size;            // Size of data
	int  index;	      // Next object starts here.
	Oref *orefs;          // Orefs of the objects
	int  *sizes;          // Sizes of the objects
    };

    int count() const;		// Return number of objects stored here
    int obj_sizes() const;	// Total object sizes (in bytes)

    const Oref* refs() {return orefs->as_pointer(); }
    // effects: Returns the list of the orefs in the Modset
    //          The length of this array is modset->size()

    void clear();		// Remove all information
  private:
    friend Elements;

    // Modset should never be copied or assigned. So these the copy constructor
    // and assignment operator are being declared here as private but are not
    // implemented 

    Modset(const Modset& r);
    Modset& operator=(const Modset& r);

    int            num;		// Number of objects
    OR_slot_array  *buffer;	// Collected slots
    Orefs          *orefs;      // Orefs of objects in buffer
    IntArray       *sizes;      // Sizes of objects in buffer
    // Sizes and orefs are kept separately so that orefs can be returned
    // Modset::refs. ith object in orefs corresponds to ith object in sizes
};

inline int Modset::count() const {
    return num;
}

inline int Modset::obj_sizes() const {
    return (buffer->size() * sizeof(OR_slot));
}

inline Modset::Elements::Elements(Modset* set) {
    data = set->buffer;
    index = 0;
    size = data->size();
    sizes = set->sizes->as_pointer();
    orefs = set->orefs->as_pointer();
    count = 0;
}

inline bool Modset::Elements::get(OR_obj*& obj, Oref& oref, int& num_slots) {
    if (index >= size) return FALSE;

    obj       = (OR_obj*) &(data->slot(index));
    oref      = orefs[count];
    num_slots = sizes[count] - OR_obj_headers;

    index += OR_obj_headers + num_slots;
    count++;
    return TRUE;
}

#endif /* _MODSET_H */
