/*
\section{Bitfield Descriptor for Objects}

There are two types of object descriptors: special and non-special.
Non-special bitfields look like this:

-----------------------------
| special : 1 = 0 | bf : 31 |
-----------------------------


The special bitfield descriptor format assumes that the sequence of
slots in an object is divided in two:

1. A prefix of at most 24 slots where reference and data slots can be
intermixed.

2. The rest of the sequence of slots can have an arbitrary size
(determined by the number of slots in the object and the actual size
of the prefix) but its elements are either all reference slots or all
data slots.

Special bitfield descriptors have the following format:

---------------------------------------------------------------------------
| special : 1 = 1 | system : 1 = 0 | bf_size : 5 | all_refs : 1 | bf : 24 |
---------------------------------------------------------------------------

Where "system" indicates whether the descriptor corresponds to a
system object.  Currently, only inlists and outlist for persistent
garbage collection have this bit set.

For special objects the remaining fields have different meanings, but
for all non-system bitfields the remaining fields have the following
meaning:

"bf" is a bitmap with one bit set for every reference in the prefix of
the sequence of slots described in 1 and "bf_size" is the size of this
prefix.

"all_refs" is set iff the rest of the sequence of slots of the object
contains only reference slots.


See common/objrefs.h for a generator interface to access bitfields.

*/

#ifndef _OBJ_BITFIELD_H
#define _OBJ_BITFIELD_H

#ifdef __cplusplus
extern "C" {
#endif

#include "utils/bits.h"

typedef Bits32 Obj_bitfield;

/*
  All system objects start with 0xC.
  OBJ_BF_VARBF is a value that indicates that the bitfield
  is stored in the second slot of an object. It is used for 
  classes with varying bitfields.
*/
#define OBJ_BF_VARBF    ((Obj_bitfield) 0xC0000000)


/*
  The inlists and outlists used in persistent garbage collection 
  have the following special values for bitfields.
  */
#define OBJ_BF_INOUTLIST   ((Obj_bitfield) 0xC0000001)
#define OBJ_BF_TRANSLIST   ((Obj_bitfield) 0xC0000002)
#define OBJ_BF_PMAP        ((Obj_bitfield) 0xC0000003)

/*
Bitfield operations.

\begin{verbatim}
    bool OBJ_BF_ISSPECIAL(Obj_bitfield b)	  
    Effects: Returns TRUE iff b is a special bitfield.

    bool OBJ_BF_ISSYS_OBJ(Obj_bitfield b)
    Requires: !OBJ_BF_ISSPECIAL(b)
    Effects: Returns TRUE iff b is a system object bitfield.
\end{verbatim}

*/

#define OBJ_BF_ISSPECIAL(b) ((b) < 0) 
#define OBJ_BF_ISSYS_OBJ(b) (((b) &  0xC0000000) == 0xC0000000)
#define OBJ_BF_ISVARIABLE(b) ((b) == OBJ_BF_VARBF)
#define OBJ_BF_ISCOLLECTABLE(b) ((~(b) & 0xC0000000) || \
				 OBJ_BF_ISVARIABLE(b))


/*
The following operations are only meaningful on non-system objects bitfields.

\begin{verbatim}

    bool OBJ_BF_REFS(Obj_bitfield b)
    Requires: !OBJ_BF_ISSYS_OBJ(b)
    Effects: Return TRUE iff suffix of slot sequence contains only references.

    void OBJ_BF_SET_REFS(Obj_bitfield b, bool val) 
    Requires: !OBJ_BF_ISSYS_OBJ(b)
    Effects: Set "all_refs" bit to val
   
    int OBJ_BF_SIZE(Obj_bitfield b)          
    Requires: !OBJ_BF_ISSYS_OBJ(b)
    Effects: Return the size of the prefix of the slot sequence.

    int OBJ_BF_SIZE_F(Obj_bitfield b)          
    Requires: !OBJ_BF_ISSYS_OBJ(b) && !OBJ_BF_ISSPECIAL(b)
    Effects: Return the size of the prefix of the slot sequence.

    void OBJ_BF_SET_SIZE(Obj_bitfield b, int sz)          
    Requires: !OBJ_BF_ISSYS_OBJ(b) && sz < MAX_PREFIX_SIZE
    Effects: Set the size of the prefix of the slot sequence.

    bool OBJ_BF_ISDATA(Obj_bitfield b, int i)      
    Requires: !OBJ_BF_ISSYS_OBJ(b) && i < OBJ_BF_SIZE(b)
    Effects: Return TRUE iff ith slot is a datum.

    bool OBJ_BF_ISREF(Obj_bitfield b, int i)       
    Requires: !OBJ_BF_ISSYS_OBJ(b) && i < OBJ_BF_SIZE(b)
    Effects: Return TRUE iff ith slot is a reference.

    void OBJ_BF_SETDATA(Obj_bitfield b, int i)     
    Requires: !OBJ_BF_ISSYS_OBJ(b) && i < OBJ_BF_SIZE(b)
    Effects: Mark ith slot as a datum.

    void OBJ_BF_SETREF(Obj_bitfield b, int i)      
    Requires: !OBJ_BF_ISSYS_OBJ(b) && i < OBJ_BF_SIZE(b)
    Effects: Mark ith slot as a reference.

\end{verbatim}
*/

/* Maximum size for slot sequence prefix */
#define MAX_PREFIX_SIZE 24

/* Maximum size for non-special bitfields */
#define MAX_NORMAL_SIZE 31

#define OBJ_BF_REFS(b)      ((b) & 0x01000000)
#define OBJ_BF_SET_REFS(b,val)  ((b) |= ((unsigned)val << MAX_PREFIX_SIZE)) 

#define OBJ_BF_SIZE(b)          ((OBJ_BF_ISSPECIAL(b)) ? MAX_NORMAL_SIZE :\
                                   ((b >> (MAX_PREFIX_SIZE+1)) & 0x1f))

#define OBJ_BF_SIZE_F(b)        ((b >> (MAX_PREFIX_SIZE+1)) & 0x1f)

#define OBJ_BF_SET_SIZE(b,sz)   ((b) = ((b) & 0xC1ffffff) | (sz << (MAX_PREFIX_SIZE+1)))

#define OBJ_BF_ISDATA(b,i)	(!((b) & (1 << (i))))
#define OBJ_BF_ISREF(b,i)	((b) & (1 << (i)))

#define OBJ_BF_SETDATA(b,i)	((b) &= ~(1 << (i)))
#define OBJ_BF_SETREF(b,i)	((b) |=  (1 << (i)))

#define OBJ_BF_ALLDATA (Obj_bitfield) 0x80000000   
#define OBJ_BF_ALLREFS (Obj_bitfield) 0x83000000

#ifdef __cplusplus
}
#endif

#endif /* _OBJ_BITFIELD_H */
