// \section{Format of Objects at the OR}
// 
// This file defines the format of objects as stored at the OR.  The
// FE and the OR exchange objects in this format.

#ifndef _OR_OBJ_H
#define _OR_OBJ_H

#include <stdio.h>
#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/array.h"
#include "slot.h"
#include "shortp.h"
#include "oref.h"
#include "xref.h"
#include "or_num.h"


unsigned int const OR_obj_headers  = 1;
unsigned int const Surrogate_mask = 0x80000000;

// \subsection{Type of an Object Slot}
union OR_slot {
    OR_num      orx;             // OR number (for surrogate)
    char	chars[4];	// Packed characters
    Bits16	value16[2];	// Packed Bits16
    Bits32	value32;	// Bits32
    Ubits32	uvalue32;	// Ubits32

    // Orefs are stored as shortp in the instance variables.
    // and must be accessed as follows.
    Oref get_oref() {return SHORTP_AS_DATA(Oref, oref);}
    void set_oref(Oref _oref) {oref = DATA_TO_SHORTP(_oref);}

    // However, some orefs, such as class orefs and those in or_surrogates
    // are not stored as shortp and must be accessed as plain_oref.
    Oref        plain_oref;      // an oref not stored as a shortp
    Slot        pad;             // For padding purposes
    
private:
    Shortp 	oref;		// an oref stored as a Shortp
};

// \subsection{Object Layout}

// For normal objects:                      For surrogate objects:
//            32 bits 
//	------------------                   ------------------
//	| 0 |Class oref  |                   | 1 | OR_num     |
//	|----------------|                   |----------------| 
//	|  Field 1 ...   |                   |    Oref        |
//	------------------                   ------------------ 

class OR_obj {
  public:
    OR_obj();
    // Default constructor: Creates uninitialized object
    // Not implemented.

    static OR_obj* create(Uint num_slots);
    // Returns a heap allocated OR_obj with "num_slots" for fields
    // (does not include slots for header).

    Oref class_oref() {
	// requires: "this" is not a surrogate object
	// effects: Returns the class oref of this object.
	th_assert(!(header[0].uvalue32 & Surrogate_mask),
		  "Surrogate passed to class_num");
	return header[0].plain_oref;
    }

    void set_class_oref(Oref class_oref) {
	// requires: "this" is not a surrogate object
	// effects: Sets the class number of this object to be class_number
	header[0].plain_oref = (Oref) class_oref & ~Surrogate_mask;
    }

    bool is_surrogate (Xref& xref) {
	// modifies: xref
	// effects: Returns TRUE iff this is a surrogate object
	//          Stores the xref of the destination in xref iff
	//          this is an xref.
	if (header[0].uvalue32 & Surrogate_mask) {
	    xref.orx = header[0].uvalue32  & ~Surrogate_mask;
	    xref.oref = header[1].plain_oref;
	    return TRUE;
	}
	return FALSE;
    }

    bool is_surrogate () { return (header[0].uvalue32 & Surrogate_mask) != 0;}

    OR_slot* fields() { return &header[1]; }
    // effects: Returns a pointer to the fields of the object

    void print(int fslots, FILE *fp = NULL);
    // requires: "this" has fslots field slots
    // effects: Print "this" on fp.

    // Instance Variables
  private:
    friend OR_obj* Surrogate_create(OR_num orx, Oref oref, void* place);
    OR_slot	header[OR_obj_headers];
};

inline unsigned int Surrogate_size() {return 2*sizeof(OR_slot);}
// effects: Returns the space required by a surrogate (in bytes)

inline OR_obj* Surrogate_create(OR_num orx, Oref oref, void* place) {
    // requires: place has sufficient space to store a surrogate
    // modifies: place
    // effects: Creates a surrogate that points to the object <or, oref> and
    //          puts it in place and returns a pointer to it also
    OR_obj* result = (OR_obj*) place;
    result->header[0].orx = orx | Surrogate_mask;
    result->header[1].plain_oref = oref;
    return result;
}

// Array of OR slots
declareArray(OR_slot_array,OR_slot)

#endif /* _OR_OBJ_H */


